/*

 OpenAndStartSessionAction.java

 Copyright 2009 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 */
package net.sqs2.omr.swing.app;

import java.awt.event.ActionEvent;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.logging.Logger;
import java.util.prefs.BackingStoreException;

import javax.swing.SwingUtilities;

import net.sqs2.omr.AppConstants;
import net.sqs2.omr.MarkReaderJarURIContext;
import net.sqs2.omr.app.MarkReaderApp;
import net.sqs2.omr.result.export.model.MarkAreaErrorModel;
import net.sqs2.omr.result.walker.ResultWalker;
import net.sqs2.omr.session.model.PageTaskErrorTableModel;
import net.sqs2.omr.session.model.PageTaskExecutionProgressModel;
import net.sqs2.omr.session.monitor.MarkReaderSessionMonitorAdapter;
import net.sqs2.omr.session.service.MarkReaderSessionService;
import net.sqs2.omr.session.service.MarkReaderSessionServices;
import net.sqs2.omr.sound.SessionFanfare;
import net.sqs2.omr.swing.session.MarkReaderSessionPanel;
import net.sqs2.omr.swing.session.MarkReaderSessionResultWalkerController;

public class OpenAndStartSessionAction{
	
	private final MarkReaderPanelController markReaderPanelController;
	private MarkReaderApp markReaderApp;
	private File sourceDirectoryRootFile;
	private MarkReaderPanelImpl markReaderPanel;
	
	public OpenAndStartSessionAction(MarkReaderPanelController markReaderPanelController, MarkReaderApp markReaderApp, File sourceDirectoryRootFile,
			MarkReaderPanelImpl markReaderPanel){
		this.markReaderPanelController = markReaderPanelController;
		this.markReaderApp = markReaderApp;
		this.sourceDirectoryRootFile = sourceDirectoryRootFile;
		this.markReaderPanel = markReaderPanel;
	}

	private String createTabName(final File sourceDirectoryRootFile, int numSameNameSessions) {
		String name = null;
		if (1 == numSameNameSessions) {
			name = sourceDirectoryRootFile.getName();
		} else if (1 < numSameNameSessions) {
			name = sourceDirectoryRootFile.getName() + "(" + numSameNameSessions + ")";
		}
		return name;
	}

	public synchronized void actionPerformed(ActionEvent ev){
		try {
			MarkReaderSessionService session = MarkReaderSessionServices.get(sourceDirectoryRootFile);
			if (session != null) {
				if (session.isInactive()) {
					session.startSession();
				}
				SwingUtilities.invokeLater(new Runnable() {
					public void run() {
						markReaderPanel.setForgroundSessionPanel(sourceDirectoryRootFile);
					}
				});
				return;
			}
			
			final MarkReaderSessionService newSession = (MarkReaderSessionService) this.markReaderApp.createSession(sourceDirectoryRootFile);
			newSession.setGUIMode(true);
			
			storeSourceDirectoryRootInPrefs(sourceDirectoryRootFile);
			
			SessionFanfare fanfare = new SessionFanfare(new URL(MarkReaderJarURIContext.getSoundBaseURI()
					+ AppConstants.SESSION_START_FANFARE_SOUND_FILENAME));
			fanfare.startFanfare();

			final PageTaskExecutionProgressModel markReaderSessionProgressModel = newSession.getPageTaskExecutionProgressModel();
			final MarkReaderSessionPanel markReaderSessionPanel = markReaderPanelController.createSessionPanel(sourceDirectoryRootFile,
					markReaderSessionProgressModel);

			SwingUtilities.invokeLater(new Runnable() {
				public void run() {
					int numSameNameSessions = MarkReaderSessionServices.countSessionsBySourceDirectory(sourceDirectoryRootFile);
					String name = createTabName(sourceDirectoryRootFile, numSameNameSessions);
					markReaderSessionPanel.setPlayStateGUI();
					markReaderPanel.setRunningTabIcon(sourceDirectoryRootFile);
					markReaderPanel.addMarkReaderSessionPanel(markReaderSessionPanel, name);
				}
			});
			
			final MarkReaderSessionPanelController markReaderSessionPanelController = createMarkReaderSessionPanelController(markReaderApp, newSession,
					markReaderPanel, markReaderPanelController, markReaderSessionPanel);

			newSession.addSessionMonitor(new MarkReaderSessionMonitorAdapter(){
				@Override
				public void notifyExportResultDirectoryStarted(File sourceDirectoryRootFile) {
					ResultWalker sessionResultWalker = newSession.getSessionResultExporter();
					MarkAreaErrorModel markAreaErrorModel = new MarkAreaErrorModel();
					sessionResultWalker.addEventConsumer(markAreaErrorModel);
					sessionResultWalker.addEventConsumer(new MarkReaderSessionResultWalkerController(markReaderSessionPanel.getSessionResultWalkerPanel(), markAreaErrorModel));
					markReaderSessionPanelController.setMarkAreaErrorModel(markAreaErrorModel);
					newSession.setMarkAreaErrorModel(markAreaErrorModel);
				}
			});

			newSession.startSession();

		} catch (IOException ex) {
			ex.printStackTrace();
		}
	}
	
	private MarkReaderSessionPanelController createMarkReaderSessionPanelController(
			MarkReaderApp markReaderApp,
			MarkReaderSessionService session,
			MarkReaderPanel markReaderPanel,
			MarkReaderPanelController markReaderPanelController,
			MarkReaderSessionPanel markReaderSessionPanel
			){
		return new MarkReaderSessionPanelController(markReaderApp, session, markReaderPanel,
				markReaderPanelController, markReaderSessionPanel,
				new PageTaskErrorTableModel());
	}

	
	private void storeSourceDirectoryRootInPrefs(final File sourceDirectoryRoot) {
		MarkReaderApp.getPreferences().put(AppConstants.SOURCE_DIRECTORY_ROOT_KEY_IN_PREFERENCES,
				sourceDirectoryRoot.getAbsolutePath());
		try {
			MarkReaderApp.getPreferences().flush();
		} catch (IllegalArgumentException ignore) {
			Logger.getLogger(OpenAndStartSessionAction.class.getName()).info(ignore.getMessage());
		} catch (BackingStoreException ignore) {
			Logger.getLogger(OpenAndStartSessionAction.class.getName()).severe(ignore.getMessage());
		}
	}

}