package net.sqs2.omr.swing.app;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.io.File;
import java.io.IOException;
import java.util.logging.Logger;

import javax.swing.AbstractAction;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.InputMap;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSlider;
import javax.swing.JSpinner;
import javax.swing.KeyStroke;
import javax.swing.SpinnerNumberModel;
import javax.swing.SwingConstants;
import javax.swing.WindowConstants;
import javax.swing.border.EmptyBorder;
import javax.swing.border.TitledBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.plaf.metal.MetalLookAndFeel;

import net.sqs2.omr.AppConstants;
import net.sqs2.omr.master.FormMaster;
import net.sqs2.omr.session.config.ConfigUtil;
import net.sqs2.omr.session.config.ConfigXMLFileFilter;
import net.sqs2.omr.session.config.MarkRecognitionConfig;
import net.sqs2.omr.session.config.SourceConfig;
import net.sqs2.omr.session.config.SourceDirectoryConfiguration;
import net.sqs2.omr.session.config.SourceDirectoryConfigurations;
import net.sqs2.omr.swing.Images;
import net.sqs2.omr.swing.Messages;

class MarkRecognitionConfigurationDialog extends JDialog {

	private static final int RECOGNITION_DENSITY = 239;
	private static final int DOUBLE_MARK_IGNORANCE = 230;

	private static final long serialVersionUID = 0L;
	ConfigXMLFileFilter configFileFilter;
	FormMaster formMaster;
	File configFile;
	SourceDirectoryConfiguration sourceDirectoryConfiguration;

	MarkRecognitionConfigurationDialog(Frame owner, SourceDirectoryConfiguration sourceDirectoryConfiguration) {
		this(owner, AppConstants.USER_CUSTOMIZE_DEFAULT_CONFIG_FILE, sourceDirectoryConfiguration,
				Messages.SESSION_PROCESS_CONFIG_DEFAULT_LABEL);
	}

	MarkRecognitionConfigurationDialog(Frame owner, File configFile,
			SourceDirectoryConfiguration sourceDirectoryConfiguration) {
		this(owner, configFile, sourceDirectoryConfiguration, Messages.SESSION_PROCESS_CONFIG_FOLDER_LABEL
				+ ": " + configFile.getAbsolutePath());
	}

	MarkRecognitionConfigurationDialog(Frame owner, File configFile,
			SourceDirectoryConfiguration sourceDirectoryConfiguration, String label) {
		super(owner, label, true);
		this.sourceDirectoryConfiguration = sourceDirectoryConfiguration;
		this.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
		setLayout(new BorderLayout());
		setAlwaysOnTop(true);

		this.configFile = configFile;

		if (!this.configFile.getParentFile().isDirectory()) {
			this.configFile.getParentFile().mkdirs();
		}

		this.configFileFilter = new ConfigXMLFileFilter(this.configFile);
		int densityThresholdValue = getDensityThresholdValue();
		int doubleMarkIgnoranceThresholdValue = getDoubleMarkIgnoranceThresholdValue();

		String label1 = Messages.SESSION_PROCESS_RECOGNITION_DENSITY_LABEL;
		String label2 = Messages.SESSION_PROCESS_DOUBLE_MARK_IGNORANCE_LABEL;

		final MarkRecognitionConfigurationInnerPanel markRecognitionConfigurationPanel = new MarkRecognitionConfigurationInnerPanel(
				label1, densityThresholdValue, label2, doubleMarkIgnoranceThresholdValue);
		add(markRecognitionConfigurationPanel, BorderLayout.CENTER);
		pack();

		AbstractAction cancelAction = new AbstractAction("Cancel") {
			private static final long serialVersionUID = 0L;

			public void actionPerformed(ActionEvent evt) {
				dispose();
			}
		};

		InputMap imap = getRootPane().getInputMap();
		imap.put(KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0), "close-it");
		getRootPane().getActionMap().put("close-it", cancelAction);

		getRootPane().setDefaultButton(markRecognitionConfigurationPanel.getCancelButton());
		markRecognitionConfigurationPanel.getCancelButton().addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent event) {
				dispose();
			}
		});

		markRecognitionConfigurationPanel.getOKButton().addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent event) {
				try {
					int densityThreshold255 = markRecognitionConfigurationPanel.getDensityThreshold255();
					int recognitionMargin255 = markRecognitionConfigurationPanel.getRecognitionMargin255();
					storeConfiguration(densityThreshold255, recognitionMargin255);
					dispose();
				} catch (IOException ex) {
					ex.printStackTrace();
				}
			}
		});
	}

	private int getDensityThresholdValue() {
		int defaultValue = RECOGNITION_DENSITY;
		try {
			if (this.configFile.exists()) {
				defaultValue = (int) (Float.parseFloat(this.configFileFilter.getDensityThreshold()) * 255);
			}
		} catch (Exception ignore) {
			Logger.getAnonymousLogger()
					.severe(
							"file=" + this.configFile
									+ ", density=null, set value from RangeConfigurationPanel.java");
		}
		return defaultValue;
	}

	private int getDoubleMarkIgnoranceThresholdValue() {
		int defaultValue = DOUBLE_MARK_IGNORANCE;
		try {
			if (this.configFile.exists()) {
				defaultValue = (int) (Float.parseFloat(this.configFileFilter
						.getDoubleMarkIgnoranceThreshold()) * 255);
			}
		} catch (Exception ignore) {
			Logger.getAnonymousLogger().severe(
					"file=" + this.configFile
							+ ", recognitionMargin=null, set value from RangeConfigurationPanel.java");
		}
		return defaultValue;
	}

	void storeConfiguration(int densityValue255, int doubleMarkIgnoranceValue255) throws IOException {
		SourceDirectoryConfigurations.setUserConfigurationEnabled(true);
		float density = (float) (densityValue255 / 255.0);
		float doubleMarkIgnoranceThreshold = (float) (doubleMarkIgnoranceValue255 / 255.0);

		if (!this.configFile.exists()) {
			ConfigUtil.createConfigFile(this.configFile);
		}

		if(this.sourceDirectoryConfiguration != null){			
			MarkRecognitionConfig markRecognitionConfig = ((SourceConfig)this.sourceDirectoryConfiguration.getConfig().getSourceConfig()).getMarkRecognitionConfig();
			markRecognitionConfig.setDensity(density);
			markRecognitionConfig.setDoubleMarkIgnoranceThreshold(doubleMarkIgnoranceThreshold);
			ConfigXMLFileFilter self = new ConfigXMLFileFilter(this.configFile);
			self.overwriteValues(density, doubleMarkIgnoranceThreshold);
		}else{
			ConfigXMLFileFilter self = new ConfigXMLFileFilter(this.configFile);
			self.overwriteValues(density, doubleMarkIgnoranceThreshold);
		}		
	}

	class MarkRecognitionConfigurationInnerPanel extends JPanel {
		private static final long serialVersionUID = 0L;

		RangeConfigurationPanel densityConfigurator;
		RangeConfigurationPanel recognitionMarginConfigurator;

		OKCancelButtonPanel okCancelButtonPanel;

		MarkRecognitionConfigurationInnerPanel(String title1, int densityThresholdDefaultValue,
				String label2, int recognitionMarginDefaultValue) {
			this.okCancelButtonPanel = new OKCancelButtonPanel();
			setBorder(new EmptyBorder(5, 5, 5, 5));
			setLayout(new BorderLayout());
			this.densityConfigurator = new RangeConfigurationPanel(title1,
					Messages.CONFIG_RECOGNITION_DENSITY_TOOLTIP, 0, 255, 10, 50, densityThresholdDefaultValue);
			this.recognitionMarginConfigurator = new RangeConfigurationPanel(label2, Messages.CONFIG_RECOGNITION_DENSITY_TOOLTIP,
					0, 255, 10, 50, recognitionMarginDefaultValue);

			JPanel centerPanel = new JPanel();
			centerPanel.setLayout(new BoxLayout(centerPanel, BoxLayout.Y_AXIS));
			centerPanel.add(this.densityConfigurator);
			centerPanel.add(this.recognitionMarginConfigurator);
			centerPanel.setBorder(new TitledBorder(Messages.CONFIG_RECOGNITION_TITLE));

			add(centerPanel, BorderLayout.CENTER);
			add(this.okCancelButtonPanel, BorderLayout.SOUTH);
		}

		public JButton getCancelButton() {
			return this.okCancelButtonPanel.getCancelButton();
		}

		public JButton getOKButton() {
			return this.okCancelButtonPanel.getOKButton();
		}

		public int getDensityThreshold255() {
			return this.densityConfigurator.getValue();
		}

		public int getRecognitionMargin255() {
			return this.recognitionMarginConfigurator.getValue();
		}
	}

	class OKCancelButtonPanel extends JPanel {
		private static final long serialVersionUID = 0L;
		JButton okButton;
		JButton cancelButton;

		OKCancelButtonPanel() {
			this.okButton = new JButton("OK", Images.ACCEPT_ICON);
			this.okButton.setPreferredSize(new Dimension(130, 32));

			this.cancelButton = new JButton("Cancel", Images.CANCEL_ICON);
			this.cancelButton.setPreferredSize(new Dimension(120, 32));
			this.cancelButton.setSelected(true);

			setBorder(new EmptyBorder(5, 5, 5, 5));
			setLayout(new BoxLayout(this, BoxLayout.X_AXIS));
			add(Box.createHorizontalGlue());
			add(this.okButton);
			add(Box.createHorizontalStrut(16));
			add(this.cancelButton);
		}

		public JButton getOKButton() {
			return this.okButton;
		}

		public JButton getCancelButton() {
			return this.cancelButton;
		}
	}

	class RangeConfigurationPanel extends JPanel {

		private static final long serialVersionUID = 0L;

		JLabel label;
		JSlider slider;
		JSpinner spinner;

		RangeConfigurationPanel(String title, String tooltipText, int min, int max, int minorTickSpacing,
				int majorTickSpacing, int defaultValue) {
			setLayout(new BorderLayout());
			setBorder(new EmptyBorder(10, 10, 10, 10));

			this.label = createLabel(title);
			this.label.setToolTipText(tooltipText);
			this.label.setPreferredSize(new Dimension(250, 80));
			this.slider = createSlider(min, max, minorTickSpacing, majorTickSpacing, defaultValue);

			this.spinner = createSpinner(min, max, 1, defaultValue);

			this.slider.addChangeListener(new ChangeListener() {
				public void stateChanged(ChangeEvent e) {
					RangeConfigurationPanel.this.spinner.setValue(RangeConfigurationPanel.this.slider
							.getValue());
				}
			});

			this.spinner.addChangeListener(new ChangeListener() {
				public void stateChanged(ChangeEvent e) {
					RangeConfigurationPanel.this.slider
							.setValue((Integer) RangeConfigurationPanel.this.spinner.getValue());
				}
			});

			// this.slider.setBorder(new EmptyBorder(0,15,0,15));

			add(this.label, BorderLayout.WEST);
			add(this.slider, BorderLayout.CENTER);
			add(this.spinner, BorderLayout.EAST);
		}

		public int getValue() {
			return this.slider.getValue();
		}

		private JSlider createSlider(int min, int max, int minorTickSpacing, int majorTickSpacing, int defaultValue) {
			JSlider slider = new JSlider();
			slider.setMinimum(min);
			slider.setMaximum(max);
			slider.setValue(defaultValue);
			slider.setPaintTicks(true);
			slider.setMinorTickSpacing(minorTickSpacing);
			slider.setMajorTickSpacing(majorTickSpacing);
			slider.setPreferredSize(new Dimension(300, 50));
			slider.setPaintLabels(true);
			slider.setUI(new GrayscaleSliderUI());
			slider.setLabelTable(slider.createStandardLabels(50));
			return slider;
		}

		private JLabel createLabel(String title) {
			JLabel label = new JLabel(title);
			// label.setHorizontalAlignment(SwingConstants.RIGHT);
			label.setBorder(new EmptyBorder(0, 0, 0, 10));
			return label;
		}

		private JSpinner createSpinner(int min, int max, int stepSize, int defaultValue) {
			SpinnerNumberModel spinnerNumberModel = new SpinnerNumberModel(defaultValue, min, max, stepSize);
			JSpinner spinner = new JSpinner(spinnerNumberModel);
			return spinner;
		}

		class GrayscaleSliderUI extends javax.swing.plaf.metal.MetalSliderUI {

			@Override
			public void paintThumb(Graphics g) {
				Graphics2D g2 = (Graphics2D) g;
				g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
				int gray = 255 * this.slider.getValue() / this.slider.getMaximum();
				g2.setColor(new Color(gray, gray, gray));
				g2.fillOval(this.thumbRect.x, this.thumbRect.y, this.thumbRect.width - 1,
						this.thumbRect.height - 1);
				g2.setColor(Color.gray);
				g2.drawOval(this.thumbRect.x, this.thumbRect.y, this.thumbRect.width - 1,
						this.thumbRect.height - 1);
			}

			/*
			 * 
			 * public void paintTrack(Graphics g) { super.paintTrack(g); int cx,
			 * cy, cw, ch; int pad;
			 * 
			 * super.paintTrack(g);
			 * 
			 * Rectangle trackBounds = this.contentRect;
			 * if(slider.getOrientation() == JSlider.HORIZONTAL ) {
			 * 
			 * Graphics2D g2 = (Graphics2D)g; int roundSize = 12; int w = 12;
			 * int h = 12; g2.setColor(Color.white);
			 * g2.fillRoundRect((int)trackBounds.getWidth() - w - 1, 0, w, h,
			 * roundSize, roundSize);
			 * 
			 * g2.setColor(Color.black); g2.fillRoundRect(0, 0, w, h, roundSize,
			 * roundSize);
			 * 
			 * g2.setColor(Color.gray);
			 * g2.drawRoundRect((int)trackBounds.getWidth() - w - 1, 0, w, h,
			 * roundSize, roundSize); g2.drawRoundRect(0, 0, w, h, roundSize,
			 * roundSize);
			 * 
			 * //g2.drawRoundRect(0, 0, (int)trackBounds.getWidth() - 1,
			 * (int)trackBounds.getHeight() - 1, 3, 3);
			 * 
			 * 
			 * int y1 = 10; g2.fillRect(0, y1, (int)(trackBounds.getWidth()
			 * slider.getValue() / (slider.getMaximum() - slider.getMinimum()))
			 * , (int)(trackBounds.getHeight() - 1 - y1));
			 * g2.setColor(Color.white);
			 * g2.fillRect((int)((trackBounds.getWidth() - 1) slider.getValue()
			 * / (slider.getMaximum() - slider.getMinimum() + 1)), y1,
			 * (int)(trackBounds.getWidth() (slider.getMaximum() -
			 * slider.getValue()) / slider.getMaximum()),
			 * (int)(trackBounds.getHeight() - 1 - y1));
			 * 
			 * } }
			 */

			@Override
			public void paintTrack(Graphics g) {
				/*
				Color trackColor = !this.slider.isEnabled() ? MetalLookAndFeel.getControlShadow()
						: this.slider.getForeground();
				 */
				boolean leftToRight = true;

				g.translate(this.trackRect.x, this.trackRect.y);

				int trackLeft = 0;
				int trackTop = 0;
				int trackRight = 0;
				int trackBottom = 0;

				// Draw the track
				if (this.slider.getOrientation() == SwingConstants.HORIZONTAL) {
					trackBottom = (this.trackRect.height - 1) - getThumbOverhang();
					trackTop = trackBottom - (getTrackWidth() - 1);
					trackRight = this.trackRect.width - 1;
				} else {
					if (leftToRight) {
						trackLeft = (this.trackRect.width - getThumbOverhang()) - getTrackWidth();
						trackRight = (this.trackRect.width - getThumbOverhang()) - 1;
					} else {
						trackLeft = getThumbOverhang();
						trackRight = getThumbOverhang() + getTrackWidth() - 1;
					}
					trackBottom = this.trackRect.height - 1;
				}

				if (this.slider.isEnabled()) {
					g.setColor(MetalLookAndFeel.getControlDarkShadow());
					g.drawRect(trackLeft, trackTop, (trackRight - trackLeft) - 1,
							(trackBottom - trackTop) - 1);

					g.setColor(MetalLookAndFeel.getControlHighlight());
					g.drawLine(trackLeft + 1, trackBottom, trackRight, trackBottom);
					g.drawLine(trackRight, trackTop + 1, trackRight, trackBottom);

					g.setColor(MetalLookAndFeel.getControlShadow());
					g.drawLine(trackLeft + 1, trackTop + 1, trackRight - 2, trackTop + 1);
					g.drawLine(trackLeft + 1, trackTop + 1, trackLeft + 1, trackBottom - 2);

				} else {
					g.setColor(MetalLookAndFeel.getControlShadow());
					g.drawRect(trackLeft, trackTop, (trackRight - trackLeft) - 1,
							(trackBottom - trackTop) - 1);

				}

				int _middleOfThumb = this.thumbRect.x + (this.thumbRect.width / 2);
				_middleOfThumb -= this.trackRect.x; // To compensate for the
				g.setColor(Color.BLACK);
				g.fillRect(trackLeft + 1, trackTop + 1, _middleOfThumb - 2, (trackBottom - trackTop) - 2);
				g.setColor(Color.WHITE);
				g.fillRect(_middleOfThumb, trackTop + 1, (trackRight - trackLeft) - _middleOfThumb - 2,
						(trackBottom - trackTop) - 2);

				// Draw the fill
				if (this.filledSlider) {
					int middleOfThumb = 0;
					int fillTop = 0;
					int fillLeft = 0;
					int fillBottom = 0;
					int fillRight = 0;

					if (this.slider.getOrientation() == SwingConstants.HORIZONTAL) {
						middleOfThumb = this.thumbRect.x + (this.thumbRect.width / 2);
						middleOfThumb -= this.trackRect.x; // To compensate for
						// the
						// g.translate()
						fillTop = !this.slider.isEnabled() ? trackTop : trackTop + 1;
						fillBottom = !this.slider.isEnabled() ? trackBottom - 1 : trackBottom - 2;

						if (!drawInverted()) {
							fillLeft = !this.slider.isEnabled() ? trackLeft : trackLeft + 1;
							fillRight = middleOfThumb;
						} else {
							fillLeft = middleOfThumb;
							fillRight = !this.slider.isEnabled() ? trackRight - 1 : trackRight - 2;
						}

					} else {
						middleOfThumb = this.thumbRect.y + (this.thumbRect.height / 2);
						middleOfThumb -= this.trackRect.y; // To compensate for
						// the
						// g.translate()
						fillLeft = !this.slider.isEnabled() ? trackLeft : trackLeft + 1;
						fillRight = !this.slider.isEnabled() ? trackRight - 1 : trackRight - 2;

						if (!drawInverted()) {
							fillTop = middleOfThumb;
							fillBottom = !this.slider.isEnabled() ? trackBottom - 1 : trackBottom - 2;
						} else {
							fillTop = !this.slider.isEnabled() ? trackTop : trackTop + 1;
							fillBottom = middleOfThumb;
						}
					}

					if (this.slider.isEnabled()) {
						g.setColor(this.slider.getBackground());
						g.drawLine(fillLeft, fillTop, fillRight, fillTop);
						g.drawLine(fillLeft, fillTop, fillLeft, fillBottom);

						// g.setColor( MetalLookAndFeel.getControlShadow() );
						g.setColor(Color.WHITE);
						g.fillRect(middleOfThumb, fillTop + 1, fillRight - fillLeft - middleOfThumb,
								fillBottom - fillTop);
					} else {
						g.setColor(Color.WHITE);
						// g.setColor( MetalLookAndFeel.getControlShadow() );
						g.fillRect(fillLeft, fillTop, fillRight - fillLeft, trackBottom - trackTop);
					}
				}

				g.translate(-this.trackRect.x, -this.trackRect.y);
			}

		}

	}

}
