/**
 *  SessionSourceInitializer.java

 Copyright 2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Created on 2007/03/10
 Author hiroya
 */
package net.sqs2.omr.session.source;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Callable;

import net.sqs2.image.ImageUtil;
import net.sqs2.omr.AppConstants;
import net.sqs2.omr.cache.CacheConstants;
import net.sqs2.omr.master.FormMasterFactory;
import net.sqs2.omr.master.NoPageMasterException;
import net.sqs2.omr.master.PageMaster;
import net.sqs2.omr.master.PageMasterAccessor;
import net.sqs2.omr.master.PageMasterException;
import net.sqs2.omr.model.SourceDirectory;
import net.sqs2.omr.session.config.SourceDirectoryConfiguration;
import net.sqs2.omr.session.monitor.SourceInitializationMonitor;
import net.sqs2.omr.util.JarExtender;
import net.sqs2.omr.util.NameBasedComparableFile;
import net.sqs2.util.PathUtil;

public class SessionSourceInitializer implements Callable<SourceDirectory>{
	private SessionSource sessionSource;
	private FormMasterFactory formMasterFactory = null;
	private boolean isSearchPageMasterFromAncestorDirectory = false;
	private SourceInitializationMonitor monitor;

	public SessionSourceInitializer(FormMasterFactory formMasterFactory,
			SessionSource sessionSource, SourceInitializationMonitor monitor,
			boolean isSearchPageMasterFromAncestorDirectory) throws IOException {
		this.formMasterFactory = formMasterFactory;
		this.sessionSource = sessionSource;
		this.monitor = monitor;
		this.isSearchPageMasterFromAncestorDirectory = isSearchPageMasterFromAncestorDirectory;
	}

	public SourceDirectory call() throws IOException, SourceInitializationStopException, PageMasterException, SessionSourceException{

		File rootDirectory = this.sessionSource.getRootDirectory();
		sessionSource.initialize();
		SourceDirectory sourceDirectoryRoot = new SourceDirectory(rootDirectory);
		sessionSource.getSessionSourceContentAccessor().putSourceDirectory("", sourceDirectoryRoot);

		String configPath = AppConstants.getResultDirectoryName() + File.separatorChar + AppConstants.SOURCE_CONFIG_FILENAME;
		createConfigurationTemplateFile(rootDirectory, configPath);

		SourceDirectory sourceDirectory = scanDescendant(sessionSource, sourceDirectoryRoot, null, configPath);

		if (sourceDirectory == null) {
			throw new RuntimeException("SourceDirectory is empty!");
		}
		sessionSource.getSessionSourceContentIndexer().createSourceDirectoryDepthOrderedListMap();

		this.monitor.notifySourceInitializeDone(rootDirectory);
		return sourceDirectoryRoot;
	}

	private void createConfigurationTemplateFile(File sourceDirectoryFile, String configPath) {
		File resultDir = new File(sourceDirectoryFile, 
				AppConstants.getResultDirectoryName());
		File configFile = new File(resultDir, configPath);
		if (! configFile.exists()) {
			new JarExtender().extend(new String[] { AppConstants.SOURCE_CONFIG_FILENAME }, resultDir);
		}
	}

	private SourceDirectory scanDescendant(SessionSource sessionSource,
			SourceDirectory sourceDirectory, PageMaster pageMaster, 
			String configPath) throws IOException, SessionSourceException, PageMasterException, SourceInitializationStopException {
		List<File> subDirectoryList = new ArrayList<File>();

		int numAddedImageFiles = 0;
		int numAddedFolders = 0;

		String base = (0 == sourceDirectory.getPath().length()) ? "" : sourceDirectory.getPath()
				+ File.separatorChar;

		
		File[] items = sourceDirectory.getDirectory().listFiles();
		if (items == null || 0 == items.length) {
			return null;
		}
		File[] files = NameBasedComparableFile.createSortedFileArray(items);

		for (File file : files) {
			String filename = file.getName();
			if (ImageUtil.isSupported(filename)) {
				numAddedImageFiles++;
			} else if (file.isDirectory() && !isIgnorableDirectory(file)) {
				subDirectoryList.add(file);
			} else if (filename.endsWith(".pdf") || filename.endsWith(".sqm")) {
				PageMaster newPageMaster = createMaster(sessionSource, file);
				if (newPageMaster != null) {
					pageMaster = newPageMaster;
				}
			}
			if (sessionSource.hasStopped()) {
				throw new SourceInitializationStopException();
			}
		}

		for (File file : subDirectoryList) {
			String filename = file.getName();
			String path = base + filename;
			SourceDirectory subSourceDirectory = new SourceDirectory(sourceDirectory.getRoot(), path);
			SourceDirectory aChild = scanDescendant(sessionSource, subSourceDirectory, pageMaster,
					configPath);
			if (aChild != null) {
				numAddedFolders++;
				sourceDirectory.addChildSourceDirectoryList(aChild);
				sessionSource.getSessionSourceContentAccessor().putSourceDirectory(path, aChild);
			}
		}

		if (0 == numAddedImageFiles && 0 == numAddedFolders) {
			return null;
		}
		sourceDirectory.setID(currentSourceDirectorySerialID++);
		return setupSourceDirectory(sessionSource, sourceDirectory, pageMaster, configPath,
				numAddedImageFiles);
	}

	private static int currentSourceDirectorySerialID = 0;


	private SourceDirectory setupSourceDirectory(SessionSource sessionSource, SourceDirectory sourceDirectory, PageMaster pageMaster, String configPath, int numAddedImageFiles) throws IOException, SessionSourceException, PageMasterException {

		this.monitor.notifyScanDirectory(sourceDirectory.getDirectory());
		
		if (pageMaster == null) {
			if (sourceDirectory != null && sourceDirectory.getChildSourceDirectoryList() != null) {
				for (SourceDirectory aChild : sourceDirectory.getChildSourceDirectoryList()) {
					pageMaster = aChild.getPageMaster();
					if (pageMaster != null) {
						break;
					}
				}
			}
			if (pageMaster == null && this.isSearchPageMasterFromAncestorDirectory) {
				try {
					pageMaster = getPageMasterFromAncestorDirectory(sessionSource, sourceDirectory
							.getDirectory().getParentFile());
				} catch (IOException ignore) {
				}
			}
		}

		if (pageMaster == null) {
			throw new NoPageMasterException(sourceDirectory.getDirectory());
		}
		
		sourceDirectory.setPageMaster(pageMaster);

		File resultDir = new File(sourceDirectory.getDirectory(), 
				AppConstants.getResultDirectoryName());
		File configFile = new File(resultDir, configPath);
		if (! configFile.exists()) {
			new JarExtender().extend(new String[] { AppConstants.SOURCE_CONFIG_FILENAME }, resultDir);
		}

		SourceDirectoryConfiguration configuration = sessionSource.getConfiguration(configPath, configFile.lastModified());
		sourceDirectory.setSourceDirectoryConfiguration(configuration);

		if (numAddedImageFiles % pageMaster.getNumPages() != 0) {
			this.monitor.notifyErrorNumOfImages(sourceDirectory, numAddedImageFiles);
			throw new InvalidNumImagesException(sourceDirectory, numAddedImageFiles);
		}
		SessionSourceContentIndexer indexer = sessionSource.getSessionSourceContentIndexer();
		indexer.setConfiguration(configuration.getFileResourceID(), configuration);
		indexer.putPageMaster(pageMaster.getFileResourceID(), pageMaster);


		indexer.addSourceDirectory(pageMaster, sourceDirectory);
		/*
		if(sourceDirectory.isRoot()){
		}
		*/
		this.monitor.notifyFoundImages(numAddedImageFiles, sourceDirectory.getDirectory());
		return sourceDirectory;
	}

	private PageMaster getPageMasterFromAncestorDirectory(SessionSource sessionSource, File directory) throws IOException {
		if (directory == null) {
			throw new IOException("No master file.");
		}
		for (File file : directory.listFiles()) {
			String filename = file.getName().toLowerCase();
			if (filename.endsWith(".pdf") || filename.endsWith(".sqm")) {
				PageMaster master = createMaster(sessionSource, file);
				if (master != null) {
					return master;
				}
			}
		}
		return getPageMasterFromAncestorDirectory(sessionSource, directory.getParentFile());
	}

	private PageMaster createMaster(SessionSource sessionSource, File file) {

		PageMaster master = null;
		try {
			String path = PathUtil.getRelativePath(file, sessionSource.getRootDirectory());
			PageMasterAccessor pageMasterAccessor = sessionSource.getSessionSourceContentAccessor()
					.getPageMasterAccessor();
			PageMaster cachedMaster = null;

			try {
				cachedMaster = pageMasterAccessor.get(path);
			} catch (Error ignore) {
				pageMasterAccessor.close();

			}
			if (cachedMaster != null && cachedMaster.getLastModified() == file.lastModified()) {
				return cachedMaster;
			}

			master = this.formMasterFactory.create(sessionSource.getRootDirectory(), path);
			if (master == null) {
				throw new PageMasterException(file);
			}
			pageMasterAccessor.put(master);
			pageMasterAccessor.flush();

			this.monitor.notifyFoundMaster(master);
			return master;
		} catch (PageMasterException ex) {
			this.monitor.notifyErrorOnPageMaster(ex);
		} catch (IOException ex) {
			ex.printStackTrace();
		}
		return null;
	}

	private boolean isIgnorableDirectory(File file) {
		return file.getName().endsWith(AppConstants.getResultDirectoryName())
				|| new File(file, CacheConstants.CACHE_ROOT_DIRNAME).exists()
				|| (new File(file, "TEXTAREA").exists() && new File(file, "CHART").exists());

	}

}
