/**
 *  SessionThreadManager.java

 Copyright 2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Created on 2007/01/31
 Author hiroya
 */

package net.sqs2.omr.session.service;

import java.io.IOException;
import java.util.concurrent.Callable;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import net.sqs2.lang.GroupThreadFactory;
import net.sqs2.net.MulticastNetworkConnection;
import net.sqs2.net.RMIRegistryMulticastAdvertisingService;
import net.sqs2.omr.MarkReaderConstants;
import net.sqs2.omr.app.config.MarkReaderAppConfiguration;
import net.sqs2.omr.model.PageTaskHolder;
import net.sqs2.omr.session.broker.impl.PageTaskExecutorSessionServerDispatcher;
import net.sqs2.omr.session.broker.impl.RemotePageTaskExecutorPeer;
import net.sqs2.omr.session.source.SessionSource;
import net.sqs2.omr.session.source.impl.SessionSourceServerImpl;


public class SessionServiceDaemons {

	public static final int TASK_PRODUCER_EXEC_INTERVAL_IN_MILLIS = 10000;// msec
	public static final int TASK_CONSUMER_EXEC_INTERVAL_IN_MILLIS = 5;// msec
	public static final int RECYCLE_INTERVAL_IN_SEC = 11; // sec

	private GroupThreadFactory GROUP_THREAD_FACTORY_NORMAL_PRIORITY = 
		new GroupThreadFactory("SessionServiceDaemons",
			Thread.NORM_PRIORITY - 1, true);

	private GroupThreadFactory GROUP_THREAD_FACTORY_HIGH_PRIORITY = 
		new GroupThreadFactory(
			"SessionServiceDaemons", Thread.NORM_PRIORITY + 1, true);
	
	private GroupThreadFactory GROUP_THREAD_FACTORY_LOW_PRIORITY = 
		new GroupThreadFactory("SessionServiceDaemons",
			Thread.MIN_PRIORITY, true);

	private ScheduledExecutorService remoteTaskRecycleService;
	private ScheduledExecutorService localTaskRecycleService;

	private ScheduledExecutorService sessionSourceFactoryService;
	private ScheduledExecutorService pageTaskProducerService;
	private ScheduledExecutorService pageTaskConsumerService;
	private ScheduledExecutorService resultExportWorkerService;

	private Future<?> localTaskRecycleFuture;
	private Future<?> remoteTaskRecycleFuture;
	private Future<SessionSource> sessionSourceFactoryFuture;
	private Future<?> pageTaskProducerFuture;
	private Future<?> pageTaskConsumerFuture;
	private Future<?> resultExportWorkerFuture;

	private RMIRegistryMulticastAdvertisingService advertisingService = null;
	private PageTaskHolder taskHolder;
	private String sessionServerName;
	
	public SessionServiceDaemons(long sessionID, RemotePageTaskExecutorPeer peer, PageTaskExecutorSessionServerDispatcher localTaskExecutorEnv, PageTaskHolder taskHolder) {
		this.advertisingService = createAdvertisingService(sessionID, peer, localTaskExecutorEnv.getKey());
		this.taskHolder = taskHolder;		
		this.remoteTaskRecycleService = Executors.newScheduledThreadPool(1, GROUP_THREAD_FACTORY_LOW_PRIORITY);
		this.localTaskRecycleService = Executors.newScheduledThreadPool(1, GROUP_THREAD_FACTORY_LOW_PRIORITY);
		this.sessionServerName = peer.getRMIServerName();
		startTaskRecycleThread();
	}
	
	private void startTaskRecycleThread() {
		this.remoteTaskRecycleFuture = this.remoteTaskRecycleService.scheduleWithFixedDelay(new RemotePageTaskRecycleService(this.taskHolder),
				RECYCLE_INTERVAL_IN_SEC, RECYCLE_INTERVAL_IN_SEC, TimeUnit.SECONDS);

		this.localTaskRecycleFuture = this.localTaskRecycleService.scheduleWithFixedDelay(new LocalPageTaskRecycleService(this.taskHolder),
				RECYCLE_INTERVAL_IN_SEC + 1, RECYCLE_INTERVAL_IN_SEC, TimeUnit.SECONDS);
	}
	
	void start(){
		this.sessionSourceFactoryService = Executors.newScheduledThreadPool(1, GROUP_THREAD_FACTORY_NORMAL_PRIORITY);
		this.pageTaskProducerService = Executors.newScheduledThreadPool(1, GROUP_THREAD_FACTORY_NORMAL_PRIORITY);
		this.pageTaskConsumerService = Executors.newScheduledThreadPool(1, GROUP_THREAD_FACTORY_HIGH_PRIORITY);
		this.resultExportWorkerService = Executors.newScheduledThreadPool(1, GROUP_THREAD_FACTORY_NORMAL_PRIORITY);
		startAdvertisement();
	}

	Future<SessionSource> startAndWaitSessionSourceFactory(Callable<SessionSource> task) {
		this.sessionSourceFactoryFuture = this.sessionSourceFactoryService.submit(task);
		return this.sessionSourceFactoryFuture;
	}

	Future<?> startPageTaskProducer(Runnable task) {
		this.pageTaskProducerFuture = this.pageTaskProducerService.submit(task);
		return this.pageTaskProducerFuture;
	}

	Future<?> startAndWaitPageTaskConsumer(Callable<Void> task) {
		this.pageTaskConsumerFuture = this.pageTaskConsumerService.submit(task);
		return this.pageTaskConsumerFuture;
	}

	Future<?> startAndWaitSessionResultWalker(Callable<Void> task) {
		this.resultExportWorkerFuture = this.resultExportWorkerService.submit(task);
		return this.resultExportWorkerFuture;
	}

	void stop() {
		stopAdvertisement();
		
		stopFuture(this.sessionSourceFactoryFuture);
		this.sessionSourceFactoryFuture = null;
		stopFuture(this.pageTaskProducerFuture);
		this.pageTaskProducerFuture = null;
		stopFuture(this.pageTaskConsumerFuture);
		this.pageTaskConsumerFuture = null;
		stopFuture(this.resultExportWorkerFuture);
		this.resultExportWorkerFuture = null;
		
	}

	private void stopFuture(Future<?> future) {
		if (future != null) {
			future.cancel(true);
		}
	}

	void close() {
		
		stop();
		
		stopFuture(this.localTaskRecycleFuture);
		this.localTaskRecycleFuture = null;
		stopFuture(this.remoteTaskRecycleFuture);
		this.remoteTaskRecycleFuture = null;

		this.localTaskRecycleService.shutdown();
		this.remoteTaskRecycleService.shutdown();
		
		this.sessionSourceFactoryService.shutdown();
		this.pageTaskProducerService.shutdown();
		this.pageTaskConsumerService.shutdown();
		this.resultExportWorkerService.shutdown();
		finishAdvertisement();
	}
	
	private RMIRegistryMulticastAdvertisingService createAdvertisingService(long sessionID, RemotePageTaskExecutorPeer peer, long key) {
		MulticastNetworkConnection multicastNetworkConnection = peer.getMulticastNetworkConnection();
		int rmiPort = peer.getRMIPort();
		if (multicastNetworkConnection != null) {
			try {
				RMIRegistryMulticastAdvertisingService ret = new RMIRegistryMulticastAdvertisingService(
						multicastNetworkConnection, key, sessionID,
						MarkReaderConstants.ADVERTISE_SERVICE_THREAD_PRIORITY, rmiPort,
						MarkReaderConstants.SESSION_SOURCE_ADVERTISE_DELAY_IN_SEC){
					/*
					public void logSendMessage(){
						//TODO
						//Toolkit.getDefaultToolkit().beep();
					}*/
				};
				return ret;
			} catch (IOException ex) {
				ex.printStackTrace();
			}
		}
		return null;
	}
	
	public String getRMIServiceName() {
		return sessionServerName;
	}

	private void startAdvertisement() {
		try {
			if (this.advertisingService != null && MarkReaderAppConfiguration.isClusterEnabled()) {
				String serviceName = getRMIServiceName();
				this.advertisingService.startAdvertising(SessionSourceServerImpl.getInstance(), serviceName);
				//Logger.getAnonymousLogger().info("Export: " + RMIRegistryUtil.createURL(this.rmiPort, serviceName));
			}
		} catch (Exception ex) {
			ex.printStackTrace();
		}
	}

	private void stopAdvertisement() {
		if (this.advertisingService != null) {
			this.advertisingService.stopAdvertising(sessionServerName);
			this.advertisingService = null;
		}
	}
	
	private void finishAdvertisement() {
		stopAdvertisement();
	}
}
