/*

 MarkReaderSession.java

 Copyright 2009 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 */
package net.sqs2.omr.session.service;

import java.awt.Desktop;
import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.concurrent.Callable;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.logging.Logger;

import net.sqs2.lang.GroupThreadFactory;
import net.sqs2.omr.app.config.MarkReaderAppConfiguration;
import net.sqs2.omr.httpd.SQSHttpdManager;
import net.sqs2.omr.master.FormMasterFactory;
import net.sqs2.omr.master.PageMasterException;
import net.sqs2.omr.model.PageTask;
import net.sqs2.omr.model.PageTaskException;
import net.sqs2.omr.model.PageTaskHolder;
import net.sqs2.omr.result.export.model.MarkAreaErrorModel;
import net.sqs2.omr.result.walker.ResultWalker;
import net.sqs2.omr.session.broker.impl.PageTaskExecutorSessionServerDispatcher;
import net.sqs2.omr.session.broker.impl.RemotePageTaskExecutorPeer;
import net.sqs2.omr.session.consumer.PageTaskConsumerRowGenerator;
import net.sqs2.omr.session.model.PageTaskExecutionProgressModel;
import net.sqs2.omr.session.monitor.MarkReaderSessionMonitor;
import net.sqs2.omr.session.monitor.MarkReaderSessionMonitorImpl;
import net.sqs2.omr.session.monitor.SourceInitializationMonitor;
import net.sqs2.omr.session.producer.SessionSourceScannerTaskProducer;
import net.sqs2.omr.session.source.SessionSource;
import net.sqs2.omr.session.source.SessionSourceException;
import net.sqs2.omr.session.source.SessionSources;
import net.sqs2.omr.session.source.SessionStopException;

class MarkReaderSessionServiceImpl implements Callable<Void>, MarkReaderSessionService{

	private static ExecutorService sessionExecutorService = 
		Executors.newSingleThreadExecutor(new GroupThreadFactory("MarkReaderSession", Thread.NORM_PRIORITY, true));
	private SessionServiceDaemons sessionThreadManager;
	private Future<?> sessionFuture;
	private SessionSource sessionSource = null;
	private FormMasterFactory formMasterFactory;
	private PageTaskExecutorSessionServerDispatcher localTaskExecutorDispatcher;

	private RemotePageTaskExecutorPeer peer;
	private boolean isGUIMode = false;
	private MarkAreaErrorModel markAreaErrorModel;

	private long sessionID = -1;
	private File sourceDirectoryRootFile;

	public enum STATE {
		NOT_INITIALIZED, INITIALIZED, STARTED, RUNNING, STOPPED, FINISHED
	}

	private STATE state = STATE.NOT_INITIALIZED;

	private PageTaskExecutionProgressModel model;
	private PageTaskHolder taskHolder;
	private MarkReaderSessionMonitorImpl monitors;
	private ResultWalker sessionResultExporter;
	
	public MarkReaderSessionServiceImpl(File sourceDirectoryRoot, 
			RemotePageTaskExecutorPeer peer,
			FormMasterFactory formMasterFactory,
			PageTaskExecutorSessionServerDispatcher localTaskExecutorEnv)
			throws IOException {
		this.sourceDirectoryRootFile = sourceDirectoryRoot;
		this.peer = peer;
		this.formMasterFactory = formMasterFactory;
		this.localTaskExecutorDispatcher = localTaskExecutorEnv;
			
		this.monitors = new MarkReaderSessionMonitorImpl();
		this.sessionID = System.currentTimeMillis();
		this.taskHolder = new PageTaskHolder();
		this.model = new PageTaskExecutionProgressModel(this.sourceDirectoryRootFile, this.taskHolder);
		this.sessionThreadManager = new SessionServiceDaemons(this.sessionID, this.peer, this.localTaskExecutorDispatcher, this.taskHolder);
	}
	
	
	public Void call() {
		try {
			try{
			SessionSourceFactoryService sessionSourceFactoryService = new SessionSourceFactoryService(this.sessionID, this.sourceDirectoryRootFile, 
					this.formMasterFactory,
					(SourceInitializationMonitor)this.monitors);
			Future<SessionSource> sessionSourceFactoryFuture = this.sessionThreadManager.startAndWaitSessionSourceFactory(sessionSourceFactoryService);
			this.sessionSource = sessionSourceFactoryFuture.get();
			
			this.state = STATE.RUNNING;
			
			SessionSourceScannerTaskProducer taskProducer = new SessionSourceScannerTaskProducer(
					this.sessionSource, (MarkReaderSessionMonitorImpl)this.monitors,
					this.taskHolder);
			
			Future<?> taskProducerFuture = this.sessionThreadManager.startPageTaskProducer(taskProducer);
			taskProducerFuture.get();
			
			this.localTaskExecutorDispatcher.setInitialized();
			this.model.setTimePageTaskProduced(System.currentTimeMillis());
			
			PageTaskConsumerRowGenerator taskConsumer = new PageTaskConsumerRowGenerator(this);
			taskConsumer.setup(this.sourceDirectoryRootFile);
			Future<?> taskConsumerFuture = this.sessionThreadManager.startAndWaitPageTaskConsumer(taskConsumer); 
			taskConsumerFuture.get();
			
			this.model.setTimeOfPageTaskHasFinished(System.currentTimeMillis());
			
			this.sessionResultExporter = new ResultWalkerFactoryService().createResultWalker(this.isGUIMode, this.sessionSource);
			this.monitors.notifyExportResultDirectoryStarted(this.sourceDirectoryRootFile);
			
			SessionResultExportService sessionResultExportTask = new SessionResultExportService(this.sessionResultExporter);
			Future<?> sessionResultWalkerFuture = this.sessionThreadManager.startAndWaitSessionResultWalker(sessionResultExportTask);
			sessionResultWalkerFuture.get();
			this.monitors.notifyExportResultDirectoryFinished(this.sourceDirectoryRootFile);
			this.sessionSource.setFinished();
			this.sessionSource.getSessionSourceContentAccessor().flush();
			finishSession();
			
			if (! isGUIMode && MarkReaderAppConfiguration.isOpenResultBrowserEnabled()) {
				openResultBrowser(this.sessionSource.getSessionID());
			}
			
			return null;
			} catch (ExecutionException ex) {
				throw ex.getCause();
			}

		} catch (SessionSourceException ignore) {
			ignore.printStackTrace();
		} catch (SessionStopException ignore) {
			ignore.printStackTrace();
		} catch (CancellationException ignore) {
		} catch (IOException ex) {
			ex.printStackTrace();
		} catch (InterruptedException ignore) {
			ignore.printStackTrace();
		} catch (PageTaskException ignore) {
			ignore.printStackTrace();
		} catch (PageMasterException ex) {
			this.monitors.notifyErrorOnPageMaster(ex);
		} catch (Throwable ex) {
			ex.printStackTrace();
		}
		stopSession();
		return null;
	}

	private void openResultBrowser(long sid) {
		try {
			Desktop.getDesktop().browse(new URI(SQSHttpdManager.getBaseURI()+"/e?sid="+sid));
		} catch (IOException e) {
			e.printStackTrace();
		} catch (URISyntaxException e) {
			e.printStackTrace();
		}
	}
	
	public void setGUIMode(boolean isGUIMode){
		this.isGUIMode = isGUIMode;
	}
	
	public synchronized void startSession() throws IOException {
		this.monitors.addSessionMonitor(this.model);
		
		new SourceDirectoryInitializeService(this.sourceDirectoryRootFile).call();
		this.state = STATE.INITIALIZED;

		this.sessionThreadManager.start();

		this.sessionFuture = sessionExecutorService.submit(this);// async execution call 
		
		this.state = STATE.STARTED;
		this.monitors.notifySessionStarted(this.sourceDirectoryRootFile);
	}
	
	public synchronized void stopSession() {
		this.state = STATE.STOPPED;
		this.taskHolder.stop();
		if(this.sessionThreadManager != null){
			this.sessionThreadManager.stop();
			this.sessionThreadManager.close();
			this.sessionThreadManager = null;
		}
		if (this.sessionFuture != null) {
			this.sessionFuture.cancel(true);
		}
		this.monitors.notifySessionStopped(this.sourceDirectoryRootFile);
	}
	
	public void finishSession() {

		long elapsedTimePageTaskExecution = getPageTaskExecutionProgressModel().getTimePageTaskFinished() - getPageTaskExecutionProgressModel().getTimePageTaskProduced();

		int totalPages = getPageTaskExecutionProgressModel().getNumTargetPages();
		int reusedPages = getPageTaskExecutionProgressModel().getNumReusedPages();
		int targetPages = totalPages - reusedPages;
		float pageParSec = ( targetPages * 1000.0f / elapsedTimePageTaskExecution);
		
		if(0 < totalPages){
			Logger.getAnonymousLogger().info("**** Number of Processed PageTasks: "+targetPages+"="+totalPages+"-"+reusedPages);
		}
		Logger.getAnonymousLogger().info("**** PageTask Execution Finished in: "+( elapsedTimePageTaskExecution / 1000.0)+" sec");
		Logger.getAnonymousLogger().info("**** Process Rate: "+pageParSec+" imgs/sec");
		Logger.getAnonymousLogger().info("**** TimeElapsed: "+getPageTaskExecutionProgressModel().getTimeElapsedString());
		
		if(this.sessionThreadManager != null){
			this.sessionThreadManager.stop();
			this.sessionThreadManager.close();
		}
		this.monitors.notifySessionFinished(this.sourceDirectoryRootFile);
	}
	
	public void closeSession() {
		try{
			SessionSource sessionSource = getSessionSource();
			if(sessionSource != null){
				getSessionSource().close();
				MarkReaderSessionServices.remove(sourceDirectoryRootFile);
			}
			SessionSources.remove(this.sessionID);
		}catch(IOException ex){
			ex.printStackTrace();
		}
	}
	
	public ResultWalker getSessionResultExporter() {
		if (this.sessionResultExporter == null) {
			throw new RuntimeException("session result exporter is null");
		}
		return this.sessionResultExporter;
	}


	public synchronized void addSessionMonitor(MarkReaderSessionMonitor monitor) {
		this.monitors.addSessionMonitor(monitor);
	}

	public synchronized void removeSessionMonitor(MarkReaderSessionMonitor monitor) {
		this.monitors.removeSessionMonitor(monitor);
	}

	public long getSessionID() {
		return this.sessionID;
	}
	
	public SessionSource getSessionSource() {
		return this.sessionSource;
	}

	public boolean isRunning() {
		return this.state == STATE.RUNNING;
	}

	public boolean hasStarted() {
		return this.state == STATE.STARTED;
	}
	
	public boolean hasStopped(){
		return this.state == STATE.STOPPED;
	}
	
	public boolean isInactive() {
		//SessionSource sessionSource = getSessionSource();
		// || (sessionSource != null && (sessionSource.hasStopped() || sessionSource.hasFinished())
		return this.state == STATE.STOPPED || this.state == STATE.FINISHED;
	}

	public PageTaskExecutionProgressModel getPageTaskExecutionProgressModel() {
		return this.model;
	}

	public long getKey() {
		return this.localTaskExecutorDispatcher.getKey();
	}
	
	public PageTaskHolder getTaskHolder() {
		return this.taskHolder;
	}

	public synchronized void notifyStoreTask(PageTask pageTask) {
		this.monitors.notifyStoreTask(pageTask);
	}

	public File getSourceDirectoryRootFile() {
		return this.sourceDirectoryRootFile;
	}
	
	public void setMarkAreaErrorModel(MarkAreaErrorModel markAreaErrorModel){
		this.markAreaErrorModel = markAreaErrorModel;
	}
	
	public MarkAreaErrorModel getMarkAreaErrorModel(){
		return markAreaErrorModel;
	}
}