/**
 *  SessionSourceScannerTaskGenerator.java

 Copyright 2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Created on 2007/01/31
 Author hiroya
 */
package net.sqs2.omr.session.producer;

import java.io.IOException;
import java.util.Calendar;
import java.util.logging.Logger;

import net.sqs2.omr.MarkReaderConstants;
import net.sqs2.omr.model.AbstractPageTask;
import net.sqs2.omr.model.AbstractTask;
import net.sqs2.omr.model.PageID;
import net.sqs2.omr.model.PageTask;
import net.sqs2.omr.model.PageTaskAccessor;
import net.sqs2.omr.model.PageTaskFactory;
import net.sqs2.omr.model.PageTaskHolder;
import net.sqs2.omr.model.RowID;
import net.sqs2.omr.model.SourceDirectory;
import net.sqs2.omr.model.SpreadSheet;
import net.sqs2.omr.session.monitor.MarkReaderSessionMonitorImpl;
import net.sqs2.omr.session.source.SessionSource;
import net.sqs2.omr.session.source.SessionSourceContentIndexer;
import net.sqs2.omr.session.source.SessionStopException;

public class SessionSourceScannerTaskProducer extends SessionSourceScanner implements Runnable {

	private MarkReaderSessionMonitorImpl monitor;
	private PageTaskHolder taskHolder;

	public SessionSourceScannerTaskProducer(SessionSource sessionSource, MarkReaderSessionMonitorImpl monitor,
			PageTaskHolder taskHolder) throws IOException {
		super(sessionSource);
		this.monitor = monitor;
		this.taskHolder = taskHolder;
	}

	@Override
	public AbstractSessionSourceScannerWorker createWorker() throws IOException {
		return new SessionSourceScannerWorker(this.sessionSource, this.taskHolder, this.monitor,
				MarkReaderConstants.SESSION_SOURCE_NEWFILE_IGNORE_SEC_THRESHOLD_IN_SEC);
	}

	@Override
	public void run() {

		super.run();

	}

	static protected class SessionSourceScannerWorker extends AbstractSessionSourceScannerWorker {

		private static final boolean INFO = false;

		int numAdded = 0;
		int numReused = 0;
		int numRetry = 0;

		private SessionSource sessionSource;
		private PageTaskHolder taskHolder;
		private MarkReaderSessionMonitorImpl monitor;
		private long newFileIgnoreSecThreshold;

		private PageTaskAccessor taskAccessor;

		private SpreadSheet currentSpreadSheet = null;
		private long now;

		SessionSourceScannerWorker(SessionSource sessionSource, PageTaskHolder taskHolder,
				MarkReaderSessionMonitorImpl monitor, long newFileIgnoreSecThreshold) throws IOException {
			this.now = Calendar.getInstance().getTimeInMillis();
			this.sessionSource = sessionSource;
			this.taskHolder = taskHolder;
			this.monitor = monitor;
			this.newFileIgnoreSecThreshold = newFileIgnoreSecThreshold;
			this.taskAccessor = sessionSource.getSessionSourceContentAccessor().getPageTaskAccessor();
		}

		@Override
		void work(SourceDirectory sourceDirectory, int pageNumber, PageID pageID, int rowIndex) throws SessionStopException {
			PageTask task = PageTaskFactory.createPageTask(sourceDirectory, pageNumber, pageID, this.sessionSource.getSessionID());
			if(isIgnorableTask(task)){
				return;
			}

			PageTask preparedTask = prepareTask(task);
			if (preparedTask != null) {
				this.taskHolder.incrementNumTargetTasks(1);
				this.taskAccessor.put(preparedTask);
				this.taskHolder.addPreparedTask(preparedTask);
				this.monitor.notifyPageTaskProduced(preparedTask);
			}
			SessionSourceContentIndexer sessionSourceIndexer = this.sessionSource.getSessionSourceContentIndexer();
			RowID rowID = new RowID(this.currentSpreadSheet, rowIndex);
			sessionSourceIndexer.putRowID(pageID, rowID);
		}

		@Override
		void startScanningSourceDirectory(SourceDirectory sourceDirectory) {
		}

		private PageTask prepareTask(PageTask task) throws SessionStopException {
			PageTask storedTask = null;
			try {
				storedTask = this.taskAccessor.get(task.toString());
			} catch (Exception ignore) {
			}

			if (isOnceExecutedTask(storedTask)) {
				if (hasError(storedTask)) {

					this.monitor.notifyErrorPageTaskReproduced(storedTask);

					if (!isExecutionRequiredTaskWithError(task, storedTask)) {
						return null;
					}

				} else if (storedTask.getTaskResult() != null) {
					if (!isExecutionRequiredTask(task, storedTask)) {
						return null;
					}
				} else {
					this.numAdded++;
					Logger.getLogger("session").info("==========ADD\t" + task);
				}
			} else {
				this.numAdded++;
				Logger.getLogger("session").info("==========ADD\t" + task);
			}
			return task;
		}

		private boolean isOnceExecutedTask(AbstractTask storedTask) {
			return storedTask != null;
		}

		private boolean isIgnorableTask(PageTask task) {
			if (isConcurrentFileModificationSuspected(task)) {
				if (INFO) {
					Logger.getLogger("source").info("IGNORE\t" + task);
				}
				return true;
			}
			/*
			 * if(isPrepareTaskd(task)){ if(INFO){
			 * Logger.getLogger("source").info("PREPARED\t"+task); } return
			 * true; } if(isLeasedTask(task)){ if(INFO){
			 * Logger.getLogger("source").info("LEASED\t"+task); } return true;
			 * }
			 */
			return false;
		}

		private boolean isExecutionRequiredTaskWithError(AbstractTask task, AbstractTask storedTask) {
			if (this.sessionSource.getSessionID() == storedTask.getSessionID()) {
				if (INFO) {
					Logger.getLogger("source").info("IGNORE ERROR\t" + task);
				}
				return false;
			} else {
				this.numRetry++;
				if (INFO) {
					Logger.getLogger("source").info("==========RETRY ERROR\t" + task);
				}
				return true;
			}
		}

		private boolean isExecutionRequiredTask(AbstractTask task, AbstractTask storedTask) {
			if (this.sessionSource.getSessionID() == storedTask.getSessionID()) {
				if (INFO) {
					// Logger.getLogger("source").info("IGNORE\t" + task);
				}
				return false;
			} else {
				if (INFO) {
					// Logger.getLogger("source").info("REUSE\t" + task);
				}
				this.numReused++;
				this.taskHolder.setNumReusedTasks(this.numReused);
				return false;
			}
		}

		/*

		private boolean isPreparedTask(AbstractTask task) {
			return this.taskHolder.isPreparedTask(task);
		}

		private boolean isLeasedTask(AbstractTask task) {
			return this.taskHolder.isLeasedTask(task);
		}

		 */

		private boolean hasError(AbstractPageTask task) {
			return task.getTaskError() != null;
		}

		private boolean isConcurrentFileModificationSuspected(PageTask task) {
			return this.newFileIgnoreSecThreshold != -1 && (this.now - task.getPageID().getFileResourceID().getLastModified()
					<= this.newFileIgnoreSecThreshold);
		}

		@Override
		void finishScan() {
			StringBuilder sb = new StringBuilder(64);
			sb.append("\nnumReused = " + this.numReused);
			sb.append("\nnumAdded = " + this.numAdded);
			sb.append("\nnumRetry = " + this.numRetry);
			Logger.getLogger("session").info("TaskProducer\n\t" + sb.toString());
		}
	}
}
