/**
 * 
 */
package net.sqs2.omr.session.broker.impl;

import java.awt.Point;
import java.awt.Rectangle;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.rmi.RemoteException;
import java.util.List;
import java.util.logging.Logger;

import net.sqs2.image.ImageFactory;
import net.sqs2.image.ImageUtil;
import net.sqs2.omr.AppConstants;
import net.sqs2.omr.master.FormArea;
import net.sqs2.omr.master.FormMaster;
import net.sqs2.omr.master.PageMaster;
import net.sqs2.omr.model.FormAreaCommand;
import net.sqs2.omr.model.PageID;
import net.sqs2.omr.model.PageTask;
import net.sqs2.omr.model.PageTaskError;
import net.sqs2.omr.model.PageTaskException;
import net.sqs2.omr.model.PageTaskResult;
import net.sqs2.omr.session.broker.AbstractExecutable;
import net.sqs2.omr.session.config.ConfigImpl;
import net.sqs2.omr.session.config.MarkRecognitionConfig;
import net.sqs2.omr.session.config.SourceConfig;
import net.sqs2.omr.session.config.SourceDirectoryConfiguration;
import net.sqs2.omr.session.config.ValidationConfig;
import net.sqs2.omr.session.logic.MarkAreaPreviewImageConstants;
import net.sqs2.omr.session.logic.PageFrameException;
import net.sqs2.omr.session.logic.PageFrameHandler;
import net.sqs2.omr.session.logic.PageImageSource;
import net.sqs2.omr.session.logic.PageImageSourceException;
import net.sqs2.omr.session.logic.TextAreaPreviewImageConstants;
import net.sqs2.sound.SoundManager;

public class PageTaskExecutable extends AbstractExecutable<PageTask,PageTaskExecutorSessionServerDispatcher>{

	public PageTaskExecutable(PageTask task, PageTaskExecutorSessionServerDispatcher sessionServiceDispatcher) {
		super(task, sessionServiceDispatcher);
	}

	public SourceDirectoryConfiguration getConfiguration() throws RemoteException {
		return this.getSessionServiceDispatcher().getConfiguration(this.getTask().getSessionID(), 
				this.getTask().getConfigHandlerFileResourceID());
	}
	
	public void call() throws RemoteException {

		long base = System.currentTimeMillis();
		PageTask pageTask = getTask();

		try {
			FormMaster formMaster = (FormMaster) getSessionServiceDispatcher().getPageMaster(
					pageTask.getSessionID(), pageTask);

			BufferedImage pageImage = retrievePageBufferedImage();
			SourceDirectoryConfiguration configuration = getConfiguration();
			if (configuration == null) {
				throw new RuntimeException("use default page task");
			}

			ConfigImpl config = (ConfigImpl) configuration.getConfig();
			SourceConfig sourceConfig = config.getSourceConfig(pageTask.getPageID().getFileResourceID().getRelativePath(), pageTask.getPageNumber());
			String formAreaImageFormat = AppConstants.FORMAREA_IMAGE_FORMAT;

			PageFrameHandler pageFrameHandler = new PageFrameHandler(sourceConfig, pageImage, 
					getTask().getPageID().toString(), formMaster);
			Point[] pageCorners = pageFrameHandler.scanPageFrameCorners();
			PageImageSource pageSource = createPageSource(formMaster, pageImage, pageCorners);

			ValidationConfig validationConfig = sourceConfig.getFrameConfig().getValidationConfig();
			if (validationConfig != null && validationConfig.isCheckUpsideDown()) {
				pageFrameHandler.checkUpsideDown(formMaster, pageSource);
			}

			PageTaskResult result = new PageTaskResult(pageCorners);

			if (validationConfig != null && validationConfig.isCheckEvenOdd()) {
				pageFrameHandler.checkPageOrder(result, formMaster, pageTask.getPageNumber(), pageSource);
			}

			setupFormAreaCommandList(result, formMaster, pageTask.getPageNumber(), pageSource,
					formAreaImageFormat, sourceConfig.getMarkRecognitionConfig());
			pageTask.setTaskResult(result);
			long submitLap = System.currentTimeMillis() - base;

			Logger.getLogger("executor").info("[[Process OK in " + submitLap + " msec]]\t" + pageTask);
			SoundManager.getInstance().play(AppConstants.TASK_EXECUTION_SOUND_FILENAME);

		} catch (PageFrameException ex) {
			setPageTaskError(ex);
		} catch (PageImageSourceException ex) {
			setPageTaskError(ex);
		} catch (RemoteException ex) {
			ex.printStackTrace();
			throw ex;
		} catch (IOException ex) {
			ex.printStackTrace();
			pageTask.setTaskError(new PageTaskError(pageTask.getPageID(), ex.getMessage()));
		}
	}
	
	public void submit() throws RemoteException{
		getSessionServiceDispatcher().getSessionService().submitPageTask(
			getSessionServiceDispatcher().getKey(), this.getTask().getSessionID(), this.getTask());
	}

	private void setPageTaskError(PageTaskException pageTaskException) {
		PageID pageID = this.getTask().getPageID();
		Logger.getLogger("executor").warning(
				"[[Process NG]]\t" + pageID.getFileResourceID().getRelativePath() + " "
						+ pageTaskException.getExceptionModel().toString());
		this.getTask().setTaskError(new PageTaskError(this.getTask().getPageID(), pageTaskException.getExceptionModel()));
	}

	private BufferedImage retrievePageBufferedImage() throws RemoteException, IOException {
		PageID pageID = this.getTask().getPageID();
		long sessionID = this.getTask().getSessionID();
		byte[] imageByteArray = this.getSessionServiceDispatcher().getImageByteArray(sessionID,
				pageID.getFileResourceID());
		String type = ImageUtil.getType(pageID.getExtension());
		if (imageByteArray == null) {
			throw new RuntimeException(pageID.toString());
		}

		BufferedImage pageImage = ImageFactory.createImage(imageByteArray, pageID.getIndexInFile(), type);
		return pageImage;
	}

	private static PageImageSource createPageSource(PageMaster pageMaster, BufferedImage pageImage, Point[] pageCorners) {
		return new PageImageSource(pageImage, pageMaster.getCorners(), pageCorners);
	}

	private static void setupFormAreaCommandList(PageTaskResult result, FormMaster pageMaster, int pageNumber, PageImageSource pageSource, String formAreaImageFormat, MarkRecognitionConfig config) throws IOException, PageImageSourceException {
		List<FormArea> formAreaList = pageMaster.getFormAreaListByPageIndex(pageNumber - 1);
		int hmargin = config.getHorizontalMargin();
		int vmargin = config.getVerticalMargin();
		for (FormArea formArea : formAreaList) {
			if (formArea.isMarkArea()) {
				BufferedImage image = pageSource.cropImage(getPreviewMarginRect(formArea.getRect()));
				int value = -1;
				if (MarkRecognitionConfig.AVERAGE_DENSITY.equals(config.getAlgorithm())) {
					value = pageSource.getGrayscaleDensityWithVerticalFilter(getMarginRect(
							formArea.getRect(), hmargin, vmargin), config.getResolutionScale());
				} else if (MarkRecognitionConfig.CONVOLUTION3X3_AVERAGE_DENSITY.equals(config.getAlgorithm())) {
					value = pageSource.getGrayscaleDensityWithFilter(getMarginRect(formArea.getRect(),
							hmargin, vmargin), config.getNoMarkThreshold(), config.getResolutionScale());
				} else {

				}
				result.addFormAreaCommand(createFormAreaCommand(formArea.getID(), formAreaImageFormat,
						ImageFactory.writeImage(image, formAreaImageFormat), value / 255.0f));
			} else if (formArea.isTextArea()) {
				BufferedImage image = pageSource.cropImage(getMarginRect(formArea.getRect(),
						TextAreaPreviewImageConstants.HORIZONTAL_MARGIN,
						TextAreaPreviewImageConstants.HORIZONTAL_MARGIN));
				result.addFormAreaCommand(createFormAreaCommand(formArea.getID(), formAreaImageFormat,
						ImageFactory.writeImage(image, formAreaImageFormat), 0));
			} else {
				throw new RuntimeException("invalid formArea:" + formArea);
			}
		}
	}

	private static FormAreaCommand createFormAreaCommand(String id, String imageType, byte[] imageByteArray, float density) {
		return new FormAreaCommand(id, imageType, imageByteArray, density);
	}

	private static Rectangle getMarginRect(Rectangle rect, int hmargin, int vmargin) {
		return new Rectangle(rect.x - hmargin, rect.y - vmargin, rect.width + hmargin * 2, rect.height
				+ vmargin * 2);
	}

	private static Rectangle getPreviewMarginRect(Rectangle rect) {
		return getMarginRect(rect, MarkAreaPreviewImageConstants.HORIZONTAL_MARGIN,
				MarkAreaPreviewImageConstants.VERTICAL_MARGIN);
	}

}
