/*

 RemoteExecutorManager.java

 Copyright 2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Created on 2007/01/11

 */
package net.sqs2.omr.session.broker;

import java.net.InetAddress;
import java.net.MalformedURLException;
import java.net.SocketException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.UnknownHostException;
import java.rmi.ConnectException;
import java.rmi.Naming;
import java.rmi.NotBoundException;
import java.rmi.RemoteException;
import java.util.Map;
import java.util.logging.Logger;

import net.sqs2.net.NetworkUtil;
import net.sqs2.omr.model.AbstractTask;
import net.sqs2.omr.session.source.RemoteSessionSourceServer;

import org.apache.commons.collections15.map.LRUMap;

public abstract class AbstractRemoteExecutorManager<T extends AbstractTask, D extends SessionServerDispatcher, E extends AbstractExecutable<T,D>> {

	private static boolean DEBUG = true;
	private Map<String, AbstractExecutorPeerBroker<T,D,E>> remoteExecutorBrokerMap;
	private ExecutorPeer<T,D> executorPeer;
	private int maximumNumberOfConnections;
	
	public AbstractRemoteExecutorManager(ExecutorPeer<T,D> executorPeer, int maximumNumberOfConnections) {
		this.executorPeer = executorPeer;
		this.maximumNumberOfConnections = maximumNumberOfConnections;
		this.remoteExecutorBrokerMap = new LRUMap<String, AbstractExecutorPeerBroker<T,D,E>>(this.maximumNumberOfConnections) {
			private static final long serialVersionUID = 0L;

			@Override
			protected boolean removeLRU(LinkEntry<String, AbstractExecutorPeerBroker<T,D,E>> entry) {
				// release resources held by entry
				(entry.getValue()).stop();
				return true; // actually delete entry
			}
		};

		new RemoteExecutorPeerBrokerConnectionManager<T,D,E>(this.remoteExecutorBrokerMap);
	}

	public synchronized void connect(String uri, long key, long sessionID) {

		try {
			if (NetworkUtil.isMyAddress(InetAddress.getAllByName(new URI(uri).getHost()))) {
				//Logger.getLogger("executor").info("received message is mine");
				if(! AbstractExecutorPeerBroker.DEBUG_CLUSTER_MODE){ 
					return;
				}
			}

			if (this.maximumNumberOfConnections <= this.remoteExecutorBrokerMap.size()) {
				Logger.getLogger("executor").info(
						"num remoteTaskBroders exceeded:" + this.remoteExecutorBrokerMap.size());
				return;
			}

			AbstractExecutorPeerBroker<T,D,E> remoteTaskBroker = this.remoteExecutorBrokerMap.get(uri);

			if (remoteTaskBroker != null
					&& (remoteTaskBroker.isConnected() == false || remoteTaskBroker.getKey() != key)) {
				this.remoteExecutorBrokerMap.remove(uri);
				Logger.getLogger("executor").info("Remove old taskBroker=" + uri);
				remoteTaskBroker = null;
			}

			if (remoteTaskBroker == null) {
				Logger.getLogger("executor").info("Create new taskBroker=" + uri);
				remoteTaskBroker = createRemoteExecutorPeerBroker(uri, key, sessionID);
			} else {
				//Logger.getLogger("executor").info("Reuse connected taskBroker=" + uri);
			}
		} catch (SocketException ex) {
			Logger.getLogger("executor").severe("SocketException:" + uri);
		} catch (UnknownHostException ex) {
			Logger.getLogger("executor").severe("URISyntaxException:" + uri);
		} catch (URISyntaxException ex) {
			Logger.getLogger("executor").severe("UnknownHostException:" + uri);
		}
	}

	private AbstractExecutorPeerBroker<T,D,E> createRemoteExecutorPeerBroker(String uriString, long remoteKey, long sessionID) {
		try {
			URI uri = new URI(uriString);
			RemoteSessionSourceServer remoteSessionService = (RemoteSessionSourceServer) Naming.lookup(uri.toString()); // connect
																										// SessionService
			long result = remoteSessionService.ping(remoteKey); // authentication
			if (DEBUG) {
				Logger.getLogger("executor").info("RemoteSessionService.URI=" + uri);
				Logger.getLogger("executor").info("Hello=" + result);
			}
			AbstractExecutorPeerBroker<T,D,E> remoteExecutorPeerBroker =
				createExecutorPeerBroker(this.executorPeer, remoteSessionService, remoteKey);
			
			this.remoteExecutorBrokerMap.put(uriString, remoteExecutorPeerBroker);
			return remoteExecutorPeerBroker;
		} catch (ConnectException ex) {
			Logger.getLogger("executor").severe("ConnectException:" + uriString);
		} catch (RemoteException ex) {
			Logger.getLogger("executor").severe("RemoteException:" + uriString);
		} catch (MalformedURLException ex) {
			Logger.getLogger("executor").severe("MalformedURLException:" + uriString);
		} catch (NotBoundException ex) {
			Logger.getLogger("executor").severe("NotBoundException:" + uriString);
		} catch (URISyntaxException ex) {
			Logger.getLogger("executor").severe("URISyntaxException:" + ex.getMessage());
		}
		return null;
	}
	
	public void shutdown() {
		for (AbstractExecutorPeerBroker<T,D,E> sessionExecutor : this.remoteExecutorBrokerMap.values()) {
			if (sessionExecutor != null) {
				sessionExecutor.shutdown();
			}
		}
	}
	
	public abstract AbstractExecutorPeerBroker<T,D,E> createExecutorPeerBroker(ExecutorPeer<T,D> executorPeer, 
			RemoteSessionSourceServer remoteSessionService, long remoteKey);
	

}
