/*

 MarkReaderApp.java

 Copyright 2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Created on 2007/01/11

 */
package net.sqs2.omr.app;

import java.io.File;
import java.io.IOException;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.util.logging.Logger;
import java.util.prefs.Preferences;

import net.sqs2.net.NetworkUtil;
import net.sqs2.net.RMIRegistryUtil;
import net.sqs2.omr.AppConstants;
import net.sqs2.omr.MarkReaderConstants;
import net.sqs2.omr.app.service.ClearSessionService;
import net.sqs2.omr.app.service.CloseSessionService;
import net.sqs2.omr.app.service.CreateSessionService;
import net.sqs2.omr.app.service.ShutdownAppService;
import net.sqs2.omr.app.service.StopSessionService;
import net.sqs2.omr.master.FormMasterFactory;
import net.sqs2.omr.session.broker.impl.PageTaskExecutorPeerBroker;
import net.sqs2.omr.session.broker.impl.PageTaskExecutorSessionServerDispatcher;
import net.sqs2.omr.session.broker.impl.RemotePageTaskExecutorPeer;
import net.sqs2.omr.session.service.MarkReaderSessionService;
import net.sqs2.omr.session.source.impl.SessionSourceServerImpl;

public class MarkReaderApp{
	
	transient Preferences prefs = null;
	private FormMasterFactory formMasterFactory;
	private PageTaskExecutorSessionServerDispatcher localPageTaskExecutorSessionServerDispatcher;
	private RemotePageTaskExecutorPeer remotePageTaskExecutorPeer;
	private PageTaskExecutorPeerBroker localPageTaskExecutorPeerBroker;
	
	
	public static final String SESSION_SERVICE_PATH = AppConstants.APP_ID + '/'+ MarkReaderConstants.SESSION_SERVICE_NAME;

	public MarkReaderApp(int rmiPort, boolean isLocalPageTaskExecutionEnabled) throws UnknownHostException, IOException {
		System.setProperty("java.rmi.server.hostname", getHostname());
		long key = MarkReaderConstants.RANDOM.nextLong();
		Logger.getAnonymousLogger().info("MarkReaderController key=" + key);

		SessionSourceServerImpl sessionServer = SessionSourceServerImpl.createInstance(key,
				MarkReaderConstants.CLIENT_TIMEOUT_IN_SEC);
		this.formMasterFactory = new MergedFormMasterFactory();
		this.localPageTaskExecutorSessionServerDispatcher = new PageTaskExecutorSessionServerDispatcher(sessionServer, null, key);
		this.remotePageTaskExecutorPeer = new RemotePageTaskExecutorPeer(rmiPort, MarkReaderApp.SESSION_SERVICE_PATH);
		exportSessionService();
		if(isLocalPageTaskExecutionEnabled){
			this.localPageTaskExecutorPeerBroker = new PageTaskExecutorPeerBroker("Local", this.remotePageTaskExecutorPeer.getExecutorPeer(),
					this.localPageTaskExecutorSessionServerDispatcher);
		}
	}	

	private void exportSessionService() {
		try {
			SessionSourceServerImpl sessionService = SessionSourceServerImpl.getInstance();
			if (sessionService != null) {
				sessionService.close();
				RMIRegistryUtil.export(sessionService, this.remotePageTaskExecutorPeer.getRMIPort(), MarkReaderApp.SESSION_SERVICE_PATH);
			}
		} catch (IOException ignore) {
		}
	}

	public RemotePageTaskExecutorPeer getNetworkPeer() {
		return this.remotePageTaskExecutorPeer;
	}

	private String getHostname() {
		try {
			return NetworkUtil.Inet4.getHostAddress();
		} catch (SocketException ex) {
			return "localhost";
		}
	}
	
	public synchronized void shutdown() {
		new ShutdownAppService(this.remotePageTaskExecutorPeer, this.localPageTaskExecutorPeerBroker).call();
	}

	public static Preferences getPreferences() {
		return Preferences.userNodeForPackage(MarkReaderApp.class);
	}

	public synchronized MarkReaderSessionService createSession(File sourceDirectoryRoot) throws IOException{
		CreateSessionService service = new CreateSessionService(remotePageTaskExecutorPeer, formMasterFactory, localPageTaskExecutorSessionServerDispatcher, sourceDirectoryRoot);
		return service.call();
	}

	public void clearSession(File sourceDirectoryRoot) throws IOException{
		new ClearSessionService(sourceDirectoryRoot).call();
	}

	public synchronized void stopSession(File sourceDirectoryRootFile){
		new StopSessionService(sourceDirectoryRootFile).call();
	}

	public synchronized void closeSession(File sourceDirectoryRoot) {
		new CloseSessionService(sourceDirectoryRoot).call();
	}
}
