/*

XLSGeneratorEventHandler.java

Copyright 2004-2007 KUBO Hiroya (hiroya@cuc.ac.jp).

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Created on 2007/06/22

*/
package net.sqs2.omr.result.module.xls;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;

import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFCellStyle;
import org.apache.poi.hssf.usermodel.HSSFFont;
import org.apache.poi.hssf.usermodel.HSSFRichTextString;
import org.apache.poi.hssf.util.HSSFColor;

import net.sqs2.exigrid.source.PageID;
import net.sqs2.exigrid.source.SourceDirectory;
import net.sqs2.omr.master.FormArea;
import net.sqs2.omr.master.FormMaster;
import net.sqs2.omr.result.MarkAreaAnswer;
import net.sqs2.omr.result.MarkAreaAnswerItem;
import net.sqs2.omr.result.ResultPathFactory;
import net.sqs2.omr.result.TextAreaAnswer;
import net.sqs2.omr.result.event.QuestionEvent;
import net.sqs2.omr.result.event.QuestionItemEvent;
import net.sqs2.omr.result.event.RowEvent;
import net.sqs2.omr.result.event.TableEvent;
import net.sqs2.omr.result.event.ResultEventHandler;
import net.sqs2.util.FileUtil;
import net.sqs2.util.StringUtil;

public class XLSGeneratorEventHandler extends ResultEventHandler{
	private static final short NUM_HEADER_ROWS = 6; 
	private static final short NUM_HEADER_COLUMNS = 4;
	private static final String ITEM_SEPARATOR_CHAR = ",";
	
	private static final String MANUAL_SELECTED_ITEM_VALUE = "!";
	private static final String NOT_MANUAL_SELECTED_ITEM_VALUE = "";
	//private static final String SELECT1_NO_ANSWER_WARNING_TEXT = "";
	//private static final String SELECT1_MULTIPLE_ANSWER_WARNING_TEXT = "#";
	private static final String AUTO_SELECTED_ITEM_VALUE = "1";
	private static final String NOT_AUTO_SELECTED_ITEM_VALUE = "";
	private static final String UNKNOWN_FORMTYPE_WARNING_TEXT = "(unknown form type)";
	
	private VirtualSheetHSSFAdapter adapter = null; 
	
	private short physicalColumnIndexOfBodyPart = 0;
	private Select1Values select1Values = null;
	private int select1State = 0;
	
	public XLSGeneratorEventHandler() throws IOException {
	}

	@Override
	public void startTable(TableEvent ev) throws IOException {
		super.startTable(ev);
		this.adapter = new VirtualSheetHSSFAdapter(NUM_HEADER_ROWS, NUM_HEADER_COLUMNS);
		FormMaster master = ev.getFormMaster();
		initVirtualSheets(master);
		setNorthHeaderCellValues(master);
	}

	private void initVirtualSheets(FormMaster master) {
		
		short sheetIndex = 1;
		short numColumns = 0;

		String sheetName = null;
		for(String qid: master.getQIDSet()){
			if(sheetName == null){
				sheetName = "Sheet"+sheetIndex;
			}
			List<FormArea> formAreaList = master.getFormAreaList(qid);
			if(formAreaList == null || formAreaList.size() == 0){
				continue;
			}
			
			FormArea defaultFormArea = formAreaList.get(0);
			short numQIDColumns = 0;
			
			if(defaultFormArea.isSelect()){
				numQIDColumns = (short)formAreaList.size();
			}else{
				numQIDColumns = 1;
			}
			
			if(NUM_HEADER_COLUMNS + numColumns + numQIDColumns < VirtualSheetHSSFAdapter.NUM_COLUMNS_MAX){
				this.adapter.setNumColumns(sheetName, (short)(numColumns += numQIDColumns));
			}else{
				this.adapter.setNumColumns(sheetName, numQIDColumns);
				sheetName = null;
				sheetIndex++;
				numColumns = numQIDColumns;
			}
		}
	}

	private void setNorthHeaderCellValues(FormMaster master) {
		int virtualColumnIndex = 0;
		for(String qid: master.getQIDSet()){
			List<FormArea> formAreaList = master.getFormAreaList(qid);
			if(formAreaList == null || formAreaList.size() == 0){
				continue;
			}
			FormArea defaultFormArea = formAreaList.get(0);
			if(defaultFormArea.isSelect1() || defaultFormArea.isTextArea()){
				setHeaderCellValuesAsQuestion(virtualColumnIndex, defaultFormArea);
				setHeaderCellValuesAsSelect1(virtualColumnIndex, formAreaList, defaultFormArea);
				virtualColumnIndex ++;
			}else if(defaultFormArea.isSelect()){
				for(FormArea formArea: formAreaList){
					setHeaderCellValuesAsQuestion(virtualColumnIndex, defaultFormArea);
					setHeaderCellValuesAsSelect(virtualColumnIndex, formArea);
					virtualColumnIndex ++;
				}
			}
		}
	}

	private void setHeaderCellValuesAsQuestion(int virtualColumnIndex, FormArea defaultFormArea) {
		HSSFCell pageCell = this.adapter.getNorthHeaderCell(0, virtualColumnIndex);
		HSSFRichTextString pageString = new HSSFRichTextString(Integer.toString(defaultFormArea.getPage())); 
		pageCell.setCellValue(pageString);

		HSSFCell qidCell = this.adapter.getNorthHeaderCell(1, virtualColumnIndex);
		HSSFRichTextString qidString = new HSSFRichTextString(defaultFormArea.getQID()); 
		qidCell.setCellValue(qidString);

		HSSFCell typeCell = this.adapter.getNorthHeaderCell(2, virtualColumnIndex);
		HSSFRichTextString typeString = new HSSFRichTextString(defaultFormArea.getType()); 
		typeCell.setCellValue(typeString);

		HSSFCell hintsCell = this.adapter.getNorthHeaderCell(3, virtualColumnIndex);
		HSSFRichTextString hintsString = new HSSFRichTextString(StringUtil.join(defaultFormArea.getHints()," ")); 
		hintsCell.setCellValue(hintsString);
	}
	

	private void setHeaderCellValuesAsSelect(int virtualColumnIndex, FormArea formArea) {
		HSSFCell itemLabelCell = this.adapter.getNorthHeaderCell(4, virtualColumnIndex);
		HSSFRichTextString itemLabelString = new HSSFRichTextString(formArea.getItemLabel()); 
		itemLabelCell.setCellValue(itemLabelString);

		HSSFCell itemValueCell = this.adapter.getNorthHeaderCell(5, virtualColumnIndex);
		HSSFRichTextString itemValueString = new HSSFRichTextString(formArea.getItemValue()); 
		itemValueCell.setCellValue(itemValueString);
	}

	private void setHeaderCellValuesAsSelect1(int virtualColumnIndex, List<FormArea> formAreaList, FormArea defaultFormArea) {
		if(defaultFormArea.isSelect1()){
			StringBuilder itemLabelString = null;
			StringBuilder itemValueString = null;
			for(FormArea formArea: formAreaList){
				if(itemLabelString == null){
					itemLabelString = new StringBuilder();
					itemValueString = new StringBuilder();
				}else{
					itemLabelString.append(ITEM_SEPARATOR_CHAR);
					itemValueString.append(ITEM_SEPARATOR_CHAR);
				}
				itemLabelString.append(formArea.getItemLabel());
				itemValueString.append(formArea.getItemValue());
			}
			if(itemLabelString != null){
				HSSFCell itemLabelCell = this.adapter.getNorthHeaderCell(4, virtualColumnIndex);
				itemLabelCell.setCellValue(new HSSFRichTextString(itemLabelString.toString()));
				HSSFCell itemValueCell = this.adapter.getNorthHeaderCell(5, virtualColumnIndex);
				itemValueCell.setCellValue(new HSSFRichTextString(itemValueString.toString()));
			}
		}
	}

	private OutputStream createXLSOutputStream(FormMaster master, ResultPathFactory resultPathFactory, 
			SourceDirectory sourceDirectory) throws IOException {
		String masterName = FileUtil.getBasename(new File(master.getPath()).getName());
		File xlsFile = new File(resultPathFactory.getResultMasterDirectory( master.getPath(), 
				sourceDirectory.getPath()),
				masterName + ".xls");
		xlsFile.getParentFile().mkdirs();
		return new BufferedOutputStream(new FileOutputStream(xlsFile));
	}

	@Override
	public void endTable(TableEvent ev) throws IOException {
		OutputStream out = createXLSOutputStream(ev.getFormMaster(),
				ev.getResultPathFactory(),
				ev.getSourceDirectory());
		this.adapter.writeTo(out);
		out.close();
		super.endTable(ev);
	}

	@Override
	public void startRow(RowEvent ev) throws IOException {
		super.startRow(ev);
		int rowIndex = ev.getRowIndex();
		int numSheets = this.adapter.getNumSheets();
		HSSFCell[][] cellArray = new HSSFCell[NUM_HEADER_COLUMNS][numSheets];
		for(short columnIndex = 0; columnIndex < NUM_HEADER_COLUMNS; columnIndex++){
			cellArray[columnIndex] = this.adapter.getWestHeaderCellArray(rowIndex, (short)columnIndex);	
		}
		for(int sheetIndex = 0; sheetIndex < numSheets; sheetIndex++){
			cellArray[0][sheetIndex].setCellValue(rowIndex);
			String path = ev.getSourceParentDirectory().getPath();
			cellArray[1][sheetIndex].setCellValue(new HSSFRichTextString(path));
			String fileNames = getFileNames(ev);
			cellArray[2][sheetIndex].setCellValue(new HSSFRichTextString(fileNames));
			String errorMessage = ""; //TODO
			cellArray[3][sheetIndex].setCellValue(new HSSFRichTextString(errorMessage));
		}
		this.physicalColumnIndexOfBodyPart = 0;
	}
	
	private String getFileNames(RowEvent ev){
		StringBuilder value = new StringBuilder(); 
		int numPages = ev.getFormMaster().getNumPages();
		List<PageID> pageIDList = ev.getPageIDList();
		for (int pageIndex = 0; pageIndex < numPages; pageIndex++) {
			PageID pageID = pageIDList.get(pageIndex + ev.getRowIndex() * numPages);
			if (0 < pageIndex) {
				value.append(ITEM_SEPARATOR_CHAR);
			}
			value.append(new File(pageID.getPath()).getName());
		}
		return value.toString();
	}
	

	private void setCellAsSelect1(QuestionEvent ev) {
		MarkAreaAnswer markAreaAnswer = (MarkAreaAnswer)ev.getAnswer();
		HSSFCell cell = this.adapter.getBodyCell(ev.getRowIndex(), this.physicalColumnIndexOfBodyPart);
		if(markAreaAnswer.isNoAnswer()){		
			/* NO ANSWER */	
			cell.setCellStyle(createCellStyle(HSSFColor.BLACK.index, HSSFColor.ROSE.index));
			//cell.setCellValue(new HSSFRichTextString(SELECT1_NO_ANSWER_WARNING_TEXT));
			this.select1Values = null;
			this.select1State = 0;
			this.physicalColumnIndexOfBodyPart++;			
		}else if(1 < markAreaAnswer.getNumMarkedAnswerItems()){
			/* MORE THAN ONE */
			this.select1Values = new Select1Values();
			//this.select1Values.add(SELECT1_MULTIPLE_ANSWER_WARNING_TEXT, HSSFColor.BLACK.index);
			this.select1State = markAreaAnswer.getNumMarkedAnswerItems();
		}else if(1 == markAreaAnswer.getNumMarkedAnswerItems()){
			/* ONE */
			this.select1Values = new Select1Values();
			this.select1State = 1;
			/*
		}else{
			HSSFCellStyle style = this.adapter.createCellStyle();
			style.setFillBackgroundColor(HSSFColor.BLACK.index);
			style.setFillPattern(HSSFCellStyle.SOLID_FOREGROUND);
			cell.setCellStyle(style);
			cell.setCellValue(new HSSFRichTextString("-"));
			this.select1CellValue = null;
			this.physicalColumnIndexOfBodyPart++;
			*/
		}
	}

	private HSSFCellStyle createCellStyle(short fontColor, short backgroundColor) {
		HSSFFont font = this.adapter.createFont();
		font.setColor(fontColor);

		HSSFCellStyle style = this.adapter.createCellStyle();
		style.setFont(font);
		style.setFillForegroundColor(backgroundColor);
		style.setFillPattern(HSSFCellStyle.SOLID_FOREGROUND);
		return style;
	}

	private void setCellAsTextArea(QuestionEvent ev) {
		HSSFCell cell = this.adapter.getBodyCell(ev.getRowIndex(), this.physicalColumnIndexOfBodyPart);
		cell.setCellStyle(createCellStyle(HSSFColor.BLACK.index, HSSFColor.GREY_25_PERCENT.index));
		cell.setCellValue(
				new HSSFRichTextString( ((TextAreaAnswer)ev.getAnswer()).getValue() )
		);
		this.physicalColumnIndexOfBodyPart++;
	}

	private void setCellAsUnknownFormType(QuestionEvent ev) {
		HSSFCell cell = this.adapter.getBodyCell(ev.getRowIndex(), this.physicalColumnIndexOfBodyPart);
		cell.setCellStyle(createCellStyle(HSSFColor.RED.index, HSSFColor.YELLOW.index));
		cell.setCellValue(
				new HSSFRichTextString(UNKNOWN_FORMTYPE_WARNING_TEXT)
		);
		this.physicalColumnIndexOfBodyPart++;
	}

	@Override
	public void startQuestion(QuestionEvent ev) throws IOException{
		
		super.startQuestion(ev);
		
		FormArea formArea = ev.getDefaultFormArea();
		if(formArea.isSelect1()){
			setCellAsSelect1(ev);
		}else if(formArea.isSelect()){
			/* setCell in startQuestionItem */
			/* so, do nothing  */ 
		}else if(formArea.isTextArea()){
			setCellAsTextArea(ev);
		}else{
			setCellAsUnknownFormType(ev);
		}
	}
	
	@Override
	public void endQuestion(QuestionEvent ev) throws IOException{
		FormArea formArea = ev.getDefaultFormArea();
		if(formArea.isSelect1() && this.select1Values != null){
			HSSFCell cell = this.adapter.getBodyCell(ev.getRowIndex(), this.physicalColumnIndexOfBodyPart);
			if(1 == this.select1State){
				setCellValue(cell, this.select1Values.get(0).toString());
			}else if(1 < this.select1State){
				HSSFRichTextString value = this.select1Values.toRichTextString();
				cell.setCellValue(value);
			}
			this.select1Values = null;
			this.physicalColumnIndexOfBodyPart++;
		}
		
		super.endQuestion(ev);

	}

	@Override
	public void endQuestionItem(QuestionItemEvent ev) throws IOException {
		FormArea formArea = ev.getFormArea();
		if(formArea.isSelect() || (formArea.isSelect1() && this.select1Values != null)){
			MarkAreaAnswerItem markAreaAnswerItem = ev.getMarkAreaAnswerItem();
			if(formArea.isSelect()){
				HSSFCell cell = this.adapter.getBodyCell(ev.getRowIndex(), this.physicalColumnIndexOfBodyPart);
				endQuestionItemAsSelect(cell, formArea, markAreaAnswerItem);
			}else if(formArea.isSelect1()){
				endQuestionItemAsSelect1(formArea, markAreaAnswerItem);
			}
		}
		super.endQuestionItem(ev);
	}

	private void endQuestionItemAsSelect(HSSFCell cell, FormArea formArea,
			MarkAreaAnswerItem markAreaAnswerItem) {
		if(0 < formArea.getItemIndex()){
		}
		String value = null;
		if (markAreaAnswerItem.isManualMode()) {
			if (markAreaAnswerItem.isManualSelected()) {
				value = MANUAL_SELECTED_ITEM_VALUE;
			}else{
				value = NOT_MANUAL_SELECTED_ITEM_VALUE;
			}
		}else{
			if (markAreaAnswerItem.isAutoSelected()) {
				value = AUTO_SELECTED_ITEM_VALUE;
			}else{
				value = NOT_AUTO_SELECTED_ITEM_VALUE;
			}
		}
		setCellValue(cell, value);
		this.physicalColumnIndexOfBodyPart++;
	}

	private void endQuestionItemAsSelect1(FormArea formArea,
			MarkAreaAnswerItem markAreaAnswerItem) {
		if(this.select1Values.size() != 0 && 1 < this.select1State){
			this.select1Values.add(ITEM_SEPARATOR_CHAR, HSSFColor.RED.index);
		}
		
		if (markAreaAnswerItem.isManualMode() && markAreaAnswerItem.isManualSelected()) {
			// manually selected 
			this.select1Values.add(formArea.getItemValue(), HSSFColor.BLUE.index);
		}else if (markAreaAnswerItem.isAutoSelected()) {
			// the most likely selected answer item 
			this.select1Values.add(formArea.getItemValue(), HSSFColor.BLACK.index);
		}
		/*
		if (markAreaAnswerItem.isMarked()) {
			// miss-marked answer item
			this.select1Values.add(formArea.getItemValue(), HSSFColor.GREY_25_PERCENT.index);
		}else{
			// not marked answer item
		}*/
	}

	private void setCellValue(HSSFCell cell, String value) {
		try{
			double num = Double.parseDouble(value);
			cell.setCellValue(num);
		}catch(NumberFormatException ex){			
			cell.setCellValue(new HSSFRichTextString( value ));	
		}
	}
	
	class Select1Values{
		ArrayList<ColoredString> list = new ArrayList<ColoredString>();
		
		public void add(String value, short color){
			this.list.add(new ColoredString(value, color));
		}
		
		public HSSFRichTextString toRichTextString(){
			
			StringBuilder builder = new StringBuilder(); 
			for(ColoredString c: list){
				builder.append(c.value);
			}

			HSSFRichTextString ret = new HSSFRichTextString(builder.toString());
			int start = 0;
			for(ColoredString c: list){
				int end = start + c.value.length() - 1;
				HSSFFont font = adapter.createFont();
				font.setColor(c.color);
				ret.applyFont(start, end, font);
				start = end + 1;
			}

			return ret;
		}
		
		public ColoredString get(int index){
			return this.list.get(index);
		}
		
		public int size(){
			return this.list.size();
		}

	}
	
	class ColoredString{
		String value;
		short color;
		ColoredString(String value, short color){
			this.value = value; 
			this.color = color;
		}
		
		@Override 
		public String toString(){
			return this.value;
		}
		
	}

}