/**
   ChartGeneratorEventHandler.java

 Copyright 2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Created on 2007/03/03
 Author hiroya
 */
package net.sqs2.omr.result.module.jfreechart;

import java.io.IOException;
import java.io.File;
import java.util.List;
import java.util.logging.Logger;

import net.sqs2.exigrid.source.SourceDirectory;
import net.sqs2.omr.master.FormArea;
import net.sqs2.omr.master.FormMaster;
import net.sqs2.omr.result.AnswerItemBag;
import net.sqs2.omr.result.ResultPathFactory;
import net.sqs2.omr.result.event.ResultEventHandler;
import net.sqs2.omr.result.event.TableEvent;
import net.sqs2.omr.result.event.TableEventContext.MarkAreaAnswerItemsByColumn;

import org.jfree.data.general.DefaultPieDataset;
import org.jfree.data.general.PieDataset;
import org.jfree.data.category.CategoryDataset;
import org.jfree.data.category.DefaultCategoryDataset;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.ChartFactory;
import org.jfree.chart.ChartUtilities;
import org.jfree.chart.plot.PlotOrientation;

public class ChartGeneratorEventHandler extends ResultEventHandler{

    public ChartGeneratorEventHandler() throws IOException {
    }

    @Override
    public void endTable(TableEvent ev) throws IOException {
    	int columnIndex = 0;
    	for(String qid: ev.getFormMaster().getQIDSet()){
    		Logger.getLogger("chart").info(Integer.valueOf(columnIndex).toString()+"/"+ev.getNumColumns());
    		outputChart(ev, columnIndex, ev.getFormMaster().getFormAreaList(qid));
    		columnIndex++;
    	}
    	super.endTable(ev);
    }
    
    private void outputChart(TableEvent ev, int columnIndex, List<FormArea> formAreaList) throws IOException {
    	// answers of this column
    	FormArea formArea = formAreaList.get(0);
    	if(! formArea.isMarkArea()) {
    		return;
    	}
    	MarkAreaAnswerItemsByColumn markAreaAnswerByColumn = ev.getTableEventContext().getMarkAreaAnswerItemMapByColumn().get(columnIndex);
    	AnswerItemBag answerItemBag = markAreaAnswerByColumn.getAnswerItemBag();
    	// check a type of this question
    	switch (formArea.getTypeCode()) {
    	case FormArea.SELECT:   // answer type: multiple select
    		outputBarChart(ev.getResultPathFactory(), ev.getFormMaster(), ev.getSourceDirectory(),  
    				columnIndex, formArea, createBarChartDataset(answerItemBag, formAreaList) );
    		break;

    	case FormArea.SELECT1:  // answer type: single select
    		outputPieChart(ev.getResultPathFactory(), ev.getFormMaster(), ev.getSourceDirectory(), 
    				columnIndex, formArea, createPieChartDataset(answerItemBag, formAreaList) );
    		break;

    	case FormArea.INPUT:    // answer type: input text
    	case FormArea.TEXTAREA:
    		// do nothing
    		break;
    	}
    }

    private void outputBarChart(ResultPathFactory resultPathFactory, FormMaster formMaster, SourceDirectory sourceDirectory, int columnIndex, FormArea formArea, CategoryDataset dset ) throws IOException {
        JFreeChart chart = ChartFactory.createBarChart(
            formArea.getHint(),         // chart title
            "選択肢",               // domain axis label
            "選択数",                  // range axis label
            dset,                     // data
            PlotOrientation.VERTICAL, // orientation
            true,                     // include legend
            true,                     // tooltips?
            false                     // URLs?
        );

        File chartFile = createChartFile(resultPathFactory, formMaster, sourceDirectory, columnIndex);
        ChartUtilities.saveChartAsJPEG(chartFile, chart, 600, 600);
    }

    private void outputPieChart(ResultPathFactory resultPathFactory, FormMaster formMaster, SourceDirectory sourceDirectory, 
    		int columnIndex, FormArea formArea, PieDataset pData) throws IOException {
        JFreeChart chart = ChartFactory.createPieChart(
            formArea.getHint(),         // chart title
            pData,                     // data
            true,                     // include legend
            true,                     // tooltips?
            false                     // URLs?
        );

        File chartFile = createChartFile(resultPathFactory, formMaster, sourceDirectory, columnIndex);
        ChartUtilities.saveChartAsJPEG(chartFile, chart, 600, 400);
    }

    private File createChartFile(ResultPathFactory resultPathFactory, FormMaster formMaster, 
    		SourceDirectory sourceDirectory, int columnIndex) {

        // create chart directory
        File chartDirectory = new File(resultPathFactory.getResultMasterDirectory(formMaster.getPath(), sourceDirectory.getPath()).getAbsolutePath()
        		+ File.separatorChar + "chart");

        if( ! chartDirectory.exists() ){
        	chartDirectory.mkdirs();
        }

        // create chart file handler
        File chartFile = new File(chartDirectory.getAbsolutePath()
        		+ File.separatorChar
        		+ columnIndex
        		+ ".jpg"
        );
        return chartFile;
    }    
    
    private PieDataset createPieChartDataset(AnswerItemBag ans, List<FormArea> formAreaList) {
        int index = 0;

        DefaultPieDataset dset = new DefaultPieDataset();
        for (FormArea area : formAreaList) {
                dset.setValue(area.getItemLabel(), new Double( ans.getCount(index) ) );
            index++;
        }
        dset.setValue("無回答", ans.countNoAnswer());

        return dset;
    }

    private CategoryDataset createBarChartDataset(AnswerItemBag ans, List<FormArea> formAreaList) {
        int index = 0;
        DefaultCategoryDataset dset = new DefaultCategoryDataset();
        for (FormArea area : formAreaList ){
            dset.setValue(ans.getCount(index), area.getItemLabel(), area.getItemLabel() );
            index++;
        }
        
        return dset;
    }

}
