/**
 *  ResultHTMLGeneratorEventHandler.java

 Copyright 2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Created on 2007/03/03
 Author hiroya
 */
package net.sqs2.omr.result.module.html;

import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Collection;
import java.util.List;
import java.util.SortedMap;

import net.sqs2.exigrid.master.PageMaster;
import net.sqs2.exigrid.source.SourceDirectory;
import net.sqs2.omr.httpd.AreaImageContentServlet;
import net.sqs2.omr.master.FormArea;
import net.sqs2.omr.master.FormMaster;
import net.sqs2.omr.result.MarkAreaAnswerItem;
import net.sqs2.omr.result.AnswerItemBag;
import net.sqs2.omr.result.ResultPathFactory;
import net.sqs2.omr.result.TextAreaAnswer;
import net.sqs2.omr.result.event.ResultEventImpl;
import net.sqs2.omr.result.event.ResultEventHandler;
import net.sqs2.omr.result.event.TableEvent;
import net.sqs2.omr.result.event.TableEventContext.MarkAreaAnswerItemsByColumn;
import net.sqs2.omr.result.event.TableEventContext.TextAreaAnswerListByColumn;
import net.sqs2.omr.result.module.core.HTMLErrorLogEventHandler;

import net.sqs2.util.FileUtil;
import net.sqs2.util.HTMLUtil;
import net.sqs2.util.PathUtil;
import net.sqs2.util.StringUtil;

public class ResultHTMLGeneratorEventHandler extends ResultEventHandler{

	private class MasterDirSourceDirIndexWriter {
		private PrintWriter writer = null;
		final String TITLE = "Contents: ";

		MasterDirSourceDirIndexWriter(ResultPathFactory pathFactory) throws IOException {
			File index = pathFactory.getResultIndexFile();
			index.getParentFile().mkdirs();
			String encoding = "UTF-8";
			this.writer = FileUtil.createPrintWriter(index, encoding);
			HTMLUtil.printHTMLHead(this.writer, encoding, TITLE +" - "+pathFactory.getSourceDirectoryRoot());
		}

		void printHeader() {
			this.writer.println("<body>");
			this.writer.println("<h1>");
			this.writer.println(HTMLUtil.escapeHTML(TITLE));
			this.writer.println("</h1>");
			this.writer.println("<ul>");
		}

		void writeIndexItem(TableEvent ev) throws IOException {
			ResultPathFactory pathFactory = ev.getResultPathFactory();
			FormMaster master = ev.getFormMaster();
			File resultMasterIndexFile = pathFactory.getResultMasterDirectoryIndexFile(master.getPath(), ev.getSourceDirectory().getPath());
			File resultIndexFile = pathFactory.getResultIndexFile();
			String masterIndexURI = PathUtil.getRelativePath(resultMasterIndexFile, resultIndexFile);
			printMasterIndexItem(masterIndexURI, master.getPath(), ev.getSourceDirectory().getPath());
		}
		
		void printMasterIndexItem(String masterIndexURI, String masterPath, String sourceDirectoryPath) {
			this.writer.print("<li><a href=\"");
			this.writer.print(HTMLUtil.encodePath(masterIndexURI));
			this.writer.print("\"> ");
			this.writer.print(FileUtil.getBasename(masterPath));
			this.writer.print(':');
			this.writer.print(HTMLUtil.escapeHTML(sourceDirectoryPath));
			this.writer.print("</a></li>");
			this.writer.println();
		}

		void printFooter() {
			this.writer.println("</ul>");
		}

		void close() {
			this.writer.println("</body>");
			this.writer.println("</html>");
			this.writer.close();
		}
	}
	
	private class SourceDirectoryIndexWriter {
		private TableEvent ev;
		private PrintWriter writer = null;
		private File sourceDirectoryIndexFile;
		final private String TITLE = "Contents: ";
		
		private SourceDirectoryIndexWriter(TableEvent ev) throws IOException {
			this.ev = ev;
			this.writer = createPrintWriter();
		}

		private PrintWriter createPrintWriter() throws IOException {
			FormMaster master = ev.getFormMaster();
			this.sourceDirectoryIndexFile = ev.getResultPathFactory().getResultMasterDirectoryIndexFile(master.getPath(), ev.getSourceDirectory().getPath());
			this.sourceDirectoryIndexFile.getParentFile().mkdirs();
			String encoding = "UTF-8";
			PrintWriter writer = FileUtil.createPrintWriter(this.sourceDirectoryIndexFile, encoding);
			HTMLUtil.printHTMLHead(writer, encoding, TITLE+" - " + FileUtil.getBasename(master.getPath()));
			writer.println("<body>");
			return writer;
		}
		
		void writeColumnIndex(TableEvent ev, int columnIndex, int numErrors, FormArea formArea) throws IOException {
			File columnFile = ev.getResultPathFactory().getColumnFile(ev.getFormMaster().getPath(), 
					ev.getSourceDirectory().getPath(), columnIndex, "");
			File resultMasterDirectory = ev.getResultPathFactory().getResultMasterDirectory(ev.getFormMaster().getPath(), 
					ev.getSourceDirectory().getPath());
			String columnFilePath = PathUtil.getRelativePath(columnFile, resultMasterDirectory);
			printColumnIndex(columnFilePath, numErrors, columnIndex, formArea.getLabel(), formArea.getType(), formArea.getHints());
		}

		private File getFile(){
			return this.sourceDirectoryIndexFile;
		}

		private void printTableTitle(TableEvent ev, PageMaster master, SourceDirectory sourceDirectory) {
			String basename = FileUtil.getBasename(master.getPath());
			String xlsPath = basename+".xls";
			String csvPath = basename+"-csv.txt";
			String context = ev.getFormMaster().getPath()+ev.getSourceDirectory().getPath();
            //String texteditPath = HTMLUtil.encodePath("/textedit/"+context)+"/?page=1";
            //String markeditPath = HTMLUtil.encodePath("/markedit/"+context)+"/?page=1";

            this.writer.print("<h2>");
			this.writer.print(HTMLUtil.escapeHTML(TITLE));
			this.writer.print("</h2>");
			this.writer.println("<h3>");
			this.writer.print(HTMLUtil.escapeHTML(basename));
			this.writer.print(": ");
			this.writer.print(HTMLUtil.escapeHTML(sourceDirectory.getPath()));
			this.writer.println("</h3>");
			
			this.writer.print("<div>");
			this.writer.print("<a href=\"");
			this.writer.print(HTMLUtil.encodePath(xlsPath));
			this.writer.print("\">");
			this.writer.print("Excel");
			this.writer.print("</a>");
			this.writer.print("|");
			this.writer.print("<a href=\"");
			this.writer.print(HTMLUtil.encodePath(csvPath));
			this.writer.print("\">");
			this.writer.print("CSV");
			this.writer.print("</a>");
			this.writer.println("</div>");

            this.writer.println("<ul>");
		}

		private void printColumnIndex(String columnFilePath, int numErrors, int columnIndex, String label, String type, String hints[]) {
			this.writer.print("<li><a href=\"");
			this.writer.print(HTMLUtil.encodePath(columnFilePath));
			this.writer.print("\">");
			//this.writer.print(HTMLUtil.escapeHTML(String.valueOf(columnIndex+1)));
			this.writer.print(HTMLUtil.escapeHTML(label));
			this.writer.print(" : ");
			this.writer.print(type);
			this.writer.print(" : ");
			this.writer.print(HTMLUtil.escapeHTML(StringUtil.join(hints, "")));
			this.writer.print("</a>");
			if (0 < numErrors) {
				this.writer.print(" (");
				this.writer.print(numErrors);
				this.writer.print(')');
			}
			this.writer.print("</li>");
			this.writer.println();
		}

		void close() {
			this.writer.println("</ul>");
			this.writer.println("</body>");
			this.writer.println("</html>");
			this.writer.close();
		}
	}
	
	private class ColumnWriter {
		protected TableEvent ev;
		protected int columnIndex;
		protected FormArea defaultFormArea;
		protected File sourceDirectoryIndexFile;
		protected PrintWriter writer = null;
		
		protected ColumnWriter(TableEvent ev, int columnIndex, FormArea defaultFormArea, File sourceDirectoryIndexFile) throws IOException {
			this.ev = ev;
			this.columnIndex = columnIndex;
			this.defaultFormArea = defaultFormArea;
			this.sourceDirectoryIndexFile = sourceDirectoryIndexFile;
			this.writer = createPrintWriter(sourceDirectoryIndexFile);
		}
		
		protected PrintWriter createPrintWriter(File sourceDirectoryIndexFile) throws IOException {
			File sourceDirectoryFile = sourceDirectoryIndexFile.getParentFile();
			sourceDirectoryFile.mkdirs();
			File columnFile = new File(sourceDirectoryFile.getAbsolutePath() + File.separatorChar + this.columnIndex + ".html");
			String encoding = "UTF-8";
			PrintWriter writer = FileUtil.createPrintWriter(columnFile, encoding);
			HTMLUtil.printHTMLHead(writer, encoding, this.ev.getFormMaster().getPath()+":"+StringUtil.join(defaultFormArea.getHints(), ""));
			return writer;
		}

		protected void printTableTitle() {
			this.writer.print("<h2>");
			this.writer.print(HTMLUtil.escapeHTML(FileUtil.getBasename(this.ev.getFormMaster().getPath())));
			this.writer.print(":");
			this.writer.print(HTMLUtil.escapeHTML(this.ev.getSourceDirectory().getPath()));
			this.writer.println("</h2>");
		}

		protected void writeColumnTitle() {
			printTableTitle();
			this.writer.print("<h3>");
			this.writer.print(HTMLUtil.escapeHTML(this.defaultFormArea.getLabel()));
			this.writer.print(":");
			this.writer.print(HTMLUtil.escapeHTML(this.defaultFormArea.getType()));
			this.writer.print(":");
			this.writer.print(HTMLUtil.escapeHTML(StringUtil.join(this.defaultFormArea.getHints(), "")));
			this.writer.println("</h3>");
			this.writer.println("<br/>");
		}
		
		protected void close() {
			this.writer.println("</body>");
			this.writer.println("</html>");
			this.writer.close();
		}
	}
	
	private class TextAreaColumnWriter extends ColumnWriter{

		protected TextAreaColumnWriter(TableEvent ev, int columnIndex, FormArea defaultFormArea, File sourceDirectoryIndexFile) throws IOException {
			super(ev, columnIndex, defaultFormArea, sourceDirectoryIndexFile);
		}

		private void writeBody(){
			SortedMap<Integer,TextAreaAnswerListByColumn> textAreaAnswerMapByColumn = this.ev.getTableEventContext().getTextAreaAnswerListMapByColumn();
			TextAreaAnswerListByColumn textAnswerByColumn = textAreaAnswerMapByColumn.get(this.columnIndex);
			int rowIndex = 0;
			//System.out.println(textAreaAnswerMapByColumn);
			for(TextAreaAnswer textAnswer : textAnswerByColumn){
				String textAreaPath = textAnswer.getAreaImageFilePath();
				if(textAreaPath == null){
					this.writer.println("(ERROR: textAreaImageFilePath == null)<br/>");
					rowIndex++;
					continue;
				}
				//Logger.getAnonymousLogger().info("TextAreaPath:"+textAreaPath);
				String sourceDirectoryIndexPath = this.sourceDirectoryIndexFile.getPath();
				String src = PathUtil.getRelativePath(textAreaPath, sourceDirectoryIndexPath);
				String alt = "rowIndex:"+rowIndex+", columnIndex:"+this.columnIndex;
				this.writer.print("<img class=\"textareaImage\" src=\"");
				this.writer.print(HTMLUtil.encodePath(src));
				this.writer.print("\" alt=\"");
				this.writer.print(HTMLUtil.escapeHTML(alt));
				this.writer.println("\"/>");
				this.writer.println("<br/>");
				rowIndex++;
			}
		}
	}

	private class MarkAreaColumnWriter extends ColumnWriter{

		MarkAreaAnswerItemsByColumn markAreaAnswerByColumn;

		protected MarkAreaColumnWriter(TableEvent ev, int columnIndex, FormArea defaultFormArea, File sourceDirectoryIndexFile) throws IOException {
			super(ev,  columnIndex, defaultFormArea, sourceDirectoryIndexFile);
			this.markAreaAnswerByColumn = this.ev.getTableEventContext().getMarkAreaAnswerItemMapByColumn().get(this.columnIndex);
		}
		
		private void writeHistgram() {
			this.writer.print("<table border='1'>");
			writeHistgramTHead();
			writeHistgramTBody(this.defaultFormArea.getQID());
			this.writer.println("</table>");
		}

		private void writeHistgramTBody(String qid) {
			this.writer.println("<tbody>");
			for (int densityIndex = 0; densityIndex < densityStatResolution; densityIndex++) {
				Collection<MarkAreaAnswerItem> answerItems = this.markAreaAnswerByColumn.getDensityAnswerItemMap().get(densityIndex);
				if (answerItems == null || answerItems.size() == 0) {
					/*
					 * masterColumnWriter.print("<tr>");
					 * masterColumnWriter.print("<td colspan='2'></td>");
					 * masterColumnWriter.println("</tr>");
					 */
					continue;
				}
				writeHistgramTR(qid, densityIndex, answerItems);
			}
			this.writer.println("</tbody>");
		}

		private void writeHistgramTR(String qid, int densityIndex, Collection<MarkAreaAnswerItem> answerItems) {
			
			this.writer.print("<tr>");

			if ((densityIndex + 1) / 1.0 / densityStatResolution < blackThreshold / 255.0) {
				this.writer.print("<th style='background:gray'>");
			} else {
				this.writer.print("<th>");
			}
			this.writer.print(densityIndex / 1.0 / densityStatResolution);
			this.writer.print("-");
			this.writer.print((densityIndex + 1) / 1.0 / densityStatResolution);
			this.writer.println("</th>");
			writeHistgramTD(qid, answerItems);
			this.writer.println("</tr>");
		}
		
		private AnswerItemBag getAnswerItemBag(){
			return this.markAreaAnswerByColumn.getAnswerItemBag();
	    }

		private void writeHistgramTD(String qid, Collection<MarkAreaAnswerItem> answerItems) {
			
			this.writer.print("<td>");

			for (MarkAreaAnswerItem answerItem : answerItems) {
				String uri = "/"+AreaImageContentServlet.getContextString()+"/"+answerItem.getAreaImageURIPath();
				HTMLUtil.printInlineImage(this.writer, uri, uri);
				this.writer.println(answerItem.getItemIndex());
			}

			this.writer.print("</td>");
		}

		private void writeHistgramTHead() {
			this.writer.print("<thead>");
			this.writer.print("<tr>");
			this.writer.println("<th>index</th>");
			this.writer.println("<th>mark</th>");
			this.writer.print("</tr>");
			this.writer.println("</thead>");
		}

		private void writeStat() throws IOException {
			this.writer.print("<table border='1'>");
			writeStatTHead();
			writeStatTBody();
			this.writer.println("</table>");

			writeMultiMarkWarnings();
		}

		private void writeStatTHead() {
			this.writer.print("<thead>");
			this.writer.print("<tr>");
			this.writer.print("<th>index</th>");
			this.writer.print("<th>label</th>");
			this.writer.print("<th>value</th>");
			this.writer.print("<th colspan='2'>count</th>");
			this.writer.print("</tr>");
			this.writer.println("</thead>");
		}

		private void writeStatTBody() {
			this.writer.println("<tbody>");
			writeAnswerItemStat();
			int c = getAnswerItemBag().countNoAnswer();
			this.writer.print("<tr>");
			this.writer.print("<th>-</th>");
			this.writer.print("<th>N/A</th>");
			this.writer.print("<td>-</td>");
			this.writer.print("<td>");
			this.writer.print(c);
			this.writer.print("</td>");
			this.writer.print("<td><span style='color:#a00'>");
			writeBar(c);
			this.writer.print("</span></td>");
			this.writer.print("</tr>");
			this.writer.println("</tbody>");
		}

		private void writeAnswerItemStat() {
			List<FormArea> formAreaList = ev.getFormMaster().getFormAreaList(this.columnIndex);
			for (FormArea formArea : formAreaList) {
				int count = getAnswerItemBag().getCount(formArea.getItemIndex());
				this.writer.print("<tr>");
				this.writer.print("<th>");
				this.writer.print(formArea.getItemIndex());
				this.writer.print("</th>");
				this.writer.print("<td>");
				this.writer.print(HTMLUtil.escapeHTML(formArea.getItemLabel()));
				this.writer.print("</td>");
				this.writer.print("<td>");
				this.writer.print(HTMLUtil.escapeHTML(formArea.getItemValue()));
				this.writer.print("</td>");
				this.writer.print("<td>");
				this.writer.print(count);
				this.writer.print("</td>");
				this.writer.print("<td><span style='color:#a00'>");
				writeBar(count);
				this.writer.print("</span></td>");
				this.writer.println("</tr>");
			}
		}

		private void writeMultiMarkWarnings() throws IOException {
			errorLogEventHandler.writeMultiMarkWarnings(this.ev);
		}

		private void writeBar(int count) {
			for (int i = count; 0 < i; i--) {
				this.writer.print("*");
			}
		}
	}
	
	
	private int blackThreshold;
	private int densityStatResolution;

	private MasterDirSourceDirIndexWriter masterDirSourceIndexWriter;
	private HTMLErrorLogEventHandler errorLogEventHandler;

	public ResultHTMLGeneratorEventHandler(int blackThreshold,
			int densityStatResolution,
			HTMLErrorLogEventHandler errorLogEventHandler) throws IOException {
		this.blackThreshold = blackThreshold;
		this.densityStatResolution = densityStatResolution;
		this.errorLogEventHandler = errorLogEventHandler;
	}
	
	@Override
	public void startResult(ResultEventImpl ev) throws IOException {
		super.startResult(ev);
		this.masterDirSourceIndexWriter = new MasterDirSourceDirIndexWriter(ev.getResultPathFactory());
		this.masterDirSourceIndexWriter.printHeader();
	}
	
	@Override
	public void endResult(ResultEventImpl ev) throws IOException {
		this.masterDirSourceIndexWriter.printFooter();
		this.masterDirSourceIndexWriter.close();
		super.endResult(ev);
	}

	@Override
	public void startTable(TableEvent ev) throws IOException {
		super.startTable(ev);
		this.masterDirSourceIndexWriter.writeIndexItem(ev);
	}
	
	@Override
	public void endTable(TableEvent ev) throws IOException {
		FormMaster master = ev.getFormMaster();
		SourceDirectoryIndexWriter sourceDirectoryIndexWriter = new SourceDirectoryIndexWriter(ev);
		sourceDirectoryIndexWriter.printTableTitle(ev, master, ev.getSourceDirectory());

		int columnIndex = 0;
		int numErrors = getNumErrors(ev);

		for(String qid: master.getQIDSet()){
			List<FormArea> formAreaList = master.getFormAreaList(qid);
			FormArea defaultFormArea = formAreaList.get(0);

			if (defaultFormArea.isTextArea() || defaultFormArea.isMarkArea()) {
				sourceDirectoryIndexWriter.writeColumnIndex(ev, columnIndex, numErrors, defaultFormArea);
			}

			if (defaultFormArea.isMarkArea()) {
				MarkAreaColumnWriter markAreaColumnWriter = new MarkAreaColumnWriter(ev, columnIndex, defaultFormArea, sourceDirectoryIndexWriter.getFile());
				markAreaColumnWriter.writeColumnTitle();				
				markAreaColumnWriter.writeStat();
				markAreaColumnWriter.writeHistgram();
				markAreaColumnWriter.close();
			}else if(defaultFormArea.isTextArea()){
				TextAreaColumnWriter textAreaColumnWriter = new TextAreaColumnWriter(ev, columnIndex, defaultFormArea, sourceDirectoryIndexWriter.getFile());
				//master, ev.getSourceDirectory(), sourceDirectoryIndexWriter.getFile(), columnIndex, formAreaList, "-"
				textAreaColumnWriter.writeColumnTitle();				
				textAreaColumnWriter.writeBody();
				textAreaColumnWriter.close();
			}
			columnIndex++;
		}
		sourceDirectoryIndexWriter.close();
		super.endTable(ev);
	}
	
	private int getNumErrors(TableEvent ev){
		 // TODO: calc numErrors
		return -1;
	}
}
