/*

 CSVGeneratorEventHandler.java

 Copyright 2004-2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Created on 2007/04/07

 */
package net.sqs2.omr.result.module.csv;

import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.List;

import net.sqs2.exigrid.master.PageMaster;
import net.sqs2.exigrid.source.PageID;
import net.sqs2.exigrid.source.SourceDirectory;
import net.sqs2.omr.master.FormArea;
import net.sqs2.omr.master.FormMaster;
import net.sqs2.omr.result.MarkAreaAnswer;
import net.sqs2.omr.result.MarkAreaAnswerItem;
import net.sqs2.omr.result.ResultPathFactory;
import net.sqs2.omr.result.event.QuestionItemEvent;
import net.sqs2.omr.result.event.ResultEventHandler;
import net.sqs2.omr.result.event.QuestionEvent;
import net.sqs2.omr.result.event.RowEvent;
import net.sqs2.omr.result.event.TableEvent;
import net.sqs2.util.FileUtil;
import net.sqs2.util.StringUtil;

public class CSVGeneratorEventHandler extends ResultEventHandler{

	private PrintWriter csvWriter = null;

	private static final char TAB = '\t';

	private static final char ITEM_SEPARATOR = ',';

	private static final boolean VERBOSE_PRINT_PAGE = true;

	private static final boolean VERBOSE_PRINT_QID = true;

	public CSVGeneratorEventHandler() throws IOException {
	}

	@Override
	public void startTable(TableEvent ev) throws IOException {
		super.startTable(ev);
		FormMaster master = ev.getFormMaster();
		this.csvWriter = createCSVWriter(ev.getResultPathFactory(), master, ev.getSourceDirectory());
		writeHeaderRows(master);
	}

	@Override
	public void endTable(TableEvent ev) throws IOException {
		this.csvWriter.close();
		super.endTable(ev);
	}

	private void writeHeaderRows(FormMaster master) {
		writeHeaderPageRow(master);
		writeHeaderQIDRow(master);
		writeHeaderTypeRow(master);
		writeHeaderHintsRow(master);
		writeHeaderItemLabelRow(master);
		writeItemValueRow(master);
	}

	private void writeItemValueRow(FormMaster master) {
		this.csvWriter.print(TAB);
		this.csvWriter.print(TAB);
		this.csvWriter.print(TAB);
		String prevQID = null;

		for (FormArea area : master.getFormAreaList()) {
			if (area.isTextArea()) {
				this.csvWriter.print(TAB);
				continue;
			}
			switch (area.getTypeCode()) {
			case FormArea.SELECT:
				this.csvWriter.print(TAB);
				this.csvWriter.print(StringUtil.escapeTSV(area.getItemValue()));
				break;
			case FormArea.SELECT1:
				if (area.getQID().equals(prevQID)) {
					this.csvWriter.print(ITEM_SEPARATOR);
				} else {
					this.csvWriter.print(TAB);
				}
				prevQID = area.getQID();
				this.csvWriter.print(StringUtil.escapeTSV(area.getItemValue()));
				break;
			}
		}
		this.csvWriter.println();
	}

	private void writeHeaderItemLabelRow(FormMaster master) {
		this.csvWriter.print(TAB);
		this.csvWriter.print(TAB);
		this.csvWriter.print(TAB);
		String prevQID = null;

		for (FormArea area : master.getFormAreaList()) {
			if (area.isTextArea()) {
				this.csvWriter.print(TAB);
				continue;
			}
			switch (area.getTypeCode()) {
			case FormArea.SELECT:
				this.csvWriter.print(TAB);
				this.csvWriter.print(StringUtil.escapeTSV(area.getItemLabel()));
				break;
			case FormArea.SELECT1:
				if (area.getQID().equals(prevQID)) {
					this.csvWriter.print(ITEM_SEPARATOR);
				} else {
					this.csvWriter.print(TAB);
				}
				prevQID = area.getQID();
				this.csvWriter.print(StringUtil.escapeTSV(area.getItemLabel()));
				break;
			}
		}
		this.csvWriter.println();
	}

	private void writeHeaderHintsRow(FormMaster master) {
		this.csvWriter.print(TAB);
		this.csvWriter.print(TAB);
		this.csvWriter.print(TAB);
		for (FormArea area : master.getFormAreaList()) {
			if (area.getTypeCode() == FormArea.SELECT1 && 0 < area.getItemIndex()) {
				continue;
			}
			this.csvWriter.print(TAB);
			if (!VERBOSE_PRINT_PAGE && area.isMarkArea() && 0 < area.getItemIndex()) {
				continue;
			}
			this.csvWriter.print(StringUtil.escapeTSV(StringUtil.join(area.getHints(), "")));
		}
		this.csvWriter.println();
	}

	private void writeHeaderTypeRow(FormMaster master) {
		this.csvWriter.print(TAB);
		this.csvWriter.print(TAB);
		this.csvWriter.print(TAB);
		for (FormArea area : master.getFormAreaList()) {
			if (area.getTypeCode() == FormArea.SELECT1 && 0 < area.getItemIndex()) {
				continue;
			}
			this.csvWriter.print(TAB);
			if (!VERBOSE_PRINT_QID && area.isMarkArea() && 0 < area.getItemIndex()) {
				continue;
			}
			this.csvWriter.print(area.getType());
		}
		this.csvWriter.println();
	}

	private void writeHeaderQIDRow(FormMaster master) {
		this.csvWriter.print(TAB);
		this.csvWriter.print(TAB);
		this.csvWriter.print(TAB);
		for (FormArea area : master.getFormAreaList()) {
			if (area.getTypeCode() == FormArea.SELECT1 && 0 < area.getItemIndex()) {
				continue;
			}
			this.csvWriter.print(TAB);
			if (!VERBOSE_PRINT_QID && area.isMarkArea() && 0 < area.getItemIndex()) {
				continue;
			}
			this.csvWriter.print(area.getQID());
		}
		this.csvWriter.println();
	}

	private void writeHeaderPageRow(FormMaster master) {
		this.csvWriter.print(TAB);
		this.csvWriter.print(TAB);
		this.csvWriter.print(TAB);
		for (FormArea area : master.getFormAreaList()) {
			if (area.getTypeCode() == FormArea.SELECT1 && 0 < area.getItemIndex()) {
				continue;
			}

			this.csvWriter.print(TAB);

			if (!VERBOSE_PRINT_PAGE && area.isMarkArea() && 0 < area.getItemIndex()) {
				continue;
			}
			this.csvWriter.print(area.getPage());
		}
		this.csvWriter.println();
	}

	private PrintWriter createCSVWriter(ResultPathFactory resultPathFactory, 
			PageMaster master,
			SourceDirectory sourceDirectory) throws IOException {
		String masterName = FileUtil.getBasename(new File(master.getPath()).getName());
		File csvFile = new File(resultPathFactory.getResultMasterDirectory( master.getPath(), sourceDirectory.getPath())
				+ File.separator + masterName + "-csv.txt");
		csvFile.getParentFile().mkdirs();
		return FileUtil.createPrintWriter(csvFile, "MS932");
	}

	@Override
	public void startRow(RowEvent ev) throws IOException {
		
		super.startRow(ev);
		
		this.csvWriter.print(ev.getRowIndex());
		this.csvWriter.print(TAB);

		this.csvWriter.print(ev.getSourceParentDirectory().getPath());	    
		this.csvWriter.print(TAB);

		int numPages = ev.getFormMaster().getNumPages();
		List<PageID> pageIDList = ev.getPageIDList();
		for (int pageIndex = 0; pageIndex < numPages; pageIndex++) {
			PageID pageID = pageIDList.get(pageIndex + ev.getRowIndex() * numPages);
			if (0 < pageIndex) {
				this.csvWriter.print(ITEM_SEPARATOR);
			}
			this.csvWriter.print(StringUtil.escapeTSV(new File(pageID.getPath()).getName()));
		}
		this.csvWriter.print(TAB);

		writeErrorCell(ev);
	}

	@Override
	public void endRow(RowEvent ev) throws IOException {
		this.csvWriter.println();
		super.endRow(ev);
	}

	private void writeErrorCell(RowEvent ev) {
		// ****** //
		// TODO: write error info 
		// ****** //
	}

	@Override
	public void startQuestion(QuestionEvent ev) throws IOException {
		super.startQuestion(ev);
		this.csvWriter.print(TAB);
	}

	@Override
	public void endQuestionItem(QuestionItemEvent ev) throws IOException {
		FormArea formArea = ev.getFormArea();
		MarkAreaAnswerItem markAreaAnswerItem = ev.getMarkAreaAnswerItem();
		if(formArea.isSelect()){
			endQuestionItemAsSelect(formArea, markAreaAnswerItem);
		}else if(formArea.isSelect1()){
			endQuestionItemAsSelect1(formArea, markAreaAnswerItem);
		}else if(formArea.isTextArea()){
			/* TEXT AREA */
			//this.csvWriter.print("(text)"); 
		}else{
			//this.csvWriter.print('(UNKNOWN_FORMTYPE)');
		}
		super.endQuestionItem(ev);
	}

	private void endQuestionItemAsSelect1(FormArea formArea,
			MarkAreaAnswerItem markAreaAnswerItem) {
		if (markAreaAnswerItem.isManualMode() && markAreaAnswerItem.isManualSelected()) {
			/* manually selected */
			this.csvWriter.print(StringUtil.escapeTSV(formArea.getItemValue()));
		}else if (markAreaAnswerItem.isAutoSelected()) {
			/* the most likely selected answer item */
			this.csvWriter.print(StringUtil.escapeTSV(formArea.getItemValue()));
		}else if (markAreaAnswerItem.isMarked()) {
			/* miss-marked answer item*/
			/*
			this.csvWriter.print('(');
			this.csvWriter.print(StringUtil.escapeTSV(formArea.getItemValue()));
			this.csvWriter.print(')');
			 */
		}else{
			/* not marked answer item*/
		}
	}

	private void endQuestionItemAsSelect(FormArea formArea,
			MarkAreaAnswerItem markAreaAnswerItem) {
		if(0 < formArea.getItemIndex()){
			this.csvWriter.print(TAB);	
		}
		if (markAreaAnswerItem.isManualMode()) {
			if (markAreaAnswerItem.isManualSelected()) {
				this.csvWriter.print('1');
			}else{
				this.csvWriter.print('0');
			}
		}else{
			if (markAreaAnswerItem.isAutoSelected()) {
				this.csvWriter.print('1');
			}else{
				this.csvWriter.print('0');
			}
		}
	}
	
	@Override
	public void endQuestion(QuestionEvent ev) throws IOException {
		FormArea area = ev.getDefaultFormArea();
		switch (area.getTypeCode()) {
		case FormArea.INPUT:
		case FormArea.TEXTAREA:
		case FormArea.SELECT:
			break;
		case FormArea.SELECT1:
			MarkAreaAnswer markAreaAnswer = (MarkAreaAnswer)ev.getAnswer();
			if(markAreaAnswer.isNoAnswer()){
				/* NO ANSWER */	
				// this.csvWriter.print('?'); 
			}
			if(1 < markAreaAnswer.getNumMarkedAnswerItems()){
				/* MORE THAN ONE */
				// this.csvWriter.print('#');
			}
			break;
		}
		super.endQuestion(ev);
	}

}
