/**
 *  HTMLErrorLogEventHandler.java

 Copyright 2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Created on 2007/03/03
 Author hiroya
 */
package net.sqs2.omr.result.module.core;

import java.awt.Point;
import java.awt.geom.Point2D;
import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Map;

import net.sqs2.exigrid.source.PageID;
import net.sqs2.exigrid.source.PageTask;
import net.sqs2.exigrid.source.PageTaskError;
import net.sqs2.exigrid.source.PageTaskExceptionCore;
import net.sqs2.image.ImageTranslator;
import net.sqs2.omr.execute.page.PageFrameExceptionCore;
import net.sqs2.omr.execute.page.PageSequenceInvalidExceptionCore;
import net.sqs2.omr.execute.page.PageUpsideDownExceptionCore;
import net.sqs2.omr.result.ImageFileUtil;
import net.sqs2.omr.result.MarkAreaAnswerItem;
import net.sqs2.omr.result.ResultPathFactory;
import net.sqs2.omr.result.event.ProcessErrorEvent;
import net.sqs2.omr.result.event.TableEvent;
import net.sqs2.omr.result.event.TableEventContext;
import net.sqs2.util.DateFormatUtil;
import net.sqs2.util.HTMLUtil;
import net.sqs2.util.PathUtil;

public class HTMLErrorLogEventHandler extends AbstractErrorLogHandler {

	private static final String ENCODING = "UTF-8";
	
	public HTMLErrorLogEventHandler(ResultPathFactory resultPathFactory) throws IOException {
		super(resultPathFactory, "xhtml", ENCODING);
		HTMLUtil.printHTMLHead(this.writer, ENCODING, resultPathFactory.getSourceDirectoryRoot().getName()+"-RESULT: ErrorLog");
		this.writer.println("<body>");
		this.writer.println("<ul>");
	}

	private String getErrorLogPath() {
		return this.errorLogFile.getAbsolutePath();
	}
	
	@Override
	public void startError(ProcessErrorEvent ev) {
		logPageTaskError(ev.getPageTask());
	}
	
	@Override
	public void log(PageTask pageTask, PageTaskError pageTaskError, String errorSourceURI) {
		this.writer.print("<li>");
		// this.writer.print(imagePathBuilder.toString());
		// this.writer.print("\t");
		logDateTime();
		this.writer.print("<br/>");
		logPagePath(pageTask, errorSourceURI);
		this.writer.print("<br/>");
		
		String errorImageURI = externalizeErrorImage(pageTask, pageTaskError);
		
		PageTaskExceptionCore core = pageTaskError.getExceptionCore();
		this.writer.print("<a href=\"");
		this.writer.print(HTMLUtil.encodePath(errorSourceURI));
		this.writer.print("\">");
		this.writer.print(core.getLocalizedMessage());
		this.writer.println("<br/>");

		if(errorImageURI != null){
			writeSVG(errorImageURI, core, pageTaskError.getScale());
		}
				
		this.writer.print("</a>");
		this.writer.println("</li>");
		this.writer.flush();
	}

	private String externalizeErrorImage(PageTask pageTask,
            PageTaskError pageTaskError) {
	    byte[] imageByteArray = pageTaskError.getSourceImageByteArray();
		String errorImageURI = null;
		
		if (imageByteArray != null) {
			File errorImageFile = this.resultPathFactory.getErrorFile(pageTask.getMasterPath(),
					pageTask.getPageID(), pageTaskError.getSourceImageType());
			errorImageFile.getParentFile().mkdirs();
			try{
				errorImageURI = PathUtil.getRelativePath(errorImageFile, this.errorLogFile);
				ImageFileUtil.writeImage(errorImageFile, imageByteArray);
			}catch(IOException ignore){}
		}else{
			File errorImageFile = new File(this.resultPathFactory.getSourceDirectoryRoot(), pageTask.getPageID().getPath());
			try{
				errorImageURI = PathUtil.getRelativePath(errorImageFile, this.errorLogFile);
			}catch(IOException ignore){}
		}
	    return errorImageURI;
    }

	private void logPagePath(PageTask pageTask, String uri) {
	    this.writer.print("<a href=\"");
		this.writer.print(HTMLUtil.encodePath(uri));
		this.writer.print("\">");
		this.writer.print(pageTask.getPageID().getPath());
		
		if (1 < pageTask.getPageID().getNumPagesInFile()) {
			this.writer.print('(');
			this.writer.print(pageTask.getPageID().getIndex());
			this.writer.print(')');
		}
		this.writer.print("</a>");
    }

	private void logDateTime() {
	    this.writer.print(DateFormatUtil.format());
    }

	private void writeSVG(String errorImageURI, PageTaskExceptionCore core, double scale) {
		this.writer.print("<svg:svg width=\"");
		this.writer.print((int) (core.getWidth() * scale));
		this.writer.print("\" height=\"");
		this.writer.print((int) (core.getHeight() * scale));
		this.writer.println("\">");

		this.writer.println("<svg:g stroke=\"magenta\" stroke-width=\"5\">");
		this.writer.print("<svg:image xlink:href=\"");
		this.writer.print(HTMLUtil.encodePath(errorImageURI));
		this.writer.print("\" x=\"0\" y=\"0\" width=\"");
		this.writer.print((int) (core.getWidth() * scale));
		this.writer.print("\" height=\"");
		this.writer.print((int) (core.getHeight() * scale));
		this.writer.println("\"/>");

		this.writer.println("<svg:rect fill=\"none\"");
		this.writer.print("x=\"0\" y=\"0\" width=\"");
		this.writer.print((int) (core.getWidth() * scale));
		this.writer.print("\" height=\"");
		this.writer.print((int) (core.getHeight() * scale));
		this.writer.println("\"/>");

		this.writer.println("</svg:g>");
		if (core instanceof PageFrameExceptionCore) {
			PageFrameExceptionCore c = (PageFrameExceptionCore) core;
			this.writer.println("<svg:g stroke=\"red\" stroke-width=\"1\">");
			writeSVGBox(c.getCorners(), scale);
			Point[] masterCorners = c.getMasterCorners();
			Point[] corners = c.getCorners();
			ImageTranslator imageTranslator = new ImageTranslator(masterCorners, corners);
			if(core instanceof PageSequenceInvalidExceptionCore){
				PageSequenceInvalidExceptionCore pageOrderExceptionCore = (PageSequenceInvalidExceptionCore)core;
				//int leftValue = pageOrderExceptionCore.getLeftValue();
				//int rightValue = pageOrderExceptionCore.getRightValue();
				writeSVGBox(imageTranslator.translate(pageOrderExceptionCore.getLeftFooterArea()), scale);
				writeSVGBox(imageTranslator.translate(pageOrderExceptionCore.getRightFooterArea()), scale);
			}else if(core instanceof PageUpsideDownExceptionCore){
				PageUpsideDownExceptionCore pageUpsideDownExceptionCore = (PageUpsideDownExceptionCore)core;
				writeSVGBox(imageTranslator.translate(pageUpsideDownExceptionCore.getHeaderCheckArea()), scale);
				writeSVGBox(imageTranslator.translate(pageUpsideDownExceptionCore.getFooterCheckArea()), scale);
			}
			this.writer.println("</svg:g>");
		}

		this.writer.println("</svg:svg>");
	}

	private void writeSVGBox(Point2D[] corners, double scale) {
		writeSVGLine(corners[0], corners[1], scale);
		writeSVGLine(corners[0], corners[2], scale);
		writeSVGLine(corners[2], corners[3], scale);
		writeSVGLine(corners[1], corners[3], scale);
	}

	private void writeSVGLine(Point2D a, Point2D b, double scale) {
		this.writer.print("<svg:line  opacity=\"0.5\"");
		this.writer.print(" x1=\"");
		this.writer.print((int) (a.getX() * scale));
		this.writer.print("\" y1=\"");
		this.writer.print((int) (a.getY() * scale));
		this.writer.print("\" x2=\"");
		this.writer.print((int) (b.getX() * scale));
		this.writer.print("\" y2=\"");
		this.writer.print((int) (b.getY() * scale));
		// this.writer.println("\" marker-end=\"url(#Triangle)\"");
		this.writer.println("\"/>");
	}

	@Override
	public void close() {
		this.writer.println("</ul>");
		this.writer.println("</body>");
		this.writer.println("</html>");
		super.close();
	}

	public void writeMultiMarkWarnings(TableEvent ev) throws IOException {
		//String errorLogFilePath = errorLogEventHandler.getAbsolutePath();
		this.writer.println("<ul>");
		for (Map.Entry<PageID, TableEventContext.ErrorByPage> entry : ev.getTableEventContext().getErrorMapByPage().entrySet()) {
			PageID pageID = entry.getKey();
			String errorSrcImgPath = ev.getSourceDirectory().getDirectoryRoot().getAbsolutePath()
			+ File.separatorChar + pageID.getPath();
			String srcImgURL = PathUtil.getRelativePath(errorSrcImgPath, getErrorLogPath());

			this.writer.print("<li>");
			// writer.print("row:"+rowIndex+",");
			this.writer.print("<a href=\"");
			this.writer.print(HTMLUtil.encodePath(srcImgURL));
			this.writer.print("\">");
			this.writer.print("page:" + HTMLUtil.escapeHTML(pageID.getPath()));
			this.writer.println("</a>");
			this.writer.println("<br/>");

			List<MarkAreaAnswerItem> errorAnswerItemList = entry.getValue().getErrorAnswerItemList();
			for (MarkAreaAnswerItem errorAnswerItem : errorAnswerItemList) {
				this.writer.println(errorAnswerItem.getItemIndex());
				String imgURL = errorAnswerItem.getAreaImageURIPath();
				String url = HTMLUtil.encodePath(PathUtil.getRelativePath(imgURL, getErrorLogPath()));
				HTMLUtil.printInlineImage(this.writer, url, url);
			}
			this.writer.println("</li>");
		}
		this.writer.println("</ul>");
	}
}
