/**
 *  AbstractResultManagerImpl.java

 Copyright 2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Created on 2007/01/31
 Author hiroya
 */
package net.sqs2.omr.result.manager;

import java.io.File;
import java.io.IOException;

import net.sqs2.exigrid.event.ResultEventManager;
import net.sqs2.exigrid.event.ResultEventMonitor;
import net.sqs2.exigrid.source.PageTask;
import net.sqs2.exigrid.source.PageTaskError;
import net.sqs2.exigrid.source.SessionSource;
import net.sqs2.omr.result.ResultPathFactory;
import net.sqs2.omr.result.event.MasterEvent;
import net.sqs2.omr.result.event.ProcessErrorEvent;
import net.sqs2.omr.result.event.QuestionEvent;
import net.sqs2.omr.result.event.QuestionItemEvent;
import net.sqs2.omr.result.event.ResultEventImpl;
import net.sqs2.omr.result.event.ResultEventFilter;
import net.sqs2.omr.result.event.ResultEventHandler;
import net.sqs2.omr.result.event.ResultEventSource;
import net.sqs2.omr.result.event.RowEvent;
import net.sqs2.omr.result.event.TableEvent;
import net.sqs2.omr.result.module.core.AreaImageGeneratorEventHandler;
import net.sqs2.omr.result.module.core.ContextGeneratorEventHandler;
import net.sqs2.omr.result.module.core.HTMLErrorLogEventHandler;
import net.sqs2.omr.result.module.core.PlainTextErrorLogEventHandler;
import net.sqs2.omr.result.module.csv.CSVGeneratorEventHandler;
import net.sqs2.omr.result.module.dump.ResultEventDumpEventHandler;
//import net.sqs2.omr.result.module.spss.SPSSGeneratorEventHandler;
//import net.sqs2.omr.result.module.dump.ResultEventDumpEventHandler;
import net.sqs2.omr.result.module.html.ResultHTMLGeneratorEventHandler;
//import net.sqs2.omr.result.module.jfreechart.ChartGeneratorEventHandler;
//import net.sqs2.omr.result.module.webui.WebUIBeanGeneratorEventHandler;
import net.sqs2.omr.result.module.xls.XLSGeneratorEventHandler;
import net.sqs2.omr.source.config.FormResultConfig;

public class ResultEventManagerImpl implements ResultEventManager {

	protected ResultEventSource resultEventSource;
	protected HTMLErrorLogEventHandler htmlErrorLogEventHandler;
	protected PlainTextErrorLogEventHandler plainTextErrorLogEventHandler;
	protected ResultEventMonitor monitor;

	protected long lastResultProducedTime = -1L;
	protected long lastPageTaskProducedTime = 0L;

	protected final static int threshold = 230;
	protected final static int doubleMarkDiffThreshold = 30;
	protected final static int densityStatResolution = 20;
	protected final static FormResultConfig formResultConfig = new FormResultConfig(threshold, doubleMarkDiffThreshold);
	private File DUMP_FILE = new File(System.getProperty("java.io.tmpdir"), "ResultEventDumpEventHandler.dump");
	
	public ResultEventManagerImpl(SessionSource sessionSource, File sourceDirectoryRoot, File resultDirectoryRoot) throws IOException {
		this(sessionSource, sourceDirectoryRoot, resultDirectoryRoot, null);
	}
	
	public ResultEventManagerImpl(SessionSource sessionSource, File sourceDirectoryRoot, File resultDirectoryRoot,
			ResultEventFilter filter) throws IOException {
		ResultPathFactory resultPathFactory = new ResultPathFactory(sourceDirectoryRoot, resultDirectoryRoot, formResultConfig,
				ResultPathFactory.MASTERNAME_SOURCEDIR_MODE);
		this.htmlErrorLogEventHandler = new HTMLErrorLogEventHandler(resultPathFactory);
		this.plainTextErrorLogEventHandler = new PlainTextErrorLogEventHandler(resultPathFactory);
		this.resultEventSource = new ResultEventSource(sessionSource, resultPathFactory, filter, new ContextGeneratorEventHandler(threshold));
		for (ResultEventHandler handler : createResultEventHandlers()) {
			 this.resultEventSource.addResultEventHandler(handler);
		}
	}

	public void notifyPageTaskReceived(PageTask pageTask) {
		PageTaskError pageTaskError = pageTask.getPageTaskError();
		if (pageTaskError != null) {
			this.plainTextErrorLogEventHandler.logPageTaskError(pageTask);
			this.htmlErrorLogEventHandler.logPageTaskError(pageTask);
		}
	}

	/**
	 * set monitor(Observer)
	 * @Override
	 * @param monitor observer of Externalization process
	 */
	 public void setResultEventMonitor(ResultEventMonitor monitor) {
		this.monitor = monitor;
	 }

	 public void execute() {
		 this.resultEventSource.start();
		 close();
		 this.lastResultProducedTime = this.lastPageTaskProducedTime;
		 /*
               if(! isSourceDirectoryObservingMode()){
                   ExigridEngine engine = ExigridEngine.getCurrentEngine();
                   engine.stop(this.resultPathFactory.getsourceDirectoryRoot(),
                               this.resultPathFactory.getResultDirectoryRoot());
               }
		  */
	 }

	 public boolean isSourceDirectoryObservingMode() {
		 return false;
	 }

	 protected ResultEventHandler[] createResultEventHandlers() throws IOException{
	        return new ResultEventHandler[]{
	        		new AreaImageGeneratorEventHandler(),
	                new ResultHTMLGeneratorEventHandler(threshold, densityStatResolution, this.htmlErrorLogEventHandler),
	                new CSVGeneratorEventHandler(),
	                new XLSGeneratorEventHandler(),
	                //new ChartGeneratorEventHandler(),
	                //new SPSSGeneratorEventHandler(),
	                //new ResultEventDumpEventHandler(DUMP_FILE, "UTF-8"),
	                new MonitorEventHandler()
	        };
	 }
	 
	 public void stop() {
		 close();
		 if (this.resultEventSource != null) {
			 this.resultEventSource.stop();
		 }
	 }
	 
	 public void shutdown() {
		 stop();
	 }

	 public void close() {
		 if (this.plainTextErrorLogEventHandler != null) {
			 this.plainTextErrorLogEventHandler.close();
			 this.plainTextErrorLogEventHandler = null;
		 }
		 if (this.htmlErrorLogEventHandler != null) {
			 this.htmlErrorLogEventHandler.close();
			 this.htmlErrorLogEventHandler = null;
		 }
	 }
	 
	 class MonitorEventHandler extends ResultEventHandler {
		 @Override
		 public void startResult(ResultEventImpl ev) throws IOException {
			 monitor.notifyResultEvent(ev);
		 }

		 @Override
		 public void startMaster(MasterEvent ev) throws IOException {
			 monitor.notifyResultEvent(ev);
		 }

		 @Override
		 public void startTable(TableEvent ev) throws IOException {
			 monitor.notifyResultEvent(ev);
		 }

		 @Override
		 public void endTable(TableEvent ev) throws IOException {
			 monitor.notifyResultEvent(ev);
		 }

		 @Override
		 public void startRow(RowEvent ev) throws IOException {
			 monitor.notifyResultEvent(ev);
		 }

		 @Override
		 public void startQuestion(QuestionEvent ev) throws IOException {
			 //monitor.notifyResultUpdated(ev);
		 }

		 @Override
		 public void startQuestionItem(QuestionItemEvent ev) throws IOException {
			 //monitor.notifyResultUpdated(ev);
		 }

		 @Override
		 public void startError(ProcessErrorEvent ev) throws IOException {
			 monitor.notifyResultEvent(ev);
		 }
	 }

	 public void updateLastPageTaskProducedTime() {
		 this.lastPageTaskProducedTime = System.currentTimeMillis();
	 }

	 public boolean isResultProducedTimeEqualsLastPageProcessedTime() {
		 return this.lastResultProducedTime == this.lastPageTaskProducedTime;
	 }
}