package net.sqs2.omr.result.event;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;

import net.sqs2.exigrid.master.PageMaster;
import net.sqs2.exigrid.source.PageID;
import net.sqs2.exigrid.source.PageTask;
import net.sqs2.exigrid.source.PageTaskError;
import net.sqs2.exigrid.source.PageTaskResult;
import net.sqs2.exigrid.source.SessionSource;
import net.sqs2.exigrid.source.SourceDirectory;
import net.sqs2.omr.result.ResultPathFactory;
import net.sqs2.omr.result.module.core.ContextGeneratorEventHandler;
import net.sqs2.omr.source.FormAreaCommand;
import net.sqs2.util.PathUtil;

public class ResultEventSource extends AbstractResultEventSource{
	
	private ResultEventFilter resultEventFilter = null;
	private ContextGeneratorEventHandler contextGeneratorEventHandler = null;

	private List<ResultEventHandler> resultEventHandlers = new ArrayList<ResultEventHandler>();

	ResultEventSource(SessionSource sessionSource, ResultPathFactory resultPathFactory){
		super(sessionSource, resultPathFactory);
	}
	
	public ResultEventSource(SessionSource sessionSource, ResultPathFactory resultPathFactory,  ResultEventFilter resultEventFilter,
			ContextGeneratorEventHandler contextGeneratorEventHandler){
		super(sessionSource, resultPathFactory);
		this.resultEventFilter = resultEventFilter;
		this.contextGeneratorEventHandler = contextGeneratorEventHandler;
	}
	
	public void addPageTaskExecutionEventHandler(ResultEventHandler listener) {
		this.resultEventHandlers.add(listener);
	}

	public void removePageTaskExecutionEventHandler(ResultEventHandler  listener) {
		this.resultEventHandlers.remove(listener);
	}

	public void addResultEventHandler(ResultEventHandler listener) {
		this.resultEventHandlers.add(listener);
	}

	public void removeResultEventHandler(ResultEventHandler listener) {
		this.resultEventHandlers.add(listener);
	}
	
	@Override
	protected boolean generateEvents() {
		try {
		    ResultEventContext context = createResultEventContext();
		    Set<PageMaster> masterSet = context.getPageMasterSet();
		    if(context == null){
		    	return false;
		    }
		    ResultEventImpl resultEvent = new ResultEventImpl(context);
		    startResult(resultEvent);
		    ResultEventSourceGenerator generator = new ResultEventSourceGeneratorImpl(this.resultEventFilter) ;
		    int masterIndex = 0; 
		    for (PageMaster master : masterSet) {
		    	if (!isRunning()) {
		    		break;
		    	}
		    	generator.generateMasterEvent(context, master, masterIndex);
		    	masterIndex++;
		    }
		    endResult(resultEvent);
		    return true;
		} catch (IOException ex) {
			ex.printStackTrace();
			Logger.getLogger("resultEvent").severe("IOException");
		}
		return false;
	}

	protected void startResult(ResultEventImpl ev){
		for (ResultEventHandler handler : this.resultEventHandlers) {
			try{
				handler.startResult(ev);
			}catch(IOException ex){
				ex.printStackTrace();
			}
		}
	}
	
	protected void endResult(ResultEventImpl ev) {
		for (ResultEventHandler handler : this.resultEventHandlers) {
			try{
				handler.endResult(ev);
			}catch(IOException ex){
				ex.printStackTrace();
			}
		}
	}
	
	protected void startMaster(MasterEvent ev) {
		for (ResultEventHandler handler : this.resultEventHandlers) {
			try{
				handler.startMaster(ev);
			}catch(IOException ex){
				ex.printStackTrace();
			}
		}
	}
	
	protected void endMaster(MasterEvent ev){
		for (ResultEventHandler handler : this.resultEventHandlers) {
			try{
				handler.endMaster(ev);
			}catch(IOException ex){
				ex.printStackTrace();
			}
		}
	}
	
	protected void startTable(TableEvent ev){
		for (ResultEventHandler handler : this.resultEventHandlers) {
			try{
				if( isLeaf(ev, handler)){
					handler.startTable(ev);
				}
			}catch(IOException ex){
				ex.printStackTrace();
			}
		}
	}
	
	protected void endTable(TableEvent ev) {
		for (ResultEventHandler handler : this.resultEventHandlers) {
			try{
				if( isLeaf(ev, handler)){
					handler.endTable(ev);
				}
			}catch(IOException ex){
				ex.printStackTrace();
			}
		}
	}
	
	protected void startRow(RowEvent ev) {
		for (ResultEventHandler handler : this.resultEventHandlers) {
			try{
				if( isLeaf(ev, handler)){
					handler.startRow(ev);
				}
			}catch(IOException ex){
				ex.printStackTrace();
			}
		}
	}
	
	protected void endRow(RowEvent ev) {
		for (ResultEventHandler handler : this.resultEventHandlers) {
			try{
				if( isLeaf(ev, handler)){
					handler.endRow(ev);
				}
			}catch(IOException ex){
				ex.printStackTrace();
			}
		}
	}
	
	protected void startQuestion(QuestionEvent ev) {
		for (ResultEventHandler handler : this.resultEventHandlers) {
			try{
				if(isLeaf(ev, handler) ){
					handler.startQuestion(ev);
				}
			}catch(IOException ex){
				ex.printStackTrace();
			}
		}
	}
	
	protected void endQuestion(QuestionEvent ev) {
		for (ResultEventHandler handler : this.resultEventHandlers) {
			try{
				if(isLeaf(ev, handler) ){
					handler.endQuestion(ev);
				}
			}catch(IOException ex){
				ex.printStackTrace();
			}
		}
	}
	
	protected void startQuestionItem(QuestionItemEvent ev) {
		for (ResultEventHandler handler : this.resultEventHandlers) {
			try{
				if(isLeaf(ev, handler) ){
					handler.startQuestionItem(ev);
				}
			}catch(IOException ex){
				ex.printStackTrace();
			}
		}
	}
	
	protected void endQuestionItem(QuestionItemEvent ev) {
		for (ResultEventHandler handler : this.resultEventHandlers) {
			try{
				if(isLeaf(ev, handler) ){
					handler.endQuestionItem(ev);
				}
			}catch(IOException ex){
				ex.printStackTrace();
			}
		}
	}
	
	protected void startError(ProcessErrorEvent ev) {
		for (ResultEventHandler handler : this.resultEventHandlers) {
			try{
				handler.startError(ev);
			}catch(IOException ex){
				ex.printStackTrace();
			}
		}
	}

	protected void endError(ProcessErrorEvent ev) {
		for (ResultEventHandler handler : this.resultEventHandlers) {
			try{
				handler.endError(ev);
			}catch(IOException ex){
				ex.printStackTrace();
			}
		}
	}

	protected boolean isLeaf(TableEvent ev,  ResultEventHandler handler) {
		return ev.isLeaf();
	}
	
	public abstract class ResultEventSourceGenerator{
		abstract protected void generateMasterEvent(ResultEventContext context, PageMaster master, int masterIndex)throws IOException;
	}
	
	public class ResultEventSourceGeneratorImpl extends ResultEventSourceGenerator{
				
		private QuestionItemEventContext prevQuestionItemEventContext = null;
		protected static final int DENSITY_STAT_RESOLUTION = 20;

		ResultEventFilter filter = null;
		
		ResultEventSourceGeneratorImpl(){
		}
		
		ResultEventSourceGeneratorImpl(ResultEventFilter filter){
			this.filter = filter;
		}
		
		@Override
		protected void generateMasterEvent(ResultEventContext context, PageMaster master, int masterIndex)throws IOException{
	    	MasterEventContext masterEventContext = new MasterEventContext(context, master, masterIndex, getNumTables(master));   
			if(this.filter != null && ! this.filter.accept(masterEventContext)){
				return;
			}
			MasterEvent masterEvent = new MasterEvent(masterEventContext);
			startMaster(masterEvent);
			
			processLeafTables(masterEventContext);
			processSharedBranchTables(masterEventContext);
			
	    	endMaster(masterEvent);
		}
		
		private void processLeafTables(MasterEventContext masterEventContext) throws IOException {
			List<SourceDirectory> sourceDirectoryList = masterEventContext.getSourceDirectoryList();
			int tableIndex = 0; 
			for (SourceDirectory sourceDirectory : sourceDirectoryList) {
				if (!isRunning()) {
					break;
				}
				generateTableEvent(masterEventContext, sourceDirectory, tableIndex, true);
				tableIndex++;
			}
		}
		
		private Set<File> createSourceDirectoryFileSet(List<SourceDirectory> sourceDirectoryList) throws IOException {
			Set<File> directories = new LinkedHashSet<File>();
			for (SourceDirectory sourceDirectory : sourceDirectoryList) {
				directories.add(sourceDirectory.getDirectory());
			}		
			return directories;
		}

		private void processSharedBranchTables(MasterEventContext masterEventContext) throws IOException {
			File[] sharedSuperDirectoryArray = createSharedBranchDirectorySet(masterEventContext);
			int tableIndex = 0;
			for (File superDirectory : sharedSuperDirectoryArray) {
				if (!isRunning()) {
					break;
				}
				if (! isSourceDirectoryMember(superDirectory)) {
					break;
				}
				SourceDirectory sourceDirectory = new SourceDirectory(masterEventContext.getSourceDirectoryRoot(), superDirectory); 
				generateTableEvent(masterEventContext, sourceDirectory, tableIndex, false);
				tableIndex++;	
			}
		}

		private File[] createSharedBranchDirectorySet(MasterEventContext masterEventContext) throws IOException {
			List<SourceDirectory> sourceDirectoryList = masterEventContext.getSourceDirectoryList();
			Set<File> sourceDirectoryFileSet = createSourceDirectoryFileSet(sourceDirectoryList);
			
			File[] dummyFiles = new File[0];
			File[] sharedSuperDirectoryArray = PathUtil.getSharedSuperDirectorySet(sourceDirectoryFileSet).toArray(dummyFiles);
			Arrays.sort(sharedSuperDirectoryArray);
			return sharedSuperDirectoryArray;
		}

		private void generateTableEvent(MasterEventContext masterEventContext, SourceDirectory sourceDirectory, int tableIndex, boolean isLeaf) throws IOException{
			TableEventContext tableEventContext = new TableEventContext(masterEventContext, sourceDirectory, tableIndex, DENSITY_STAT_RESOLUTION,  isLeaf);
			TableEvent tableEvent = new TableEvent(tableEventContext);

			startTable(tableEvent);
			processTable(tableEventContext);
			endTable(tableEvent);
		}
		
		private int processTable(TableEventContext tableEventContext) throws IOException {
			if(this.filter != null && ! this.filter.accept(tableEventContext)){
				return 0;
			}
			int numRows = processRows(tableEventContext);
			return numRows;
		}

		private int processRows(TableEventContext tableEventContext) throws IOException {
			SourceDirectory sourceDirectory = tableEventContext.getSourceDirectory();
			int numPages = tableEventContext.getFormMaster().getNumPages();

			int numRowsTotal = 0;
			Map<File, List<PageID>> pageIDListMap = sourceDirectory.getPageIDListMap();
			if(pageIDListMap == null){
				return 0;
			}
			for (Map.Entry<File,List<PageID>> entry: pageIDListMap.entrySet()) {
				if (!isRunning()) {
					break;
				}
				File directory = entry.getKey();
				int numRows = entry.getValue().size() / numPages;
				for (int rowIndex = 0; rowIndex < numRows; rowIndex++) {
					int pageIDIndex = numPages * rowIndex;
					RowEventContext rowEventContext = new RowEventContext(tableEventContext, directory, pageIDIndex, rowIndex, numRows);
					if(this.filter != null && ! this.filter.accept(rowEventContext)){
						continue;
					}
					generateRowEvent(tableEventContext, rowEventContext);
				}
				numRowsTotal += numRows;
			}
			return numRowsTotal;
		}

		private void generateRowEvent(TableEventContext tableEventContext, RowEventContext rowEventContext) throws IOException {
	        RowEvent rowEvent = new RowEvent(rowEventContext);
	        startRow(rowEvent);
	        processColumns(tableEventContext, rowEventContext);
	        endRow(rowEvent);
        }

		private void processColumns(TableEventContext tableEventContext, RowEventContext rowEventContext) throws IOException {
	        int columnIndex = 0;
	        for (String qid : tableEventContext.getFormMaster().getQIDSet()) {
	        	if (!isRunning()) {
	        		break;
	        	}
	        	QuestionEventContext questionEventContext = new QuestionEventContext(rowEventContext, qid, columnIndex);
	        	processQuestion(questionEventContext);
	        	columnIndex++;
	        }
        }

		private void processQuestion(QuestionEventContext questionEventContext) throws IOException {
			if(this.filter != null && ! this.filter.accept(questionEventContext)){
				return;
			}
			generateQuestionEvent(questionEventContext);
		}

		private void generateQuestionEvent(QuestionEventContext questionEventContext) throws IOException {
	        QuestionEvent questionEvent = new QuestionEvent(questionEventContext);
			startQuestion(questionEvent);

			if(contextGeneratorEventHandler != null){
				for (int itemIndex = 0; itemIndex < questionEventContext.getNumQuestionItems(); itemIndex++) {
					if (!isRunning()) {
						break;
					}

					QuestionItemEventContext questionItemEventContext = new QuestionItemEventContext(questionEventContext, itemIndex, this.prevQuestionItemEventContext);
				
					this.prevQuestionItemEventContext = questionItemEventContext;

					PageTask pageTask = questionItemEventContext.getPageTask();
					if (pageTask == null) {
						break;
					}
					PageTaskResult pageTaskResult = pageTask.getPageTaskResult();
					if (pageTaskResult == null) {
						processError(questionItemEventContext, pageTask.getPageTaskError());
						break;
					}

					int areaIndexInPage = questionItemEventContext.getFormArea().getAreaIndexInPage();

					if(0 == pageTaskResult.getPageAreaCommandList().size()){
						break;
					}
					
					FormAreaCommand formAreaCommand = (FormAreaCommand)pageTaskResult.getPageAreaCommandList().get(areaIndexInPage);
					questionItemEventContext.setFormAreaCommand(formAreaCommand);
					contextGeneratorEventHandler.startQuestionItem(new QuestionItemEvent(questionItemEventContext));
				}
				contextGeneratorEventHandler.endQuestion(questionEvent);
			}
			
			for (int itemIndex = 0; itemIndex < questionEventContext.getNumQuestionItems(); itemIndex++) {
				if (!isRunning()) {
					break;
				}

				QuestionItemEventContext questionItemEventContext = new QuestionItemEventContext(questionEventContext, itemIndex, this.prevQuestionItemEventContext);
				
				this.prevQuestionItemEventContext = questionItemEventContext;

				PageTask pageTask = questionItemEventContext.getPageTask();
				if (pageTask == null) {
					break;
				}
				PageTaskResult pageTaskResult = pageTask.getPageTaskResult();
				if (pageTaskResult == null) {
					break;
				}
				
				if(0 == pageTaskResult.getPageAreaCommandList().size()){
					break;
				}

				int areaIndexInPage = questionItemEventContext.getFormArea().getAreaIndexInPage();
				FormAreaCommand formAreaCommand = (FormAreaCommand)pageTaskResult.getPageAreaCommandList().get(areaIndexInPage);
				questionItemEventContext.setFormAreaCommand(formAreaCommand);
				
				processQuestionItem(questionItemEventContext, formAreaCommand);
			}
			endQuestion(questionEvent);
        }

		private void processQuestionItem(QuestionItemEventContext questionItemEventContext, FormAreaCommand formAreaCommand) throws IOException {
			QuestionItemEvent questionItemEvent = new QuestionItemEvent(questionItemEventContext);
			startQuestionItem(questionItemEvent);
			endQuestionItem(questionItemEvent);
		}

		private void processError(QuestionItemEventContext questionItemEventContext, PageTaskError pageTaskError) throws IOException{
			ProcessErrorEventContext processErrorEventContext = new ProcessErrorEventContext(questionItemEventContext, pageTaskError);
			ProcessErrorEvent processErrorEvent = new ProcessErrorEvent(processErrorEventContext);
			startError(processErrorEvent);
			endError(processErrorEvent);
		}
	}
}
