/*

 ImageSourceImpl.java

 Copyright 2004-2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Created on 2004/07/16

 */
package net.sqs2.omr.page;

import java.awt.Point;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;
import java.awt.image.AffineTransformOp;
import java.awt.image.BufferedImage;
import java.awt.image.WritableRaster;

import net.sqs2.image.ImageTranslationFilter;
import net.sqs2.image.ImageUtil;

/**
 * @author hiroya
 * 
 */
public class PageSource extends ImageTranslationFilter{
	static final float OVER_SAMPLING_FACTOR = 2.0f;

	AffineTransformOp operator = null;

	private int blackThreshold;
	private float scale;

	public PageSource(BufferedImage image, Point[] master, Point[] guide, float scale, int blackThreshold) {
		super(image, master, guide);
		this.setScale(scale);
		this.blackThreshold = blackThreshold;
	}

	public void setScale(float scale){
		this.scale = scale;
		AffineTransform t = AffineTransform.getScaleInstance(1 / OVER_SAMPLING_FACTOR / this.scale, 1 / OVER_SAMPLING_FACTOR / this.scale);
		this.operator = new AffineTransformOp(t, new RenderingHints(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BILINEAR));
	}

	public BufferedImage cropImage(Rectangle rect) {
		return this.cropImage(rect.x, rect.y, rect.width, rect.height);
	}

	public BufferedImage cropImage(float x, float y, float w, float h) {
		BufferedImage image = new BufferedImage((int)(w * OVER_SAMPLING_FACTOR * this.scale),
				(int)(h * OVER_SAMPLING_FACTOR * this.scale),
				BufferedImage.TYPE_INT_RGB);
		WritableRaster raster = image.getRaster();
		Point2D.Float p = new Point2D.Float();
		int[] rgbarr = new int[3];
		for (int j = (int)(h * OVER_SAMPLING_FACTOR * this.scale) - 1; 0 <= j; j--) {
			for (int i = (int)(w * OVER_SAMPLING_FACTOR * this.scale) - 1; 0 <= i; i--) {
				ImageUtil.rgb2arr(getRGB(x + i / OVER_SAMPLING_FACTOR / this.scale, y + j / OVER_SAMPLING_FACTOR / this.scale, p), rgbarr);
				raster.setPixel(i, j, rgbarr);
			}
		}
		BufferedImage newImage = new BufferedImage((int)w, (int)h, BufferedImage.TYPE_INT_RGB);
		if(this.operator == null){
			throw new IllegalArgumentException("setScale is not called.");
		}
		this.operator.filter(image, newImage);
		image.flush();
		return newImage;
	}

	/**
	 * @return grayscale value (0-255)
	 */
	 public int getGrayscaleDensity(Rectangle rect) {
		 int total = 0;
		 Point2D p = new Point2D.Float();
		 //System.err.println(rect);
		 //System.err.println(getPoint(rect.x,  rect.y, p)+"-"+getPoint(rect.x + rect.height - 1, rect.y + rect.height - 1, p));

		 for (int j = (int)(rect.height * OVER_SAMPLING_FACTOR) -1 ; 0 <= j; j--) {
			 for (int i = (int)(rect.width * OVER_SAMPLING_FACTOR) - 1; 0 <= i; i--) {
				 total += ImageUtil.rgb2gray(getRGB(rect.x + i / OVER_SAMPLING_FACTOR, rect.y + j / OVER_SAMPLING_FACTOR, p));
			 }
		 }
		 return (int)(total / (rect.width * rect.height * OVER_SAMPLING_FACTOR * OVER_SAMPLING_FACTOR));
	 }

	 public int getGrayscaleDensityWithFilter(Rectangle rect, int size) {
		 int[][] src = createSource(rect);
		 return filter(src, rect.width, rect.height, size);
	 }

	 private int[][] createSource(Rectangle rect) {
		 Point2D p = new Point2D.Float();
		 int[][] src = new int[(int)(rect.height * OVER_SAMPLING_FACTOR)][(int)(rect.width * OVER_SAMPLING_FACTOR)];
		 for (int y = (int)(rect.height * OVER_SAMPLING_FACTOR) - 1 ; 0 <= y; y--) {
			 for (int x = (int)(rect.width * OVER_SAMPLING_FACTOR) - 1; 0 <= x; x--) {
				 int gray = ImageUtil.rgb2gray(getRGB(rect.x + x / OVER_SAMPLING_FACTOR, rect.y + y / OVER_SAMPLING_FACTOR, p));
				 src[y][x] = gray;
			 }
		 }
		 return src;
	 }

	 private int filter(int[][] src, int w, int h, int size) {
		 long total = 0;
		 int edge = size/2;
		 int filterThreshold = size*size / 2 + 1;
		 for (int y = edge; y < src.length - edge; y++) {
			 for (int x = edge; x < src[y].length - edge; x++) {
				 int numBlackPixels = countBlackPixels(src, this.blackThreshold, x, y, size);
				 if(src[y][x] < this.blackThreshold){//center is black
					 if(numBlackPixels <= filterThreshold){//black is less than 2
						 total += 255; //set white
					 }else{
						 //total += 0; // set black
					 }
				 }else{//center is white
					 if(filterThreshold <= numBlackPixels){ //black is more than 4
						 //total += 0; //set black
					 }else{
						 total += 255; //set white
					 }
				 }
			 }
		 }
		 return (int)(total / ((src[0].length - edge*2) * (src.length - edge*2)));
	 }

	 private int countBlackPixels(int[][] src, int threshold, int x, int y, int size) {
		 int numBlackPixels = 0;
		 for(int yy = y - (size / 2); yy <= y + (size / 2); yy++){
			 for(int xx = x - (size / 2); xx <= x + (size / 2); xx++){
				 if(src[yy][xx] < threshold){
					 numBlackPixels++;
				 }		
			 }
		 }
		 if(src[y][x] < threshold){
			 numBlackPixels--;
		 }
		 return numBlackPixels;
	 }

	 /*
	private int countBlackPixels(int[][] src, int threshold, int x, int y) {
		int numBlackPixels = 0;
		if(src[y-1][x-1] < threshold){
			numBlackPixels++;
		}
		if(src[y-1][x] < threshold){
			numBlackPixels++;
		}
		if(src[y-1][x+1] < threshold){
			numBlackPixels++;
		}
		if(src[y+1][x-1] < threshold){
			numBlackPixels++;
		}
		if(src[y+1][x] < threshold){
			numBlackPixels++;
		}
		if(src[y+1][x+1] < threshold){
			numBlackPixels++;
		}
		if(src[y][x-1] < threshold){
			numBlackPixels++;
		}
		if(src[y][x+1] < threshold){
			numBlackPixels++;
		}
		return numBlackPixels;
	}
	  */

}
