/*

 PDFAttachmentFormMasterFactory.java

 Copyright 2004-2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Created on 2006/07/10

 */
package net.sqs2.omr.master.pdfattachment;

import java.awt.Point;
import java.awt.Rectangle;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.xml.parsers.FactoryConfigurationError;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerException;

import net.sqs2.xml.XMLUtil;
import net.sqs2.exigrid.master.PageMaster;
import net.sqs2.exigrid.master.PageMasterFactory;
import net.sqs2.exigrid.master.PageMasterException;
import net.sqs2.exigrid.master.PageMasterFileException;
import net.sqs2.omr.master.FormArea;
import net.sqs2.omr.master.FormMaster;
import net.sqs2.translator.TranslatorException;
import net.sqs2.util.FileResource;

import org.apache.xpath.XPathAPI;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;


public class PDFAttachmentFormMasterFactory implements PageMasterFactory {

	static PageMasterFactory createInstance(){
		return new PDFAttachmentFormMasterFactory();
	}
	
	private Rectangle createRectangle(Element rect){
		return new Rectangle(Integer.parseInt(rect.getAttribute("x")),
				Integer.parseInt(rect.getAttribute("y")),
				Integer.parseInt(rect.getAttribute("width")),
				Integer.parseInt(rect.getAttribute("height")));
	}

	public FormMaster create(int masterIndex, File sourceRoot, String path) throws PageMasterException {
		try {
			long lastModified;
			File file = new File(sourceRoot, path);
			lastModified = file.lastModified();
			FormMaster master = new FormMaster(masterIndex, new FileResource(path, lastModified));
			Document document = createDocument(master, file);
			Element svgNode = document.getDocumentElement();
			
			setMasterMetadata(svgNode, master);
			
			int prevPage = -1;
			String prevQID = null;
			int areaIndexInPage = 0;
			int itemIndex = 0;

			NodeList pageElementList = XPathAPI.selectNodeList(svgNode, "svg:pageSet/svg:page");
			
			for(int pageIndex = 0; pageIndex < pageElementList.getLength(); pageIndex++){
				//System.err.println("=====pageIndex "+pageIndex+"/"+pageElementList.getLength());
				NodeList gElementList = XPathAPI.selectNodeList(pageElementList.item(pageIndex), "svg:g");
				ArrayList<FormArea> areaListByPageIndex = new ArrayList<FormArea>();
				master.addFormAreaList(areaListByPageIndex);

		
				for(int gIndex = 0; gIndex < gElementList.getLength(); gIndex++){
					//System.err.println("=====itemIndex "+gIndex+"/"+ gElementList.getLength());
					
					Element gElement = (Element)gElementList.item(gIndex);
					
					FormArea area = FormAreaFactory.create(gElement, pageIndex);
					if(area == null){
						throw new PageMasterFileException(file);
					}
					
					List<FormArea> areaListByQID = getAreaListByQID(master, area.getQID());

					areaListByPageIndex.add(area);
					areaListByQID.add(area);

					master.getFormAreaList().add(area);
					master.putFormArea(area.getID(), area);
					master.setAreaIndexInPage(area.getQID(), areaIndexInPage);

					//System.err.println(area.getID()+"-> "+area.getPage()+"   "+area.getQID()+" - "+areaIndexInPage);
					
					if (prevPage == area.getPage()) {
						areaIndexInPage++;
					}else{
						areaIndexInPage = 0;
					}
					if (area.getQID().equals(prevQID)) {
						itemIndex++;
					}else{
						master.addQID(area.getQID());
						itemIndex = 0;
					}
					
					area.setIndex(itemIndex, areaIndexInPage);
					prevQID = area.getQID();
					prevPage = area.getPage();
	
				}				
			}
			Logger.getLogger("master").log(Level.INFO, master.getPath());
			return master;
		} catch (TransformerException e) {
			throw new PageMasterException("invalid format:"+path);
		}
	}

	private void setMasterMetadata(Element svgNode, FormMaster master) throws TransformerException {
		Element metadataNode = (Element)XPathAPI.selectSingleNode(svgNode, "/svg:svg/svg:pageSet/svg:masterPage/svg:metadata");
		setMasterMaster(metadataNode, master);
		setMasterCorners(metadataNode, master);
		setMasterCheckers(metadataNode, master);
	}

	private void setMasterCheckers(Element metadataNode, FormMaster master) throws TransformerException {
		Element upsideDownCheckerHeaderRectangle = 
			(Element)XPathAPI.selectSingleNode(metadataNode, "master:upsideDownChecker/master:checkerArea[@side='header']/svg:rect");
		Element upsideDownCheckerFooterRectangle = 
			(Element)XPathAPI.selectSingleNode(metadataNode, "master:upsideDownChecker/master:checkerArea[@side='footer']/svg:rect");
		
		Element evenOddCheckerLeftRectangle =
			(Element)XPathAPI.selectSingleNode(metadataNode, "master:evenOddChecker/master:checkerArea[@side='left']/svg:rect");
		Element evenOddCheckerRightRectangle =
			(Element)XPathAPI.selectSingleNode(metadataNode, "master:evenOddChecker/master:checkerArea[@side='right']/svg:rect");
		
		master.setHeaderCheckArea(createRectangle(upsideDownCheckerHeaderRectangle));
		master.setFooterCheckArea(createRectangle(upsideDownCheckerFooterRectangle));
		master.setFooterLeftRectangle(createRectangle(evenOddCheckerLeftRectangle));
		master.setFooterRightRectangle(createRectangle(evenOddCheckerRightRectangle));
	}

	private void setMasterMaster(Element metadataNode, FormMaster master) throws TransformerException {
		Element masterNode = (Element)XPathAPI.selectSingleNode(metadataNode, "master:master");	
		master.setNumPages(Integer.parseInt(masterNode.getAttribute("numPages")));
		master.setVersion(masterNode.getAttribute("version"));
	}

	private void setMasterCorners(Element metadataNode, FormMaster master) throws TransformerException {
		Element cornerNode = (Element)XPathAPI.selectSingleNode(metadataNode, "master:corner");	
		Point[] masterGuideCorners = new Point[4]; 
		for(int i = 1; i <= 4; i++){
			masterGuideCorners[i-1] = new Point(
					Integer.parseInt(cornerNode.getAttribute("x"+i)),
					Integer.parseInt(cornerNode.getAttribute("y"+i)));
		}
		master.setCorners(masterGuideCorners);
	}

	private Document createDocument(PageMaster master, File pdfFile) throws PageMasterException {
		InputStream pdfInputStream = null;
		InputStream sqmInputStream = null;
		try {	
			pdfInputStream = new BufferedInputStream(new FileInputStream(pdfFile));
			PDFAttachmentToSQMTranslator translator = new PDFAttachmentToSQMTranslator();
			sqmInputStream = translator.translate(pdfInputStream, pdfFile.toURI().toString());
			Document doc = XMLUtil.createDocumentBuilder().parse(sqmInputStream);
			return doc;
		} catch (SAXException ex) {
			throw new PageMasterException(ex);
		} catch (IOException ex) {
			throw new PageMasterException(ex);
		} catch (TranslatorException ex) {
			throw new PageMasterException(ex);
		} catch (FactoryConfigurationError ex) {
			throw new RuntimeException(ex);
		} catch (ParserConfigurationException ex) {
			throw new RuntimeException(ex);
		}finally{
			try{
				sqmInputStream.close();
				pdfInputStream.close();
			}catch(Exception ignore){
			}
		}
	}

	private ArrayList<FormArea> getAreaListByQID(FormMaster master, String qid) {
		ArrayList<FormArea> areaList = master.getFormAreaList(qid);
		if (areaList == null) {
			areaList = new ArrayList<FormArea>();
			master.putFormAreaList(qid, areaList);
		}
		return areaList;
	}

}
