/**
 * 
 */
package net.sqs2.omr.execute.page;

import java.awt.Point;
import java.awt.Rectangle;
import java.awt.image.BufferedImage;

import net.sqs2.image.ImageUtil;

class PageGuideArea{
	
	private static final float HEADER_GUIDEBLOCK_WIDTH_MIN = 0.027f;
	private static final float HEADER_GUIDEBLOCK_WIDTH_MAX = 0.051f;
	private static final float HEADER_GUIDEBLOCK_HEIGHT_MIN = 0.13f; 
	private static final float HEADER_GUIDEBLOCK_HEIGHT_MAX = 0.35f; 
	private static final float GUIDEBLOCK_RECOGNITION_DENSITY_THRESHOLD = 0.75f;
	
	BufferedImage image;
	int imageWidth;
	int imageHeight;
	
	int bitmapWidth;
	int bitmapHeight;

	int horizontalMargin;
	int pageGuideAreaHeight;
	int verticalMargin;

	int[] numSilhouettes = new int[2];
	int[][] silhouetteIndexArray = new int[2][];
	boolean[][] bitmaps;
	
	public PageGuideArea(BufferedImage image, float pageGuideAreaHeightRatio, float horizontalMarginRatio, float verticalMarginRatio,
			float blackThreshold, int bitmapWidth, int bitmapHeight){
		this.image = image;
		this.imageWidth = image.getWidth();
		this.imageHeight = image.getHeight();
		this.bitmapWidth = bitmapWidth;
		this.bitmapHeight = bitmapHeight;
		
		this.horizontalMargin = (int)(this.imageWidth *horizontalMarginRatio);
		this.pageGuideAreaHeight = (int)(this.imageHeight * pageGuideAreaHeightRatio);
		this.verticalMargin = (int)(this.imageHeight * verticalMarginRatio);

		this.bitmaps  = createBitmap(image, pageGuideAreaHeightRatio,
				horizontalMarginRatio, verticalMarginRatio, blackThreshold,
                bitmapWidth, bitmapHeight);
	}
	
	public Point[] scanCorners()throws PageFrameException{
		Point[] corners = new Point[4];
		extractMetadata(this.bitmapWidth, this.bitmapHeight, corners, 0);
		extractMetadata(this.bitmapWidth, this.bitmapHeight, corners, 1);
		/*
		System.err.println("0 "+corners[0]);
		System.err.println("1 "+corners[1]);
		System.err.println("2 "+corners[2]);
		System.err.println("3 "+corners[3]);
		*/
		return corners;
	}

	private void extractMetadata(int bitmapWidth, int bitmapHeight, Point[] corners, int type) throws PageFrameException{
	    ImageSilhouetteExtract ise = new ImageSilhouetteExtract(this.bitmaps[type], bitmapWidth, bitmapHeight);
		this.silhouetteIndexArray[type] = ise.getSilhouetteIndexArray();
		int[] areaArray = filterArea(ise, bitmapWidth, bitmapHeight, type);

		int[] xMinArray = new int[areaArray.length]; 
		int[] xMaxArray = new int[areaArray.length];
		int[] yMinArray = new int[areaArray.length]; 
		int[] yMaxArray = new int[areaArray.length]; 

		int[] gxTotal = new int[areaArray.length]; 
		int[] gyTotal = new int[areaArray.length];
		
		initXYMinMaxArray(type, areaArray, bitmapWidth, bitmapHeight, xMinArray, xMaxArray, yMinArray, yMaxArray, gxTotal, gyTotal);
		
		int silhouetteIndexMin = -1;
		int silhouetteIndexMax = -1;
		int gxMin = Integer.MAX_VALUE;
		int gxMax = Integer.MIN_VALUE;
		
		int W_MIN = (int)(this.bitmapWidth  * HEADER_GUIDEBLOCK_WIDTH_MIN / (type + 1));
		int W_MAX = (int)(this.bitmapWidth * HEADER_GUIDEBLOCK_WIDTH_MAX / (type + 1));
		int H_MIN = (int)(HEADER_GUIDEBLOCK_HEIGHT_MIN * this.bitmapHeight);
		int H_MAX = (int)(HEADER_GUIDEBLOCK_HEIGHT_MAX * this.bitmapHeight);
		
		for(int silhouetteIndex = areaArray.length - 1; 0 <= silhouetteIndex; silhouetteIndex--){
			int area = areaArray[silhouetteIndex];
			
			int w = xMaxArray[silhouetteIndex] - xMinArray[silhouetteIndex];
			int h = yMaxArray[silhouetteIndex] - yMinArray[silhouetteIndex];
			
			if(0  < area){
				if(W_MIN <= w && w  <= W_MAX && H_MIN <= h && h  <= H_MAX &&
						w * h * GUIDEBLOCK_RECOGNITION_DENSITY_THRESHOLD <= area){
					int gx = gxTotal[silhouetteIndex] / areaArray[silhouetteIndex];				
					if(gx <= gxMin){
						silhouetteIndexMin = silhouetteIndex;
						gxMin = gx;
						//System.err.println(type+"["+silhouetteIndex+"]\t"+" "+ gxMin );
					}
					if(gxMax <= gx){
						silhouetteIndexMax = silhouetteIndex;
						gxMax = gx;
						//System.err.println(type+"["+silhouetteIndex+"]\t\t\t "+ gxMax);
					}
				}else{
					if(true){
						System.err.println("t "+type +"   " + xMinArray[silhouetteIndex]);
						System.err.println("w "+W_MIN +" <= " + w +" <= " + W_MAX);
						System.err.println("h "+H_MIN +" <= " + h +" <= " + H_MAX);
						System.err.println("a "+(w * h * GUIDEBLOCK_RECOGNITION_DENSITY_THRESHOLD) +" <= "+ area);
						System.err.println();
					}
				}
				//}else{
				//areaArray[silhouetteIndex] = 0;
			}
		}
		
		if(silhouetteIndexMin == -1 || silhouetteIndexMax == -1){
			throw new PageFrameException(new PageGuideBlockMissingExceptionCore(image.getWidth(), image.getHeight(), type));
		}

		corners[type*2+0] = getBlockCenterPoint(type, xMinArray, xMaxArray, yMinArray, yMaxArray, silhouetteIndexMin);
		corners[type*2+1] = getBlockCenterPoint(type, xMinArray, xMaxArray, yMinArray, yMaxArray, silhouetteIndexMax);
	}

	private Point getBlockCenterPoint(int type, int[] xMinArray, int[] xMaxArray, int[] yMinArray, int[] yMaxArray,
            int silhouetteIndex) {
		Rectangle focusAreaRight = null;
		if(type == 0){
		    int x0 = translateX(xMinArray[silhouetteIndex]);
		    int y0 = translateHeaderY(yMinArray[silhouetteIndex]);
			int x1 = translateX(xMaxArray[silhouetteIndex]);
			int y1 = translateHeaderY(yMaxArray[silhouetteIndex]);
			focusAreaRight = new Rectangle(x0 - 3, y0 - 3, x1 - x0 + 9, y1 - y0 + 9);
		}else{
		    int x0 = translateX(xMinArray[silhouetteIndex]);
			int y0 = translateFooterY(yMinArray[silhouetteIndex]); 
			int x1 = translateX(xMaxArray[silhouetteIndex]);
			int y1 = translateFooterY(yMaxArray[silhouetteIndex]); 
			focusAreaRight = new Rectangle(x0 - 3, y1 - 3, x1 - x0 + 9, y0 - y1 + 9);
		}
		//System.err.println(type+":"+focusAreaRight);
		BlockCenterPointExtract bcepRight = new BlockCenterPointExtract(this.image, focusAreaRight);
		return bcepRight.getBlockCenterPoint();
    }
	
	private void initXYMinMaxArray(int type, 
            int[] areaArray, int bitmapWidth, int bitmapHeight,
            int[] xMinArray, int[] xMaxArray, int[] yMinArray,
            int[] yMaxArray,
            int[] gxTotal, int[] gyTotal) {
		
	    for(int silhouetteIndex = areaArray.length - 1; 0 <= silhouetteIndex; silhouetteIndex --){
			xMinArray[silhouetteIndex] = Integer.MAX_VALUE;
			xMaxArray[silhouetteIndex] = Integer.MIN_VALUE;
			yMinArray[silhouetteIndex] = Integer.MAX_VALUE;
			yMaxArray[silhouetteIndex] = Integer.MIN_VALUE;
		}

	    int pixelIndex = bitmapWidth * bitmapHeight - 1;
		int[] s = this.silhouetteIndexArray[type];
		for(int y = bitmapHeight - 1; 0 <= y; y--){
			for(int x = bitmapWidth - 1; 0 <= x; x--){
				int silhouetteIndex = s[pixelIndex];
				int area = areaArray[silhouetteIndex];
				if(0  < area){
					xMinArray[silhouetteIndex] = Math.min(xMinArray[silhouetteIndex], x); 
					xMaxArray[silhouetteIndex] = Math.max(xMaxArray[silhouetteIndex], x);
					yMinArray[silhouetteIndex] = Math.min(yMinArray[silhouetteIndex], y); 
					yMaxArray[silhouetteIndex] = Math.max(yMaxArray[silhouetteIndex], y);
					gxTotal[silhouetteIndex] += x;
					gyTotal[silhouetteIndex] += y;
				}else{
					s[pixelIndex] = 0;
				}
				pixelIndex--;
			}
		}
    }

	private int translateHeaderY(int y){
		return translateY(y);
	}

	private int translateFooterY(int y){
		return this.imageHeight - translateY(y);
	}
	
	private int translateX(int x) {
	    return this.horizontalMargin + (x) * (this.imageWidth - this.horizontalMargin * 2) / this.bitmapWidth;
    }
	
	private int translateY(int y) {
	    return this.verticalMargin + (y) * this.pageGuideAreaHeight / this.bitmapHeight;
    }

	private int[] filterArea(ImageSilhouetteExtract ise, int width, int height, int type)throws PageFrameException{
		int w_ = (int)Math.floor(HEADER_GUIDEBLOCK_WIDTH_MIN * width);
		int _w = (int)Math.ceil(HEADER_GUIDEBLOCK_WIDTH_MAX * width);
		int h_ = (int)Math.floor(HEADER_GUIDEBLOCK_HEIGHT_MIN * height);
		int _h = (int)Math.ceil(HEADER_GUIDEBLOCK_HEIGHT_MAX * height);

		int a_ = w_ * h_ / (type + 1);
		int _a = _w * _h / (type + 1);

		int[] areaArray = ise.getAreaArray();
		this.numSilhouettes[type] = areaArray.length;

		for(int silhouetteIndex = areaArray.length - 1; 0 <= silhouetteIndex; silhouetteIndex--){
			int area = areaArray[silhouetteIndex];
			//char c = getLabelChar(silhouetteIndex);
			if(area <= a_ || _a <= area){
				//System.err.println(type+"["+c+"] "+ a_ +"   "+area+"   "+_a);
				areaArray[silhouetteIndex] = 0;
				this.numSilhouettes[type] --;
			}else{
				//System.err.println(type+"["+c+"] "+ a_ +" < "+area+" < "+_a);
			}
		}
		return areaArray;
	}

	private boolean[][] createBitmap(BufferedImage image,
            float pageGuideAreaHeightRatio, float horizontalMarginRatio,
            float verticalMarginRatio, float blackThreshold, 
            int bitmapWidth, int bitmapHeight) {
		
		boolean[][] bitmap = new boolean[2][bitmapWidth*bitmapHeight];
		int[][] _bitmap = new int[2][bitmapWidth*bitmapHeight];
		int[] count = new int[bitmapWidth*bitmapHeight];

		for(int y = verticalMargin; y < pageGuideAreaHeight; y++){
			int bitmapY = (y - verticalMargin) * bitmapHeight / pageGuideAreaHeight;
			for(int x = horizontalMargin; x < this.imageWidth - horizontalMargin; x++){
				int bitmapX = (x - this.horizontalMargin ) * bitmapWidth / (this.imageWidth - 2 * this.horizontalMargin);
				int index = bitmapX + bitmapY * bitmapWidth; 
				_bitmap[0][index] += ImageUtil.rgb2gray(image.getRGB(x, y));  
				_bitmap[1][index] += ImageUtil.rgb2gray(image.getRGB(x,  this.imageHeight - y));
				count[index]++;
			}
		}

		for(int type = 0; type < 2; type++){
			int[] _b = _bitmap[type];
			boolean[] b = bitmap[type];
			for(int y = 0; y < bitmapHeight; y++){
				for(int x = 0; x < bitmapWidth; x++){
					int index = x + y * bitmapWidth;
					if(count[index] != 0 && _b[index] / count[index] < 255 * blackThreshold ){
						b[index] = true;
						//System.err.print("*");
					}else{
						//System.err.print(".");
					}
					//if(x == bitmapWidth - 1){
					//	System.err.println();
		        	//}
		        }
			}
			//System.err.println("------------------------------------------");
		}
		
		return bitmap;
    }

	@Override
	public String toString(){
		
		StringBuilder builder = new StringBuilder(1024); 
		
		for(int type = 0; type < 2; type++){
			int pixelIndex = 0;
			int[] a = this.silhouetteIndexArray[type];
			if(a == null){
				continue;
			}
			
			builder.append("################################################################\n");
			if(this.numSilhouettes[type] == 12){
				builder.append("OK: "+this.numSilhouettes[type]);
			}else{			
				builder.append("NG: "+this.numSilhouettes[type]);
			}
			builder.append("\n");
			
			for(int y = 0; y < this.bitmapHeight; y++){
				for(int x = 0; x < this.bitmapWidth; x++){
					builder.append(getLabelChar(a[pixelIndex++]));
				}
				builder.append("\n");
			}
		}
		return builder.toString();
	}
	
	final static String LABEL = ".0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";
	final static int LABEL_LENGTH = LABEL.length();

	private static char getLabelChar(int p) {
	    char c;
	    if(p < LABEL_LENGTH){
	    	c = LABEL.charAt(p);
	    }else{
	    	c = '*';
	    }
	    return c;
    }
}
