/*

 PageFrameScannerImpl.java

 Copyright 2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Created on Apr 7, 2007

 */
package net.sqs2.omr.execute.page;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.image.BufferedImage;

import net.sqs2.exigrid.source.PageTaskResult;
import net.sqs2.omr.master.FormMaster;
import net.sqs2.omr.page.PageSource;
import net.sqs2.omr.source.config.GuideAreaExtractionConfig;
import net.sqs2.omr.source.config.FrameConfig;
import net.sqs2.omr.source.config.SourceConfig;
import net.sqs2.omr.source.config.ValidationConfig;

public class PageFrameScanner{

	static final float PAGE_GUIDE_BLOCK_RECOGNITION_BLACK_THRESHOLD = 0.50f;
	public static final int bitmapWidth = 400; 
	public static final int bitmapHeight = 40; 

	FormMaster formMaster;
	PageGuideArea pageGuideArea;
	protected final SourceConfig sourceConfig;
	protected final BufferedImage image;
	int guideAreaStartY = 0;
	int guideAreaEndY = 0;

	public PageFrameScanner(SourceConfig sourceConfig, BufferedImage image){
		this(sourceConfig, image, null);
	}
	
	public PageFrameScanner(SourceConfig sourceConfig, BufferedImage image, FormMaster formMaster){
		this.formMaster = formMaster;
		this.sourceConfig = sourceConfig;
		this.image = image;

		FrameConfig frameConfig = getFrameConfig();
		GuideAreaExtractionConfig guideAreaExtractionConfig = getGuideAreaExtractionConfig(frameConfig);
		this.pageGuideArea = new PageGuideArea(this.image, 
				(float)guideAreaExtractionConfig.getPageGuideAreaHeightRatio(),
				(float)guideAreaExtractionConfig.getHorizontalMarginRatio(),
				(float)guideAreaExtractionConfig.getVerticalMarginRatio(), 
				PAGE_GUIDE_BLOCK_RECOGNITION_BLACK_THRESHOLD,
				bitmapWidth, bitmapHeight); 
	}

	public Point[] scanPageFrameCorners() throws PageFrameException {
		Point[] corners = this.pageGuideArea.scanCorners();
		if(formMaster != null){
			validate(getFrameConfig().getValidation(), formMaster.getCorners(), corners);
		}
		return corners;
	}

	private Point[] validate(ValidationConfig validationConfig, Point[] masterCorners, Point[] corners)
	throws PageFrameException {
		double h1 = corners[0].distance(corners[1]);
		double h2 = corners[2].distance(corners[3]);
		double dh = Math.min(h1, h2) / Math.max(h1, h2);

		double v1 = corners[0].distance(corners[2]);
		double v2 = corners[1].distance(corners[3]);
		double dv = Math.min(v1, v2) / Math.max(v1, v2);

		if (dh < validationConfig.getHorizontalThreshold()) {
			throw new PageFrameException(new PageFrameDistortionExceptionCore(this.image.getWidth(), this.image.getHeight(), 
					masterCorners, corners));
		}
		if (dv < validationConfig.getVerticalThreshold()) {
			throw new PageFrameException(new PageFrameDistortionExceptionCore(this.image.getWidth(), this.image.getHeight(), 
					masterCorners, corners));
		}
		return corners;
	}

	/*
	private static final Rectangle UPSIDEDOWN_CHECK_AREA_HEADER = new Rectangle(89, 19, 20, 20); 
	private static final Rectangle UPSIDEDOWN_CHECK_AREA_FOOTER = new Rectangle(84, 800, 20, 20);
	private static final Rectangle EVENODD_CHECK_AREA_LEFT = new Rectangle(31, 788, 24, 11);
	private static final Rectangle EVENODD_CHECK_AREA_RIGHT = new Rectangle(554, 788, 24, 11);
	 */
	
	public void checkUpsideDown(FormMaster omrMaster, PageSource pageSource)
	throws PageFrameException {
		int headerDensity = pageSource.getGrayscaleDensity(omrMaster.getHeaderCheckArea());
		int footerDensity = pageSource.getGrayscaleDensity(omrMaster.getFooterCheckArea());
		//int headerDensity = pageSource.getGrayscaleDensity(UPSIDEDOWN_CHECK_AREA_HEADER);
		//int footerDensity = pageSource.getGrayscaleDensity(UPSIDEDOWN_CHECK_AREA_FOOTER);
		if(headerDensity < footerDensity){
			return;
		}
		throw new PageFrameException(new PageUpsideDownExceptionCore(this.image.getWidth(), this.image.getHeight(), pageSource.getMasterCorners(), 
				pageSource.getBlockCenters(), 
				omrMaster.getHeaderCheckArea(),
				omrMaster.getFooterCheckArea(),
				headerDensity, footerDensity));
	}

	public void checkPageOrder(PageTaskResult result, FormMaster omrMaster, int pageNumber,
			PageSource pageSource) throws PageFrameException {
		Rectangle footerAreaLeft = omrMaster.getFooterLeftRectangle();
		Rectangle footerAreaRight = omrMaster.getFooterRightRectangle();
		int left = pageSource.getGrayscaleDensity(footerAreaLeft);
		int right = pageSource.getGrayscaleDensity(footerAreaRight);

		if(pageNumber % 2 == 1){
			if(left > right){
				return;
			}
		}else{
			if(left < right){
				return;
			}
		}
		throw new PageFrameException(new PageSequenceInvalidExceptionCore(this.image.getWidth(), this.image.getHeight(), 
				pageSource.getMasterCorners(), pageSource.getBlockCenters(), footerAreaLeft, left, footerAreaRight, right));
	}

	protected GuideAreaExtractionConfig getGuideAreaExtractionConfig(FrameConfig frameConfig) {
		return frameConfig.getGuideAreaExtraction();
	}

	protected FrameConfig getFrameConfig() {
		return this.sourceConfig.getFrame();
	}
}

