/*

 SessionWorkerImpl.java

 Copyright 2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Created on 2007/01/11

 */
package net.sqs2.omr.execute;

import java.awt.Point;
import java.awt.Rectangle;

import java.awt.geom.AffineTransform;
import java.awt.image.AffineTransformOp;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.rmi.RemoteException;
import java.util.List;
import java.util.logging.Logger;


import net.sqs2.exigrid.execute.SessionExecutorCore;
import net.sqs2.exigrid.execute.SessionExecutorTask;
import net.sqs2.exigrid.master.PageMaster;

import net.sqs2.exigrid.source.PageID;
import net.sqs2.exigrid.source.PageTask;
import net.sqs2.exigrid.source.PageTaskConfig;
import net.sqs2.exigrid.source.PageTaskError;
import net.sqs2.exigrid.source.PageTaskException;
import net.sqs2.exigrid.source.PageTaskResult;
import net.sqs2.image.ImageFactory;
import net.sqs2.image.ImageUtil;
import net.sqs2.omr.execute.page.PageFrameException;
import net.sqs2.omr.execute.page.PageFrameScanner;
import net.sqs2.omr.master.FormArea;
import net.sqs2.omr.master.FormMaster;
import net.sqs2.omr.page.PageSource;
import net.sqs2.omr.source.FormAreaCommand;
import net.sqs2.omr.source.config.ConfigConfigImpl;
import net.sqs2.omr.source.config.SourceConfig;

public class SessionExecutorCoreImpl implements SessionExecutorCore{

	public static final int TEXTAREA_MARGIN = 0;
	
	public static final int MARKAREA_MARGIN_X = 3;
	public static final int MARKAREA_MARGIN_Y = -4;
	public static final int MARKAREA_PREVIEW_MARGIN_X = 3;
	public static final int MARKAREA_PREVIEW_MARGIN_Y = 0;
	
	public static final int WAKEUP_IMAGEFILTER_THRESHOLD = 100;
	public static final int DEFAULT_BLACK_WHITE_THRESHOLD = 230;

	public SessionExecutorCoreImpl(){
	}

	public void execute(SessionExecutorTask sessionTask)throws RemoteException{
		long base = System.currentTimeMillis();
		PageTask pageTask = sessionTask.getPageTask();

		try{
			FormMaster formMaster = (FormMaster)sessionTask.getSessionExecutorResource().getPageMaster(pageTask);

			BufferedImage pageImage = retrievePageBufferedImage(sessionTask);
			PageTaskConfig pageTaskConfig = sessionTask.getPageTaskConfig();
			if(pageTaskConfig == null){
				pageTaskConfig = sessionTask.getSessionExecutorResource().getDefaultPageTaskConfig();
			}

			ConfigConfigImpl configConfig = (ConfigConfigImpl)pageTaskConfig.getConfigConfig(); 
			SourceConfig sourceConfig = configConfig.getSourceConfig(pageTask);
			String formAreaImageFormat = configConfig.getResult().getTextAreaImageFormat();
			if(formAreaImageFormat == null){
				formAreaImageFormat = "png";
			}
			PageFrameScanner pageFrameScanner = new PageFrameScanner(sourceConfig, pageImage, formMaster);
			Point[] pageCorners = pageFrameScanner.scanPageFrameCorners();
			PageSource pageSource = createPageSource(formMaster, pageImage, pageCorners);
			pageFrameScanner.checkUpsideDown(formMaster, pageSource);
			PageTaskResult result = new PageTaskResult(pageCorners);
			pageFrameScanner.checkPageOrder(result, formMaster, pageTask.getPageNumber(), pageSource);
			addFormAreaCommandList(result, formMaster, pageTask.getPageNumber(), pageSource, formAreaImageFormat);
			pageTask.setResult(result);
			long submitLap = System.currentTimeMillis() - base;
			Logger.getLogger("executor").info("[[Process OK in "+submitLap+" msec]]\t"+pageTask);			
		}catch(PageFrameException ex){
			setPageTaskError(sessionTask, ex);
		}catch(RemoteException ex){
			//ex.printStackTrace();
			throw ex;
		}catch(IOException ex){
			ex.printStackTrace();
			pageTask.setPageTaskError(new PageTaskError(ex.getMessage()));
		}
	}

	private static void setPageTaskError(SessionExecutorTask sessionTask, PageTaskException pageTaskException) {
		PageTask pageTask = sessionTask.getPageTask();
		PageID pageID = pageTask.getPageID();
		Logger.getLogger("executor").warning("[[Process NG]]\t"+pageID.getPath()+" "+pageTaskException.getExceptionCore().toString());
		byte[] imageByteArray = null;
		String imageType = null;
		double scale = 0;
		try{
			if(! isWebBrowserSupportedImageType(pageID)){
				imageType = "png";
				BufferedImage image = retrievePageBufferedImage(sessionTask);
				scale = 400.0/image.getWidth();
				BufferedImage dest = new BufferedImage((int)(image.getWidth()*scale), (int)(image.getHeight()*scale), image.getType()); 
				new AffineTransformOp(AffineTransform.getScaleInstance(scale, scale), AffineTransformOp.TYPE_BILINEAR).filter(image, dest);
				imageByteArray = ImageFactory.writeImage(image, imageType);
			}else{
				scale = 1.0;
			}
		}catch(IOException ex){
			ex.printStackTrace();
		}
		pageTask.setPageTaskError(new PageTaskError(pageTask.getPageID(), imageByteArray, imageType, scale, pageTaskException.getExceptionCore()));
	}

	private static boolean isWebBrowserSupportedImageType(PageID pageID) {
		return pageID.getExtension().equals("png") || pageID.getExtension().equals("gif") || pageID.getExtension().equals("jpg") || pageID.getExtension().equals("jpeg");
	}

	private static BufferedImage retrievePageBufferedImage(SessionExecutorTask sessonTask) throws RemoteException,IOException{
		PageID pageID = sessonTask.getPageTask().getPageID();
		byte[] imageByteArray = sessonTask.getSessionExecutorResource().getImageByteArray(pageID.getPath());
		String type = ImageUtil.getType(pageID.getExtension());
		if(imageByteArray == null){
			throw new RuntimeException(pageID.toString());
		}
		BufferedImage pageImage = ImageFactory.createImage(imageByteArray, pageID.getIndex(), type);
		return pageImage;
	}

	private static PageSource createPageSource(PageMaster pageMaster, BufferedImage pageImage, Point[] pageCorners) {
		return new PageSource(pageImage, pageMaster.getCorners(), pageCorners, 1.0f, DEFAULT_BLACK_WHITE_THRESHOLD);
	}

	private static void addFormAreaCommandList(PageTaskResult result, FormMaster pageMaster, int pageNumber, PageSource pageSource, String formAreaImageFormat)throws IOException{
		List<FormArea> formAreaList = pageMaster.getFormAreaListByPageIndex(pageNumber - 1);
		for(FormArea formArea: formAreaList){
			if(formArea.isTextArea()){
				BufferedImage image = pageSource.cropImage(getMarginRect(formArea.getRect(), 
						TEXTAREA_MARGIN, TEXTAREA_MARGIN, 
						TEXTAREA_MARGIN, TEXTAREA_MARGIN));
				result.addFormAreaCommand(createFormAreaCommand(formArea.getID(), formAreaImageFormat,
						ImageFactory.writeImage(image, formAreaImageFormat), 0));
			}else if(formArea.isMarkArea()){
				/*
				BufferedImage image = pageSource.cropImage(getMarginRect(formArea.getRect(),
						MARKAREA_PREVIEW_MARGIN_X, MARKAREA_PREVIEW_MARGIN_X, 
						MARKAREA_PREVIEW_MARGIN_Y, MARKAREA_PREVIEW_MARGIN_Y));
				*/
				int density = pageSource.getGrayscaleDensity(getMarginRect(formArea.getRect()));
				if(WAKEUP_IMAGEFILTER_THRESHOLD <= density){ //&& density <= DEFAULT_BLACK_WHITE_THRESHOLD
					density = pageSource.getGrayscaleDensityWithFilter(getMarginRect(formArea.getRect()), 5);
				}
				//ImageFactory.writeImage(image, formAreaImageFormat)
				result.addFormAreaCommand(createFormAreaCommand(formArea.getID(), formAreaImageFormat,
						null, density/255.0f));
			}else{
				throw new RuntimeException("invalid formArea:"+formArea);
			}
		}
	}

	private static FormAreaCommand createFormAreaCommand(String id, String imageType, byte[] imageByteArray, float density){
		return new FormAreaCommand(id, imageType, imageByteArray, density);
	}

	private static Rectangle getMarginRect(Rectangle rect, int marginLeft, int marginRight, int marginTop, int marginBottom){
		return new Rectangle(rect.x-marginLeft, rect.y-marginTop, rect.width + marginLeft + marginRight, rect.height + marginTop + marginBottom);
	}
	
	public static Rectangle getMarginRect(Rectangle rect){
		return getMarginRect(rect, MARKAREA_MARGIN_X, MARKAREA_MARGIN_X, MARKAREA_MARGIN_Y, MARKAREA_MARGIN_Y);
	}

	public static Rectangle getPreviewMarginRect(Rectangle rect){
		return getMarginRect(rect, MARKAREA_PREVIEW_MARGIN_X, MARKAREA_PREVIEW_MARGIN_X, MARKAREA_PREVIEW_MARGIN_Y, MARKAREA_PREVIEW_MARGIN_Y);
	}

}
