/*

 SessionProgressModel.java

 Copyright 2004-2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Created on 2006/01/10

 */
package net.sqs2.exigrid.swing;

import java.awt.Color;
import java.io.File;
import java.io.Serializable;
import java.text.DecimalFormat;
import java.util.Map;
import java.util.Observable;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;

import net.sqs2.exigrid.event.ResultEvent;
import net.sqs2.exigrid.event.ResultManagerMonitor;
import net.sqs2.exigrid.master.PageMaster;
import net.sqs2.exigrid.session.PageTaskHolder;
import net.sqs2.exigrid.source.PageTask;
import net.sqs2.exigrid.source.PageTaskConfig;
import net.sqs2.exigrid.source.SessionSource;
import net.sqs2.lang.GroupThreadFactory;

public class SessionProgressModel extends Observable implements ResultManagerMonitor, Serializable {

	private final static long serialVersionUID = 0L;

	final static private DecimalFormat DECIMAL_FORMAT_00 = new DecimalFormat("00");

	final static private DecimalFormat DECIMAL_FORMAT_0_00 = new DecimalFormat("0.00");

	protected Color colTotal = Color.DARK_GRAY;

	protected Color colReused = Color.GRAY;

	protected Color colPrepared = Color.LIGHT_GRAY;

	protected Color colLeasedLocal = Color.YELLOW;

	protected Color colLeasedRemote = Color.CYAN;

	protected Color colSubmitted = Color.BLUE;

	protected Color colError = Color.MAGENTA;

	protected Color colExternalized = Color.GREEN;

	protected String statusMessage = "";

	Color[] COLORS = new Color[] { colTotal, colReused, colPrepared,
			colLeasedLocal, colLeasedRemote, colSubmitted, colError,
			colExternalized };

	private long timeStarted = 0L;

	private String timeElapsedString;

	private String timeRemainsString = null;

	private String pageParSecString = null;

	protected PageTaskHolder holder;

	transient protected Future<?> timer = null;
	
	boolean checkFinishedMessage = false;

	transient ScheduledExecutorService scheduledExecutorService;

	public SessionProgressModel(PageTaskHolder holder) {
		this.holder = holder;
	}
	
	public File getResultDirectoryRoot(){
		return SessionSource.getInstance().getResultDirectoryRoot();
	}
	
	public boolean checkFinishedMessage(){
		if(this.checkFinishedMessage){
			this.checkFinishedMessage = false;
			return true; 
		}
		return false;
	}

	public void notifyStarted() {
		Logger.getLogger("progressModel").info("started");
		setStatusMessage("session started....");
		startTimer();
		update();
	}

	public void notifyFinished() {
		
		Logger.getLogger("progressModel").info("finished");
		setStatusMessage("session finished.");
		stopTimer();
		this.checkFinishedMessage = true;
		update();
	}

	public void notifyStopped() {
		Logger.getLogger("progressModel").info("stopped");
		setStatusMessage("session stopped.");
		stopTimer();
		update();
	}

	public void notifyProducePageTask(PageTask pageTask) {
		startTimer();
		setStatusMessage("prepare:" + pageTask.getPageID().getPath());
		update();
	}

	public void notifyConsumePageTask(PageTask pageTask) {
		setStatusMessage("externalize:" + pageTask.getPageID().getPath());
		update();
	}

	public void notifyFoundMaster(PageMaster master, boolean scanDescendant) {
		Logger.getLogger("source").info("[[Found Master]]\t" + master.getPath());
	}

	public void notifyFoundConfig(PageTaskConfig config, boolean scanDescendant) {
		Logger.getLogger("source").info("[[Config]]\t" + config);
	}

	public void notifyFoundImages(int numAddedImages, File sourceDirectory) {
		Logger.getLogger("source").info("[[Found " + numAddedImages + " imgaes]]\t" + sourceDirectory);
	}

	public void notifyScanningDone(File sourceDirectoryRootFile) {
		Logger.getLogger("source").info("done " + sourceDirectoryRootFile);
	}

	public void update() {
		setChanged();
		notifyObservers();
	}

	public void setStatusMessage(String statusMessage) {
		this.statusMessage = statusMessage;
	}

	public String getStatusMessage() {
		return this.statusMessage;
	}

	private void startTimer() {
		if (this.timer != null) {
			this.timeStarted = System.currentTimeMillis();
			stopTimer();
		}

		GroupThreadFactory groupThreadFactory = new GroupThreadFactory(
				"net.sqs2.exigrid.swing.SessionProgressModel#Timer", Thread.NORM_PRIORITY, true);
		this.scheduledExecutorService = Executors.newScheduledThreadPool(1, groupThreadFactory);

		this.timer = this.scheduledExecutorService.scheduleAtFixedRate(
				new Runnable() {
					public void run() {
						long timeElapsed = (SessionProgressModel.this.timeStarted == 0) ? 0
								: System.currentTimeMillis() - SessionProgressModel.this.timeStarted + 1;
						timeElapsedString = createTimeElapsedString(timeElapsed);
						int interval = 3;
						if ((timeElapsed / 1000) % interval == interval - 1) {
							int processed = getNumExternalizedPages() + getNumErrorPages();
							double pageParSec = createPageParSec(timeElapsed,
									processed);
							pageParSecString = createPageParSecString(pageParSec, processed);
							timeRemainsString = createTimeRemainsString(timeElapsed, pageParSec);
						}
						update();
					}

					private double createPageParSec(long timeElapsed,
							int processed) {
						if (0 < processed && 1000 < timeElapsed) {
							return (double) (processed) * 1000 / timeElapsed;
						} else {
							return 0;
						}
					}

					private String createPageParSecString(double pageParSec,
							int processed) {
						if (0 < processed) {
							return SessionProgressModel.DECIMAL_FORMAT_0_00.format(pageParSec);
						} else {
							return null;
						}
					}

					private String createTimeRemainsString(long timeElapsed,
							double pageParSec) {
						StringBuilder timeRemainsSB = new StringBuilder(8);
						double timeRemains = getNumPreparedPages() / pageParSec;
						if (0 < timeRemains) {
							if (3600 <= timeRemains && 3000L < timeElapsed) {
								timeRemainsSB.append(SessionProgressModel.DECIMAL_FORMAT_00.format(timeRemains / 3600));
								timeRemainsSB.append(":");
							}
							timeRemainsSB.append(SessionProgressModel.DECIMAL_FORMAT_00.format(timeRemains / 60 % 60));
							timeRemainsSB.append(":");
							timeRemainsSB.append(SessionProgressModel.DECIMAL_FORMAT_00.format(timeRemains % 60));
							return timeRemainsSB.toString();
						} else {
							return null;
						}
					}

					private String createTimeElapsedString(long timeElapsed) {
						StringBuilder timeElapsedSB = new StringBuilder(8);
						if (0 < timeElapsed / 1000 / 60 / 60) {
							timeElapsedSB
							.append(SessionProgressModel.DECIMAL_FORMAT_00.format(timeElapsed / 1000 / 60 / 60));
							timeElapsedSB.append(":");
						}
						timeElapsedSB.append(SessionProgressModel.DECIMAL_FORMAT_00.format(timeElapsed / 1000 / 60 % 60));
						timeElapsedSB.append(":");
						timeElapsedSB.append(SessionProgressModel.DECIMAL_FORMAT_00.format(timeElapsed / 1000 % 60));
						return timeElapsedSB.toString();
					}
				}, 0, 1, TimeUnit.SECONDS);
	}

	private void stopTimer() {
		synchronized (this) {
			if (this.timer != null) {
				this.timer.cancel(true);
				this.timer = null;
			}
		}
		if (this.scheduledExecutorService != null) {
			this.scheduledExecutorService.shutdown();
		}
	}

	public Set<Map.Entry<Integer, String>> getErrorMapEntrySet() {
		return this.holder.getErrorMapEntrySet();
	}

	public String getTimeRemainsString() {
		return this.timeRemainsString;
	}

	public String getPageParSecString() {
		return this.pageParSecString;
	}

	public String getTimeElapsedString() {
		return this.timeElapsedString;
	}

	public int getNumTotalPages() {
		return this.holder.getNumTotalPages();
	}

	public int getNumTargetPages() {
		return this.holder.getNumTargetPages();
	}

	public int getNumReusedPages() {
		return this.holder.getNumReusedPages();
	}

	public int getNumPreparedPages() {
		return this.holder.getNumPreparedPages();
	}

	public int getNumSubmittedPages() {
		return this.holder.getNumSubmittedPages();
	}

	public int getNumLocalLeasedPages() {
		return this.holder.getNumLocalLeasedPages();
	}

	public int getNumRemoteLeasedPages() {
		return this.holder.getNumRemoteLeasedPages();
	}

	public int getNumExternalizedPages() {
		return this.holder.getNumExternalizedPages();
	}

	public int getNumErrorPages() {
		return this.holder.getNumErrorPages();
	}

	public void notifyResultEvent(ResultEvent ev) {
		System.err.println("SessionProgress:"+ev.toString());
		// TODO : ResultUpdateProgress
	}

}