/**
 *  ExigridLauncher.java

 Copyright 2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Created on 2007/04/29
 Author hiroya
 */

package net.sqs2.exigrid.swing;

import java.awt.Color;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.logging.Logger;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;

import javax.swing.JButton;

import net.sqs2.exigrid.engine.ExigridConstants;
import net.sqs2.exigrid.engine.ExigridEngine;
import net.sqs2.omr.httpd.SQSHttpdManager;
import net.sqs2.swing.DirectoryChooserEventListener;
import net.sqs2.swing.DirectoryChooserPanel;
import net.sqs2.swing.FileChooserEvent;
import net.sqs2.swing.process.RemoteWindowDecorator;

public class ExigridLauncher {

	private static final String SOURCE_DIRECTORY_ROOT_KEY = "sourceDirectoryRoot";
	transient Preferences prefs = null;
	Executor startTriggerExecutor = Executors.newSingleThreadExecutor();
	
	protected ExigridEngine engine = null;

	protected SessionProgressModel model;
	protected ExigridFrame frame;
	
	public ExigridLauncher(){
		this.frame = new ExigridFrame(this);
	}
	
	public void initialize(ExigridEngine engine, String sourceDirectoryRootPath){
		this.engine = engine;
		this.model = new SessionProgressModel(this.engine.getPageTaskHolder());
		this.engine.addSessionManagerMonitor(this.model);
		File sourceDirectoryRoot = null;
		if(sourceDirectoryRootPath != null){
			sourceDirectoryRoot = new File(sourceDirectoryRootPath);
		}else{
			sourceDirectoryRoot = new File(getPreferences().get(SOURCE_DIRECTORY_ROOT_KEY,
					System.getProperty("user.dir")));
		}		
		this.frame.initialize(this.model, sourceDirectoryRoot);
		this.frame.addWindowListener(new ExitAdapter());
		this.frame.directoryChooserPanel.addDirectoryChooserEventListener(new OMRDirectoryChooserEventListener(this.frame.directoryChooserPanel));
		initializeListeners(this.engine, this.frame.directoryChooserPanel, this.frame.startButton, this.frame.stopButton);
		
		SQSHttpdManager.initHttpds();
	}

	public String getTitle(){
		return this.engine.getTitle();
	}

	public ExigridFrame getFrame(){
		return this.frame;
	}

	private boolean isControlKeyDown(final ActionEvent ev) {
		return (ev.getModifiers() & ActionEvent.CTRL_MASK) != 0;
	}

	protected void initializeListeners(final ExigridEngine engine, final DirectoryChooserPanel directoryChooserPanel, final JButton startButton, final JButton stopButton) {

		startButton.addActionListener(new ActionListener(){
			public void actionPerformed(final ActionEvent ev){
				startTriggerExecutor.execute(new OMRSessionStartTrigger(engine, isControlKeyDown(ev)));
			}
		});

		stopButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ev){
				frame.stop();
				new Thread(){
					public void run(){
						File sourceDirectoryRoot = directoryChooserPanel.getFile(); 
						File resultDirectoryRoot = createResultDirectoryRoot(directoryChooserPanel.getFile());
						engine.stop(sourceDirectoryRoot, resultDirectoryRoot);
					}
				}.start();
			}
		});

		/*
	engine.addEngineEventListener(new EngineEventAdapter(){
		public void start(EngineStartStopEvent ev){
			//OMRGUILauncher.this.start();
		}
	public void started(){}
		public void notifyProducePageTask(EngineProcessEvent ev){
			//
		}
		public void notifyConsumePageTask(EngineProcessEvent ev){
			//
		}
		public void finished(){}
		public void stop(EngineStartStopEvent ev){
			//OMRGUILauncher.this.stop();
		}
	});
		 */

		engine.addSessionManagerMonitor(model);
	}

	protected Preferences getPreferences() {
		if(this.prefs == null){
			this.prefs = Preferences.userNodeForPackage(this.getClass());
		}
		return this.prefs;
	}

	public void activate(int port) {
		RemoteWindowDecorator.activate(this.frame, port);
	}

	protected void exit() throws Exception{
		RemoteWindowDecorator.inactivate(this.engine.getRMIPort());
		File sourceDirectoryRoot = frame.directoryChooserPanel.getFile();
		File resultDirectoryRoot = createResultDirectoryRoot(frame.directoryChooserPanel.getFile());		
		this.engine.stop(sourceDirectoryRoot, resultDirectoryRoot);
		this.engine.shutdown();
		this.frame.setVisible(false);
		System.runFinalization();
		System.exit(0);
		//throw new RuntimeException("exit");
	}

	protected File createResultDirectoryRoot(File sourceDirectoryRoot) {
		return new File(sourceDirectoryRoot.getAbsolutePath()+ExigridConstants.RESULT_FOLDER_SUFFIX);
	}

	class ExitAdapter extends WindowAdapter{
		public void windowClosing(WindowEvent e) {
			frame.exitConfirmation();
		}
	}

	class OMRSessionStartTrigger implements Runnable{
		boolean doReset;
		ExigridEngine engine;
		OMRSessionStartTrigger(ExigridEngine engine, boolean doReset){
			this.engine = engine;
			this.doReset = doReset;			
		}

		public void run(){
			final File sourceDirectoryRoot = frame.directoryChooserPanel.getFile(); 
			final File resultDirectoryRoot = createResultDirectoryRoot(sourceDirectoryRoot);
			if(! checkDirectoryExistence(sourceDirectoryRoot, resultDirectoryRoot)){
				return;
			}
			
			frame.start();	
			storeSourceDirectoryRootInPrefs(sourceDirectoryRoot);
			try{
				this.engine.start(sourceDirectoryRoot, resultDirectoryRoot, this.doReset);
			}catch(Exception ex){
				Logger.getLogger("swing").severe(ex.getMessage());
				ex.printStackTrace();
			}
		}

		private void storeSourceDirectoryRootInPrefs(final File sourceDirectoryRoot) {
			getPreferences().put(SOURCE_DIRECTORY_ROOT_KEY, sourceDirectoryRoot.getAbsolutePath());
			try{
				getPreferences().flush();
			}catch(IllegalArgumentException ignore){
				Logger.getLogger("swing").info(ignore.getMessage());
			}catch(BackingStoreException ignore){
				Logger.getLogger("swing").severe(ignore.getMessage());
			}
		}

		private boolean checkDirectoryExistence(final File sourceDirectoryRoot, final File resultDirectoryRoot) {
			if(! sourceDirectoryRoot.exists() || ! sourceDirectoryRoot.isDirectory() || ! sourceDirectoryRoot.canRead()
					 || sourceDirectoryRoot.getName().endsWith("-RESULT")){
				return false;
			}
			resultDirectoryRoot.mkdirs();
			if(! resultDirectoryRoot.exists() || ! resultDirectoryRoot.isDirectory() || ! resultDirectoryRoot.canWrite()){
				return false;
			}
			return true;
		}

	}

	class OMRDirectoryChooserEventListener implements DirectoryChooserEventListener{
		DirectoryChooserPanel directoryChooserPanel;
		OMRDirectoryChooserEventListener(DirectoryChooserPanel directoryChooserPanel){
			this.directoryChooserPanel = directoryChooserPanel;
		}

		public void chooseDirectory(FileChooserEvent ev){
			File sourceDirectoryRoot = ev.getFile();
			if(! sourceDirectoryRoot.isDirectory()){
				this.directoryChooserPanel.setForeground(Color.red);
			}else{
				this.directoryChooserPanel.setForeground(Color.black);
			}
		}
	}
}