/**
 *  SourceDirectoryScanner.java

 Copyright 2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Created on 2007/03/10
 Author hiroya
 */
package net.sqs2.exigrid.source;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.LinkedList;

import net.sqs2.exigrid.engine.ExigridConstants;
import net.sqs2.exigrid.master.PageMaster;
import net.sqs2.exigrid.master.PageMasterAccessor;
import net.sqs2.exigrid.master.PageMasterFactory;
import net.sqs2.exigrid.master.PageMasterException;
import net.sqs2.exigrid.master.PageMasterFileException;
import net.sqs2.image.ImageUtil;
import net.sqs2.util.PathUtil;

class SourceDirectoryScanner{

	private int masterIndex = 0;
	private PageMasterFactory pageMasterFactory;
	private PageTaskConfigFactory pageTaskConfigFactory; 

	private File sourceDirectoryRootFile = null;
	private LinkedList<PageMaster> currentMasterList = new LinkedList<PageMaster>(); 
	private PageTaskConfig currentPageTaskConfig = null;
	private LinkedHashMap<PageMaster, ArrayList<SourceDirectory>> sourceDirectoryListMap = new LinkedHashMap<PageMaster,ArrayList<SourceDirectory>>();
	private LinkedHashMap<String, PageMaster> pageMasterMap = new LinkedHashMap<String, PageMaster>();
	private PageMaster[] pageMasterArray = new PageMaster[0];  
	private SourceDirectoryScannerMonitor monitor = null;

	SourceDirectoryScanner(File sourceDirectoryRootFile, File resultDirectoryRootFile,
			SourceDirectoryScannerMonitor monitor, 
			PageMasterFactory pageMasterFactory,
			PageTaskConfigFactory pageTaskConfigFactory )throws IOException{//,boolean recursive
		this.sourceDirectoryRootFile = sourceDirectoryRootFile;
		this.monitor = monitor;
		this.pageMasterFactory = pageMasterFactory;
		this.pageTaskConfigFactory = pageTaskConfigFactory; 
		this.currentPageTaskConfig = this.pageTaskConfigFactory.getDefaultInstance();

		checkSourceDirectory(sourceDirectoryRootFile);
		SourceDirectory currentSourceDirectory = new SourceDirectory(sourceDirectoryRootFile); //, recursive
		PageMasterAccessor omrMasterAccessor = new PageMasterAccessor(sourceDirectoryRootFile, resultDirectoryRootFile);

		scanDescendant(currentSourceDirectory, omrMasterAccessor);

		if(this.sourceDirectoryListMap.isEmpty()){
			this.currentMasterList.clear();
			this.currentPageTaskConfig = this.pageTaskConfigFactory.getDefaultInstance();
			this.pageMasterMap.clear();
			scanAncester(currentSourceDirectory, omrMasterAccessor);
			scanDescendant(currentSourceDirectory, omrMasterAccessor);
		}

		this.pageMasterArray = this.pageMasterMap.values().toArray(this.pageMasterArray);
		
		if(this.monitor != null){
			this.monitor.notifyScanningDone(sourceDirectoryRootFile);
		}
	}

	LinkedHashMap<PageMaster,ArrayList<SourceDirectory>> getSourceDirectoryListMap(){
		return this.sourceDirectoryListMap;
	}

	PageMaster getPageMaster(String relativePath){
		return this.pageMasterMap.get(relativePath);
	}

	PageMaster getPageMaster(int masterIndex){
		return this.pageMasterArray[masterIndex];
	}

	private void checkSourceDirectory(File sourceDirectoryRootFile) throws FileNotFoundException {
		if(! sourceDirectoryRootFile.exists()){
			throw new FileNotFoundException(sourceDirectoryRootFile.getAbsolutePath());
		}
		if(! sourceDirectoryRootFile.isDirectory()){
			throw new FileNotFoundException(sourceDirectoryRootFile.getAbsolutePath());				
		}
	}

	private String ignorableDirectorySuffix = ExigridConstants.RESULT_FOLDER_SUFFIX; // TODO: null when disable this feature


	private void scanAncester(SourceDirectory sourceDirectory, PageMasterAccessor omrMasterAccessor)throws IOException{
		File parent = sourceDirectory.getDirectory().getParentFile();
		if(parent == null){
			return;
		}

		File[] items = parent.listFiles();
		SourceDirectory parentSourceDirectory = sourceDirectory.createParentSourceDirectory();
		int numAddedMasters = processFiles(parentSourceDirectory, items, true, omrMasterAccessor);
		if(0 < numAddedMasters){
			return;
		}else{
			scanAncester(parentSourceDirectory, omrMasterAccessor);
		}
	}

	private void scanDescendant(SourceDirectory sourceDirectory, PageMasterAccessor omrMasterAccessor)throws IOException{
		File[] items = sourceDirectory.getDirectory().listFiles();
		if(items == null || 0 == items.length){
			return;
		}
		Arrays.sort(items);
		int numAddedMasters = processFiles(sourceDirectory, items, true, omrMasterAccessor);
		sourceDirectory.setPageTaskConfig(this.currentPageTaskConfig);
		for(File file: items){
			if(file.isDirectory()){
				if(! isIgnorableDirectory(file)){
					scanDescendant(new SourceDirectory(sourceDirectory.getDirectoryRoot(),
							new File(sourceDirectory.getAbsolutePath(), file.getName())), omrMasterAccessor); //, this.recursive
				}
			}
		}
		removeMasters(numAddedMasters);
	}

	private boolean isIgnorableDirectory(File file) {
		return this.ignorableDirectorySuffix != null && file.getName().endsWith(this.ignorableDirectorySuffix);
	}

	private int processFiles(SourceDirectory sourceDirectory, File[] items, boolean scanDescendant, PageMasterAccessor omrMasterAccessor) throws IOException{
		int numAddedMasters = 0;
		int numAddedImages = 0;			
		for(File file : items){
			if(file.isDirectory()){
				continue;
			}
			if(file.getName().toLowerCase().endsWith(".pdf")){
				try{
					PageMaster master = createMaster(file, omrMasterAccessor, scanDescendant);
					if(master != null){
						numAddedMasters++;
						continue;
					}
				}catch(PageMasterFileException ignore){
				}
			}


			PageTaskConfig config = createPageTaskConfig(sourceDirectory.getDirectoryRoot(), file);
			if(config != null){
				this.currentPageTaskConfig = config;
				if(this.monitor != null){
					this.monitor.notifyFoundConfig(config, scanDescendant);
				}
				continue;
			}
			if(scanDescendant){
				if(ImageUtil.isSupported(file.getName())){
					numAddedImages++;
				}
			}
		}

		if(scanDescendant && 0 < numAddedImages){
			// TODO: configuration: enableOverwrappedMasterMode
			boolean enableOverwrappedMasterMode = true;
			if(enableOverwrappedMasterMode){
				for(PageMaster master : this.currentMasterList){
					ArrayList<SourceDirectory> sourceDirectorySet = this.sourceDirectoryListMap.get(master);
					sourceDirectorySet.add(sourceDirectory);
				}
			}else{
				ArrayList<SourceDirectory> sourceDirectorySet = this.sourceDirectoryListMap.get(this.currentMasterList.getFirst());
				sourceDirectorySet.add(sourceDirectory);
			}

			if(this.monitor != null){
				this.monitor.notifyFoundImages(numAddedImages, sourceDirectory.getDirectory());
			}
		}
		return numAddedMasters;
	}

	private PageMaster createMaster(File file, PageMasterAccessor omrMasterAccessor, boolean scanDescendant) throws PageMasterFileException,IOException {
		PageMaster master = createMasterCore(file, omrMasterAccessor);
		if(master != null){
			//String relativePath = file.getAbsolutePath().substring(sourceDirectoryRootFile.getAbsolutePath().length() + 1);//+1 as pathSeparator
			String path = PathUtil.getRelativePath(file, this.sourceDirectoryRootFile);
			this.pageMasterMap.put(path, master);
			this.currentMasterList.addFirst(master);						
			ArrayList<SourceDirectory> sourceDirectoryList = null;
			sourceDirectoryList = this.sourceDirectoryListMap.get(master);
			if(sourceDirectoryList == null){
				sourceDirectoryList = new ArrayList<SourceDirectory>();
				this.sourceDirectoryListMap.put(master, sourceDirectoryList);
			}
			if(this.monitor != null){
				this.monitor.notifyFoundMaster(master, scanDescendant);
			}
		}
		return master;
	}

	private PageMaster createMasterCore(File masterFile, PageMasterAccessor masterAccessor) throws PageMasterFileException{		
		PageMaster master = null;
		try{
			String path = PathUtil.getRelativePath(masterFile, this.sourceDirectoryRootFile);
			PageMaster cachedMaster = (PageMaster)masterAccessor.get(path);
			if(cachedMaster != null && cachedMaster.getLastModified() == masterFile.lastModified()){
				return cachedMaster;
			}
			master = this.pageMasterFactory.create(masterIndex++, this.sourceDirectoryRootFile, path);
			if(master == null){
				throw new PageMasterFileException(masterFile);
			}
			masterAccessor.put(master);
			masterAccessor.flush();
			return master;
		}catch(PageMasterException ex){
			throw new PageMasterFileException(masterFile);
		}catch(IOException ex){
			throw new PageMasterFileException(masterFile);
		}
	}

	private void removeMasters(int numAddedMasters) {
		for(int i = 0; i < numAddedMasters; i++){
			this.currentMasterList.removeFirst();
		}
	}

	private PageTaskConfig createPageTaskConfig(File base, File configFile){
		if(configFile.getName().toLowerCase().equals("config.xml")){
			return this.pageTaskConfigFactory.create(base.getAbsolutePath(), configFile.getAbsolutePath(), configFile.lastModified());
		}else{
			return null;
		}
	}
}