/*

 SourceDirectory.java

 Copyright 2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Created on 2007/01/11

 */
package net.sqs2.exigrid.source;

import java.io.File;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

import net.sqs2.exigrid.master.PageMaster;

public class SourceDirectory {
	private File sourceDirectoryRoot;
	private File sourceDirectory;
	private PageTaskConfig pageTaskConfig;

	private Map<File,List<PageID>> pageIDListMap = null;

	public SourceDirectory(File sourceDirectoryRoot){
		this(sourceDirectoryRoot, sourceDirectoryRoot);
	}

	public SourceDirectory(File sourceDirectoryRoot, File sourceDirectory){
		this.sourceDirectoryRoot = sourceDirectoryRoot;
		this.sourceDirectory = sourceDirectory;
	}

	public boolean isRoot(){
		return this.sourceDirectory.equals(this.sourceDirectoryRoot);
	}

	public SourceDirectory createParentSourceDirectory(){
		return new SourceDirectory(this.sourceDirectoryRoot, this.sourceDirectory.getParentFile());//, this.isRecursive
	}

	private void initPageIDListMap(){
		this.pageIDListMap = PageIDListMapFactory.createPageIDListMap(this.sourceDirectoryRoot, this.sourceDirectory); //, this.isRecursive
	}

	public File getDirectoryRoot(){
		return this.sourceDirectoryRoot;
	}

	public String getPath(){
		try{
			return this.sourceDirectory.getAbsolutePath().substring(this.sourceDirectoryRoot.getAbsolutePath().length());
		}catch(StringIndexOutOfBoundsException ex){
			Logger.getLogger("SourceDirectory").severe("source "+this.sourceDirectory.getAbsolutePath());
			Logger.getLogger("SourceDirectory").severe("root "+this.sourceDirectoryRoot.getAbsolutePath());
			throw ex;
		}
	}

	public String getAbsolutePath(){
		return this.sourceDirectory.getAbsolutePath();
	}

	public File getDirectory(){
		return this.sourceDirectory;
	}

	void setPageTaskConfig(PageTaskConfig pageTaskConfig){
		this.pageTaskConfig = pageTaskConfig;
	}

	PageTaskConfig getPageTaskConfig(){
		return this.pageTaskConfig;
	}

	public Map<File,List<PageID>> getPageIDListMap(){
		if(this.pageIDListMap == null){
			initPageIDListMap();
		}
		return this.pageIDListMap;
	}
	
	public List<PageID> getPageIDList(){
		return getPageIDListMap().get(this.sourceDirectory.getAbsoluteFile());
	}

	public int size(){
		int num = 0;
		for(List<PageID> pageIDList: getPageIDListMap().values()){
			num += pageIDList.size();
		}
		return num;
	}

	void close(){
		this.pageIDListMap.clear();
		this.pageIDListMap = null;
	}

	public PageTask createPageTask(PageMaster master, int pageNumber, PageID pageID, long sessionID){
		try{
			return new PageTaskImpl(this.sourceDirectoryRoot.getAbsolutePath(),
					master.getFileResource(),
					this.pageTaskConfig.getFileResource(),
					pageNumber, 
					pageID,
					sessionID);
		}catch(Exception ex){
			ex.printStackTrace();
			return null;
		}
	}

	@Override
	public String toString(){
		return this.sourceDirectory.getAbsolutePath();
	}

	@Override
	public int hashCode(){
		return this.sourceDirectory.hashCode();
	}

	@Override
	public boolean equals(Object o){
		if(o == this){
			return true;
		}else if(o instanceof SourceDirectory){
			SourceDirectory sourceDirectory = (SourceDirectory) o;
			return sourceDirectory.getDirectory().equals(this.getDirectory());
		}else{
			return false;
		}
	}
}
