/*

 SessionSourceImpl.java

 Copyright 2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Created on 2007/01/11

 */
package net.sqs2.exigrid.source;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections15.map.LRUMap;
import org.apache.commons.io.output.ByteArrayOutputStream;

import net.sqs2.exigrid.master.PageMaster;
import net.sqs2.exigrid.master.PageMasterFactory;
import net.sqs2.util.FileResource;
import net.sqs2.util.FileUtil;

public class SessionSource implements Serializable{

	private static SessionSource singleton;
	
	public static final int SESSION_SOURCE_IMAGE_BYTEARRAY_CACHE_SIZE = 16;// size

	private static final long serialVersionUID = 0L;

	private File sourceDirectoryRoot;
	private File resultDirectoryRoot;

	transient private SourceDirectoryScanner sourceDirectoryScanner = null;
	transient private PageMasterFactory pageMasterFactory = null;
	transient private PageTaskConfigFactory pageTaskConfigFactory = null;

	private Map<FileResource, PageTaskConfig> pageTaskConfigCache = new HashMap<FileResource, PageTaskConfig>();
	private Map<String,byte[]> imageByteArrayCache = new LRUMap<String,byte[]>(SESSION_SOURCE_IMAGE_BYTEARRAY_CACHE_SIZE);

	private long omrSessionID = System.currentTimeMillis();

	public SessionSource(File sourceDirectoryRoot, File resultDirectoryRoot, 
			PageMasterFactory pageMasterFactory, PageTaskConfigFactory pageTaskConfigFactory)throws PageTaskException, IOException{
		this.sourceDirectoryRoot = sourceDirectoryRoot;
		this.resultDirectoryRoot = resultDirectoryRoot;
		this.pageMasterFactory = pageMasterFactory;
		this.pageTaskConfigFactory = pageTaskConfigFactory;
		singleton = this;
	}

	public static SessionSource getInstance(){
		return singleton;
	}
	
	public void scanSourceDirectory(SourceDirectoryScannerMonitor monitor)throws IOException{ //boolean recursive, 
		this.sourceDirectoryScanner = new SourceDirectoryScanner(this.sourceDirectoryRoot, this.resultDirectoryRoot, monitor, 
				this.pageMasterFactory, this.pageTaskConfigFactory); //, recursive
	}

	public File getSourceDirectoryRoot(){
		return this.sourceDirectoryRoot;
	}

	public File getResultDirectoryRoot(){
		return this.resultDirectoryRoot;
	}

	public long getSessionID(){
		return this.omrSessionID;
	}

	public Set<PageMaster> getPageMasterSet(){
		if(this.sourceDirectoryScanner != null){
			return this.sourceDirectoryScanner.getSourceDirectoryListMap().keySet();
		}else{
			return null;
		}
	}

	public ArrayList<SourceDirectory> getSourceDirectoryList(PageMaster master){ 
		return this.sourceDirectoryScanner.getSourceDirectoryListMap().get(master);
	}

	public ArrayList<SourceDirectory> createSourceDirectoryList(PageMaster master){
		ArrayList<SourceDirectory> set = getSourceDirectoryList(master);
		ArrayList<SourceDirectory> list = new ArrayList<SourceDirectory>(set.size());
		for(SourceDirectory d: set){
			list.add(d);
		}
		return list;
	}	

	public PageMaster getPageMaster(String relativePath){
		return this.sourceDirectoryScanner.getPageMaster(relativePath);
	}

	public PageMaster getPageMaster(int masterIndex){
		return this.sourceDirectoryScanner.getPageMaster(masterIndex);
	}

	public PageTaskConfig getPageTaskConfig(FileResource pathWithLastModified){
		PageTaskConfig pageTaskConfig = this.pageTaskConfigCache.get(pathWithLastModified);
		if(pageTaskConfig != null){
			return pageTaskConfig;
		}else{
			return this.pageTaskConfigFactory.getDefaultInstance();
		}
	}

	public byte[] getImageByteArray(String relativePath, long lastModified)throws IOException{
		byte[] bytes;
		synchronized(this.imageByteArrayCache){
			bytes = (byte[])this.imageByteArrayCache.get(relativePath);
			if (bytes == null) {
				bytes = createByteArray(relativePath);
				this.imageByteArrayCache.put(relativePath, bytes);
			}
		}
		if (getLastModified(relativePath) != lastModified) {
			return null;
		}
		return bytes;
	}

	private byte[] createByteArray(String relativePath) throws IOException{
		BufferedInputStream in = null;
		ByteArrayOutputStream out = null;
		try{
			File file = new File(this.sourceDirectoryRoot.getAbsolutePath(), relativePath);
			in = new BufferedInputStream(new FileInputStream(file));
			out = new ByteArrayOutputStream();
			FileUtil.connect(in, out);
		}finally{
			if(in != null){
				in.close();
			}
			if(out != null){
				out.close();
			}
		}
		return out.toByteArray();
	}

	public long getLastModified(String relativePath) {
		File file = new File(this.sourceDirectoryRoot.getAbsolutePath(), relativePath);
		return file.lastModified();
	}

}
