/*

 PageTaskImpl.java

 Copyright 2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Created on 2007/01/11

 */
package net.sqs2.exigrid.source;

import java.io.Serializable;
import java.util.Comparator;
import java.util.concurrent.Delayed;
import java.util.concurrent.TimeUnit;

import net.sqs2.util.FileResource;

public class PageTaskImpl implements PageTask, Serializable{

	private static final long serialVersionUID = 5L;

	public static final int PAGETASK_LEASE_TIMEOUT_IN_SEC = 31;// sec

	String id;

	String sourceDirectoryRootPath;

	FileResource master;
	FileResource pageTaskConfig;
	int pageNumber;

	PageID pageID;

	long expiredTime = 0L;
	long omrSessionID = 0L;

	PageTaskResult pageTaskResult = null;

	PageTaskError pageTaskError = null;

	public PageTaskImpl(){}

	public PageTaskImpl(String sourceDirectoryRootPath, 
			FileResource master,
			FileResource configSource,
			int pageNumber,
			PageID pageID,
			long omrSessionID) {
		this.sourceDirectoryRootPath = sourceDirectoryRootPath;
		this.master = master;
		this.pageTaskConfig = configSource;
		this.pageNumber = pageNumber;
		this.pageID = pageID;
		this.omrSessionID = omrSessionID;
		this.id = createID(master, pageNumber, pageID);
	}

	public static String createID(FileResource master, int pageNumber, PageID pageID){
		return master+"#"+pageNumber+"\t"+pageID.createID();
	}

	public String getID(){
		return this.id;
	}

	public String getSourceDirectoryRootPath(){
		return this.sourceDirectoryRootPath;
	}

	public String getMasterPath(){
		return this.master.getPath();
	}

	public long getMasterLastModified(){
		return this.master.getLastModified();
	}

	public int getPageNumber(){
		return this.pageNumber;
	}

	public PageID getPageID(){
		return this.pageID;
	}

	public String getPageTaskConfigPath(){
		return this.pageTaskConfig.getPath();
	}

	public long getPageTaskConfigLastModified(){
		return this.pageTaskConfig.getLastModified();
	}

	public PageTaskResult getPageTaskResult(){
		return this.pageTaskResult;
	}

	public void setResult(PageTaskResult resultData){
		this.pageTaskResult = resultData;
	}

	public long getOMRSessionID(){
		return this.omrSessionID;
	}

	public String toString(){
		return this.id;
	}

	public boolean equals(Object o){
		try{
			PageTaskImpl pageTaskSource = (PageTaskImpl) o; 
			return this.id.equals(pageTaskSource.id);
		}catch(ClassCastException ignore){
		}
		return false;
	}

	public void setLeased(){
		this.expiredTime = System.currentTimeMillis() + PAGETASK_LEASE_TIMEOUT_IN_SEC * 1000;
	}

	public long getDelay(TimeUnit unit){
		return unit.convert(this.expiredTime - System.currentTimeMillis(), TimeUnit.MILLISECONDS);
	}

	public int compareTo(Delayed o) {
		try{
			PageTaskImpl pageTaskSource = (PageTaskImpl) o;
			int diff = 0;
			if(this.id.equals(pageTaskSource.id)){
				return 0;
			}
			if((diff = this.master.compareTo(pageTaskSource.master)) != 0){
				return diff; 
			}
			if((diff = this.pageTaskConfig.compareTo(pageTaskSource.pageTaskConfig)) != 0){
				return diff;
			}
			if((diff = this.pageNumber - pageTaskSource.pageNumber) != 0){
				return diff;
			}
			if((diff = this.pageID.compareTo(pageTaskSource.pageID)) != 0){
				return diff;
			}
		}catch(ClassCastException ignore){
		}
		return 1;
	}

	public PageTaskError getPageTaskError(){
		return this.pageTaskError;
	}

	public void setPageTaskError(PageTaskError pageTaskError){
		this.pageTaskError = pageTaskError;
	}

	public int hashCode(){
		return (int) this.id.hashCode();
	}

	public static class PageTaskComparator implements Comparator<PageTaskImpl>{
		public int compare(PageTaskImpl a, PageTaskImpl b){
			int diff = a.getPageID().getPath().compareTo(b.getPageID().getPath());
			if(diff != 0){
				return diff;
			}
			diff = a.getPageID().getIndex() - b.getPageID().getIndex();
			if(diff != 0){
				return diff;
			}
			diff = a.getMasterPath().compareTo(b.getMasterPath());
			if(diff != 0){
				return diff;
			}
			diff = a.getPageTaskConfigPath().compareTo(b.getPageTaskConfigPath());
			if(diff != 0){
				return diff;
			}else{
				return (int)(a.pageTaskConfig.getLastModified() - b.pageTaskConfig.getLastModified());
			}
		}
	}
}
