/*

 PageIDListFactory.java

 Copyright 2004-2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Created on 2007/2/17

 */
package net.sqs2.exigrid.source;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import net.sqs2.exigrid.engine.ExigridConstants;
import net.sqs2.image.ImageFactory;
import net.sqs2.image.ImageUtil;

public class PageIDListMapFactory {

	public static Map<File,List<PageID>> createPageIDMap(SourceDirectory sourceDirectory){
		return createPageIDListMap(sourceDirectory.getDirectoryRoot(), sourceDirectory.getDirectory());
	}
	
	private static class PageIDCrowlerContext{
		private File sourceDirectory;
		private int size;
		private Map<File,List<PageID>> pageIDListMap = null;
		private List<PageID> subPageIDList = null;
		
		PageIDCrowlerContext(File sourceDirectory, int size){
			this.sourceDirectory = sourceDirectory;
			this.size = size;
		}

		void addAll(List<PageID> subPageIDList){
			if(this.pageIDListMap == null){
				this.pageIDListMap = new LinkedHashMap<File,List<PageID>>();
			}
			this.subPageIDList = this.pageIDListMap.get(this.sourceDirectory);
			if(this.subPageIDList == null){
				this.subPageIDList = new ArrayList<PageID>(this.size);
				this.pageIDListMap.put(this.sourceDirectory, this.subPageIDList);
			}
			this.subPageIDList.addAll(subPageIDList);
		}
		
		void putAll(Map<File,List<PageID>> subPageIDListMap){
			if(this.pageIDListMap == null){
				this.pageIDListMap = new LinkedHashMap<File,List<PageID>>();
			}
			this.pageIDListMap.putAll(subPageIDListMap);
		}
		
		Map<File,List<PageID>> getPageIDListMap(){
			return this.pageIDListMap;
		}
	}

	public static Map<File,List<PageID>> createPageIDListMap(File sourceDirectoryRoot, File sourceDirectory){
		
		boolean isEmptyDirectory = true;
		
		File[] files = sourceDirectory.listFiles();
		Arrays.sort(files);

		PageIDCrowlerContext context = new PageIDCrowlerContext(sourceDirectory, files.length); 

		for(File file: files){
			try{
				if(! file.isDirectory()){
					List<PageID> subPageIDList = PageIDListMapFactory.initPageIDList(sourceDirectoryRoot, file);
					if(subPageIDList != null){
						context.addAll(subPageIDList);
						isEmptyDirectory = false;
					}
				}else if(! file.getName().endsWith(ExigridConstants.RESULT_FOLDER_SUFFIX)){
					Map<File,List<PageID>> pageIDListSubMap = createPageIDListMap(sourceDirectoryRoot, file);
					if(pageIDListSubMap != null){
						context.putAll(pageIDListSubMap);
						isEmptyDirectory = false;
					}
				}
			}catch(IOException ignore){}
		}
		
		if(isEmptyDirectory){
			return null;
		}
		
		return context.getPageIDListMap();
	}

	private static List<PageID> initPageIDList(File sourceDirectoryRoot, File file)throws IOException{
		if(! ImageUtil.isSupported(file.getName())){
			return null;
		}
		List<PageID> ret = new ArrayList<PageID>(1);
		int numPages = ImageFactory.getNumPages(file);
		String path = file.getAbsolutePath().substring(sourceDirectoryRoot.getAbsolutePath().length() + 1);
		if(numPages == 1){
			ret.add(new PageID(path, file.lastModified(), 0, 1));
		}else if(1 < numPages){
			for(int i = 0; i < numPages; i++){
				ret.add(new PageID(path, file.lastModified(), i, numPages));
			}
		}
		return ret;
	}
}
