/**
 *  SessionThreadManager.java

 Copyright 2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Created on 2007/01/31
 Author hiroya
 */

package net.sqs2.exigrid.session;

import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import net.sqs2.exigrid.source.PageTask;
import net.sqs2.lang.GroupThreadFactory;

public class SessionThreadManager{

	public static final int SESSION_EXECUORS_THREAD_PRIORITY = Thread.NORM_PRIORITY;
	public static final int PAGETASK_PRODUCER_EXEC_INTERVAL_IN_MILLIS = 5000;// msec
	public static final int PAGETASK_CONSUMER_EXEC_INTERVAL_IN_MILLIS = 5;// msec
	public static final int RECYCLE_INTERVAL = 3; //sec

	private GroupThreadFactory groupThreadFactory;

	private ScheduledExecutorService pageTaskRecycleService;
	private ScheduledExecutorService sourceProducerService;
	private ScheduledExecutorService recycleExecutorService;
	private ScheduledExecutorService resultExecutorService;

	private Future<?> sourceProducerFuture;
	private Future<?> remoteRecycleExecutorFuture;
	private Future<?> resultExecutorFuture;
	private Future<?> pageTaskRecycleFuture;

	SessionThreadManager(){
		this.groupThreadFactory = new GroupThreadFactory("net.sqs2.exigrid.session.SessionThreadManager", SESSION_EXECUORS_THREAD_PRIORITY, true);
		this.pageTaskRecycleService = Executors.newScheduledThreadPool(1, this.groupThreadFactory);
		this.sourceProducerService = Executors.newSingleThreadScheduledExecutor(this.groupThreadFactory);
		this.recycleExecutorService = Executors.newScheduledThreadPool(1, this.groupThreadFactory);
		this.resultExecutorService = Executors.newScheduledThreadPool(1, this.groupThreadFactory);
	}

	void startPageTaskRecycleThread(final PageTaskHolder pageTaskHolder) {		
		this.remoteRecycleExecutorFuture = this.pageTaskRecycleService.scheduleWithFixedDelay(new Runnable(){
			public void run(){
				try{
					PageTask pageTask = pageTaskHolder.takeRemoteLeasedPageTask();
					pageTaskHolder.preparePageTask(pageTask);
				}catch(InterruptedException ignore){
				}
			}
		}, RECYCLE_INTERVAL, RECYCLE_INTERVAL, TimeUnit.SECONDS);
	}

	Future<?> startDynamicPageTaskSourceProducer(Runnable task){
		return (this.sourceProducerFuture = this.sourceProducerService.scheduleWithFixedDelay(task,
				0, 
				PAGETASK_PRODUCER_EXEC_INTERVAL_IN_MILLIS,
				TimeUnit.MILLISECONDS));
	}

	Future<?> startPageTaskConsumer(Runnable task){
		return (this.resultExecutorFuture = this.resultExecutorService.scheduleWithFixedDelay(task,
				0,
				PAGETASK_CONSUMER_EXEC_INTERVAL_IN_MILLIS,
				TimeUnit.MILLISECONDS));
	}

	void stop(){
		if(this.pageTaskRecycleFuture != null){
			this.pageTaskRecycleFuture.cancel(true);
		}
		if(this.sourceProducerFuture != null){
			this.sourceProducerFuture.cancel(true);
		}
		if(this.remoteRecycleExecutorFuture != null){
			this.remoteRecycleExecutorFuture.cancel(true);
		}
		if(this.resultExecutorFuture != null){
			this.resultExecutorFuture.cancel(true);
		}
	}

	void shutdown(){
		this.pageTaskRecycleService.shutdown();
		this.sourceProducerService.shutdown();
		this.recycleExecutorService.shutdown();
		this.resultExecutorService.shutdown();
	}
}
