/*

 SessionServiceImpl.java

 Copyright 2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Created on 2007/01/11

 */
package net.sqs2.exigrid.session;

import java.io.IOException;
import java.io.Serializable;
import java.rmi.RemoteException;
import java.rmi.server.ServerNotActiveException;
import java.rmi.server.UnicastRemoteObject;
import java.util.Collections;
import java.util.Map;
import java.util.TreeMap;
import java.util.logging.Logger;

import net.sqs2.exigrid.master.PageMaster;
import net.sqs2.exigrid.source.PageTask;
import net.sqs2.exigrid.source.PageTaskConfig;
import net.sqs2.exigrid.source.SessionSource;
import net.sqs2.util.FileResource;

public class SessionServiceImpl extends UnicastRemoteObject implements SessionService,SessionServiceHandler,Serializable{
	private static final long serialVersionUID = 0L;

	transient private SessionSource sessionSource;
	transient private PageTaskHolder pageTaskHolder;
	
	private Map<String,ClientInfo> clientInfoMap = null;
	
	/*
	private Map<String,Integer> leaseCountMap = Collections.synchronizedSortedMap(new TreeMap<String,Integer>());
	private Map<String,Integer> submitCountMap = Collections.synchronizedSortedMap(new TreeMap<String,Integer>());	
	 */
	class ClientInfo{
		String host;
		int numLease = 0;
		int numSubmit = 0;
		ClientInfo(){
		}
		public int getNumLease() {
        	return numLease;
        }
		public int getNumSubmit() {
        	return numSubmit;
        }
		public void incrementNumLease(){
			numLease++;
		}
		public void incrementNumSubmit(){
			numSubmit++;
		}
	}
	
	private long key;
	
	SessionServiceImpl(long key, SessionSource sessionSource, PageTaskHolder pageTaskHolder) throws RemoteException {
		super();
		Logger.getLogger("session").info("Start Service:"+key);
		this.key = key;
		this.sessionSource = sessionSource;
		this.pageTaskHolder = pageTaskHolder;
		this.clientInfoMap = Collections.synchronizedSortedMap(new TreeMap<String,ClientInfo>());	
	}

	public long ping(long key) throws RemoteException{
		checkKey(key);
		return System.currentTimeMillis();	
	}

	private boolean checkKey(long key) throws RemoteException{
		if(this.key != key){
			throw new RemoteException("Invalid Key");
		}
		return true;
	}

	public PageTask leasePageTask(long key) throws RemoteException {
		checkKey(key);
		try {
			String clientHost = null;
			try {
				clientHost = UnicastRemoteObject.getClientHost();
				ClientInfo clientInfo = getClientInfo(clientHost);
				clientInfo.incrementNumLease();
			} catch (ServerNotActiveException ex) {
			}
			PageTask pageTask = this.pageTaskHolder.leasePageTask(); // block
			if(pageTask != null){
				pageTask.setLeased();
				if(clientHost == null){
					this.pageTaskHolder.leaseLocal(pageTask);
				}else{
					this.pageTaskHolder.leaseRemote(pageTask);
				}
			}
			return pageTask;
		} catch (InterruptedException ex) {
			//throw new RemoteException(ex.getClass().getName() + ":" + ex.getMessage());
			return null;
		}
	}

	public void submitPageTask(long key, PageTask pageTask)throws RemoteException {
		checkKey(key);
		try {
			String clientHost = UnicastRemoteObject.getClientHost();
			ClientInfo clientInfo = getClientInfo(clientHost);
			clientInfo.incrementNumSubmit();
		} catch (ServerNotActiveException ex) {						
		}
		this.pageTaskHolder.submitPageTask(pageTask);
	}

	private ClientInfo getClientInfo(String clientHost) {
	    ClientInfo clientInfo = this.clientInfoMap.get(clientHost);
	    if(clientInfo == null){
	    	clientInfo = new ClientInfo() ;
	    	this.clientInfoMap.put(clientHost, clientInfo);
	    }
	    return clientInfo;
    }

	public PageMaster getMaster(long key, String relativePath, long lastModified)throws RemoteException{
		checkKey(key);
		return this.sessionSource.getPageMaster(relativePath);
	}

	public PageTaskConfig getPageTaskConfig(long key, String relativePath, long lastModified) throws RemoteException{
		checkKey(key);
		return this.sessionSource.getPageTaskConfig(new FileResource(relativePath, lastModified));
	}

	public byte[] getImageByteArray(long key, String relativePath, long lastModified) throws RemoteException {
		checkKey(key);
		try {
			return this.sessionSource.getImageByteArray(relativePath, lastModified);
		} catch (IOException ex) {
			throw new RemoteException(ex.getMessage());
		}
	}

	public long getLastModified(long key, String relativePath) throws RemoteException{
		checkKey(key);
		return this.sessionSource.getLastModified(relativePath);
	}

	public int getNumRemoteSlaveExecutors()throws RemoteException{
		return this.clientInfoMap.size();
	}

	/*
	public void removeRemoteSlaveExecutor(String clientHost)throws RemoteException{
		this.clientInfoMap.remove(clientHost);
	}*/

	public SessionSource getSessionSource(){
		return this.sessionSource; 
	}
}
