/**
 *  PageTaskProducer.java

 Copyright 2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Created on 2007/01/31
 Author hiroya
 */
package net.sqs2.exigrid.session;

import java.io.File;
import java.io.IOException;
import java.util.Calendar;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

import net.sqs2.exigrid.master.PageMaster;
import net.sqs2.exigrid.source.PageID;
import net.sqs2.exigrid.source.PageTask;
import net.sqs2.exigrid.source.PageTaskAccessor;
import net.sqs2.exigrid.source.SessionSource;
import net.sqs2.exigrid.source.SourceDirectory;

public class PageTaskProducer implements Runnable{

	private AbstractSession monitor = null;

	private File sourceDirectoryRoot;
	private File resultDirectoryRoot;

	private PageTaskAccessor pageTaskAccessor;
	private long newFileIgnoreThreshold;
	private AbstractSession sessionManager;

	PageTaskProducer(AbstractSession sessionManager,
			File sourceDirectoryRoot, File resultDirectoryRoot, long newFileIgnoreThreshold)throws IOException{
		this.sessionManager = sessionManager;
		this.sourceDirectoryRoot = sourceDirectoryRoot; 
		this.resultDirectoryRoot = resultDirectoryRoot; 
		this.pageTaskAccessor = new PageTaskAccessor(this.sourceDirectoryRoot, this.resultDirectoryRoot);
		this.newFileIgnoreThreshold = newFileIgnoreThreshold;
	}

	public void setMonitor(AbstractSession monitor){
		this.monitor = monitor;
	}

	public void run() {

		if(this.sessionManager.isNoPageTask()){
			long start = System.currentTimeMillis();

			try{
				producePageTasks();
			}catch(PageTaskStopException ignore){
			}

			long end = System.currentTimeMillis();
			try{
				Thread.sleep(max(10000 , (end - start) * 10));
			}catch(InterruptedException ignore){}
		}else{
			return;
		}

	}

	static long max(long a, long b){
		return (a<=b)?b:a;
	}

	private class PageAddCounter{
		int numAdded = 0;
		int numReused = 0; 
		int numRetry = 0; 
	}

	private boolean producePageTasks()throws PageTaskStopException{
		SessionSource sessionSource = this.sessionManager.getSessionServiceHandler().getSessionSource();

		this.sessionManager.setNowProducingPageTasks(true);

		PageAddCounter counter = new PageAddCounter();
		try{

			sessionSource.scanSourceDirectory(monitor);
			for(PageMaster master: sessionSource.getPageMasterSet()){
				int numPages = master.getNumPages();
				for(SourceDirectory sourceDirectory: sessionSource.getSourceDirectoryList(master)){
					Map<File,List<PageID>> pageIDListMap = sourceDirectory.getPageIDListMap();
					int pageIDIndex = 0;
					if(pageIDListMap == null){
						continue;
					}
					for(List<PageID> pageIDList: pageIDListMap.values()){
						for(PageID pageID: pageIDList){					    				   
							int pageNumber = pageIDIndex % numPages + 1;
							PageTask pageTask = sourceDirectory.createPageTask(master, pageNumber, pageID, sessionSource.getSessionID());
							producePageTask(sessionSource, pageTask, counter);
							pageIDIndex++;
						}
					}
				}
			}
			this.sessionManager.getPageTaskHolder().incrementNumTargetPages(counter.numAdded + counter.numRetry);

			StringBuilder sb = new StringBuilder(64);
			sb.append("\nnumReused = " + counter.numReused);
			sb.append("\nnumAdded = " + counter.numAdded);
			sb.append("\nnumRetry = " + counter.numRetry);
			Logger.getLogger("session").info("PageTaskProducer\n\t" + sb.toString());	

		}catch(PageTaskStopException ignore){
		}catch(Exception ex){
			ex.printStackTrace();
		}

		this.sessionManager.setNowProducingPageTasks(false);

		return 0 < (counter.numAdded + counter.numRetry);
	}

	static final boolean INFO = true; 
	private void producePageTask(SessionSource sessionSource, PageTask pageTask, PageAddCounter counter)throws PageTaskStopException{

		if(sessionManager.isStopped()){
			throw new PageTaskStopException();
		}

		if(cancelProduceTask(pageTask)){
			return;
		}

		//try{
			PageTask storedPageTask = null;
			try{
				storedPageTask = (PageTask)this.pageTaskAccessor.get(pageTask.toString());
			}catch(Exception ignore){
				//ex.printStackTrace();
			}
			if(isOnceExecutedPageTask(storedPageTask)){
				if(hasError(storedPageTask)){
					//Logger.getLogger("PageTaskProducer").info("hasError");
					if(! isExecutionRequiredPageTaskWithError(sessionSource, pageTask, counter, storedPageTask)){
						return;
					}
				}else if(storedPageTask.getPageTaskResult() != null){
					if(! isExecutionRequiredPageTask(sessionSource, pageTask, counter, storedPageTask)){						
						return;
					}
				}else{
					execNewTask(pageTask, counter);	
				}
			}else{
				execNewTask(pageTask, counter);	
			}
			this.pageTaskAccessor.put(pageTask);
			this.sessionManager.getPageTaskHolder().preparePageTask(pageTask);
			this.monitor.notifyProducePageTask(pageTask);
			/*
		}catch(Exception ex){
			ex.printStackTrace();
			throw new RuntimeException(ex);
		}finally{
		}*/
	}

	private boolean isOnceExecutedPageTask(PageTask storedPageTask) {
		return storedPageTask != null;
	}

	private boolean cancelProduceTask(PageTask pageTask) {
		if(isConcurrentFileModificationSuspected(pageTask)){
			if(INFO){
				Logger.getLogger("source").info("IGNORE\t"+pageTask);
			}
			return true;
		}
		if(isPrepared(pageTask)){
			if(INFO){
				Logger.getLogger("source").info("PREPARED\t"+pageTask);
			}
			return true;
		}
		if(isLeased(pageTask)){
			if(INFO){
				Logger.getLogger("source").info("LEASED\t"+pageTask);
			}
			return true;
		}
		return false;
	}

	private boolean isExecutionRequiredPageTaskWithError(SessionSource sessionSource, PageTask pageTask, PageAddCounter counter, PageTask storedPageTask) {
		if(sessionSource.getSessionID() == storedPageTask.getOMRSessionID()){
			if(INFO){
				Logger.getLogger("source").info("IGNORE ERROR\t" + pageTask);
			}
			return false;
		}else{
			counter.numRetry++;
			if(INFO){
				Logger.getLogger("source").info("==========RETRY ERROR\t" + pageTask);
			}
			return true;
		}
	}

	private boolean isExecutionRequiredPageTask(SessionSource sessionSource, PageTask pageTask, PageAddCounter counter, PageTask storedPageTask) {
		if(storedPageTask.getOMRSessionID() == sessionSource.getSessionID()){
			if(INFO){
				//Logger.getLogger("source").info("IGNORE\t" + pageTask);
			}
			return false;
		}else{
			if(INFO){
				//Logger.getLogger("source").info("REUSE\t" + pageTask);
			}
			counter.numReused++;
			this.sessionManager.getPageTaskHolder().setNumReusedPages(counter.numReused);
			return false;
		}
	}

	private void execNewTask(PageTask pageTask, PageAddCounter counter) {
		counter.numAdded++;
		Logger.getLogger("session").info("==========ADD\t" + pageTask);
	}

	private boolean isPrepared(PageTask pageTask){
		return this.sessionManager.getPageTaskHolder().isPrepared(pageTask);
	}

	private boolean isLeased(PageTask pageTask){
		return this.sessionManager.getPageTaskHolder().isLeased(pageTask);
	}

	private boolean hasError(PageTask pageTask){
		return pageTask.getPageTaskError() != null;
	}

	private boolean isConcurrentFileModificationSuspected(PageTask pageTask){
		return (Calendar.getInstance().getTimeInMillis() - this.newFileIgnoreThreshold) < pageTask.getPageID().getLastModified();
	}

}