/**
 *  PageTaskHolder.java

 Copyright 2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Created on 2007/01/31
 Author hiroya
 */
package net.sqs2.exigrid.session;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;

import net.sqs2.exigrid.source.PageTask;

public class PageTaskHolder implements Externalizable{
	private static final long serialVersionUID = 0L;

	private int numTargetPages;
	private int numReusedPages;	
	private int numExternalizedPages;

	transient private BlockingQueue<PageTask> preparedPageTaskQueue = null;
	transient private Set<PageTask> preparedPageTaskSet = null;
	transient private DelayQueue<PageTask> localLeasedPageTaskQueue = null;
	transient private DelayQueue<PageTask> remoteLeasedPageTaskQueue = null;
	transient private BlockingQueue<PageTask> submittedPageTaskQueue = null;
	transient private Map<Integer,String> errorMap = null;
	transient private int externalizedPageIndex = 0;

	public PageTaskHolder(){
		initQueue();
	}
	
	private void initQueue(){
		this.preparedPageTaskQueue = new LinkedBlockingQueue<PageTask>();
		this.preparedPageTaskSet = Collections.synchronizedSet(new HashSet<PageTask>());
		this.localLeasedPageTaskQueue = new DelayQueue<PageTask>();
		this.remoteLeasedPageTaskQueue = new DelayQueue<PageTask>();
		this.submittedPageTaskQueue = new LinkedBlockingQueue<PageTask>();
		this.errorMap = new LinkedHashMap<Integer, String>();
	}
	
	public void readExternal(ObjectInput in) throws IOException, java.lang.ClassNotFoundException{
		initQueue();
    }

	public void writeExternal(ObjectOutput in) throws IOException{
		// do nothing
    }

	/* (non-Javadoc)
	 * @see net.sf.sqs_xml.omr.session.PageTaskHolder#getNumTargetPages()
	 */
	public int getNumTargetPages(){
		return this.numTargetPages;
	}

	public void incrementNumTargetPages(int numTargetPages){
		this.numTargetPages += numTargetPages;
	}

	/* (non-Javadoc)
	 * @see net.sf.sqs_xml.omr.session.PageTaskHolder#getNumTotalPages()
	 */
	public int getNumTotalPages(){
		return getNumTargetPages(); // + getNumReusedPages()
	}

	/* (non-Javadoc)
	 * @see net.sf.sqs_xml.omr.session.PageTaskHolder#getNumReusedPages()
	 */
	public int getNumReusedPages(){
		return this.numReusedPages;
	}

	public void setNumReusedPages(int numReusedPages){
		this.numReusedPages = numReusedPages;
	}

	/* (non-Javadoc)
	 * @see net.sf.sqs_xml.omr.session.PageTaskHolder#getNumPreparedPages()
	 */
	public int getNumPreparedPages(){
		return this.preparedPageTaskSet.size(); 
	}

	/* (non-Javadoc)
	 * @see net.sf.sqs_xml.omr.session.PageTaskHolder#getNumLocalLeasedPages()
	 */
	public int getNumLocalLeasedPages(){
		return this.localLeasedPageTaskQueue.size();
	}

	/* (non-Javadoc)
	 * @see net.sf.sqs_xml.omr.session.PageTaskHolder#getNumRemoteLeasedPages()
	 */
	public int getNumRemoteLeasedPages(){
		return this.remoteLeasedPageTaskQueue.size();
	}

	/* (non-Javadoc)
	 * @see net.sf.sqs_xml.omr.session.PageTaskHolder#getNumSubmittedPages()
	 */
	public int getNumSubmittedPages(){
		return this.submittedPageTaskQueue.size();
	}

	/* (non-Javadoc)
	 * @see net.sf.sqs_xml.omr.session.PageTaskHolder#getNumErrorPages()
	 */
	public int getNumErrorPages(){
		return this.errorMap.size();
	}

	public int incrementNumErrorPages(PageTask pageTask){
		this.errorMap.put(this.externalizedPageIndex, pageTask.getPageID().getPath() +"="+pageTask.getPageTaskError().getDescription());
		this.externalizedPageIndex++;
		return this.errorMap.size();
	}

	/* (non-Javadoc)
	 * @see net.sf.sqs_xml.omr.session.PageTaskHolder#getNumExternalizedPages()
	 */
	public int getNumExternalizedPages(){
		return this.numExternalizedPages;
	}

	public int incrementNumExternalizedPages(){		
		this.externalizedPageIndex++;
		return this.numExternalizedPages++;		
	}

	void setNumExternalizedPages(int numExternalizedPages){
		this.numExternalizedPages = numExternalizedPages;
		//update();		
	}

	/* (non-Javadoc)
	 * @see net.sf.sqs_xml.omr.session.PageTaskHolder#start()
	 */
	public void reset(){
		this.numReusedPages = 0;
		this.numTargetPages = 0;
		this.numExternalizedPages = 0;
		this.externalizedPageIndex = 0;
		this.preparedPageTaskQueue.clear();
		this.preparedPageTaskSet.clear();
		this.localLeasedPageTaskQueue.clear();
		this.remoteLeasedPageTaskQueue.clear();
		this.submittedPageTaskQueue.clear();
		this.errorMap.clear();
	}

	public void preparePageTask(PageTask pageTask){
		this.preparedPageTaskSet.add(pageTask);
		this.preparedPageTaskQueue.offer(pageTask);
	}

	public PageTask leasePageTask()throws InterruptedException{
		PageTask pageTask = this.preparedPageTaskQueue.take();
		this.preparedPageTaskSet.remove(pageTask);
		return pageTask;
	}

	public void leaseLocal(PageTask pageTask){
		this.localLeasedPageTaskQueue.add(pageTask);
	}

	public void leaseRemote(PageTask pageTask){
		this.remoteLeasedPageTaskQueue.add(pageTask);
	}

	public boolean isPrepared(PageTask pageTask){
		return this.preparedPageTaskSet.contains(pageTask);
	}

	public boolean isEmpty(){
		return this.preparedPageTaskQueue.isEmpty() &&
		this.localLeasedPageTaskQueue.isEmpty() &&
		this.remoteLeasedPageTaskQueue.isEmpty() && 
		this.submittedPageTaskQueue.isEmpty();
	}

	PageTask takeLocalLeasedPageTask()throws InterruptedException{
		return this.localLeasedPageTaskQueue.take();
	}

	public PageTask takeRemoteLeasedPageTask()throws InterruptedException{
		return this.remoteLeasedPageTaskQueue.take();
	}	

	public boolean isLeased(PageTask pageTask){
		return this.localLeasedPageTaskQueue.contains(pageTask) || this.remoteLeasedPageTaskQueue.contains(pageTask);
	}

	public PageTask submitPageTask(PageTask pageTask){
		if(this.localLeasedPageTaskQueue.remove(pageTask)){
			this.submittedPageTaskQueue.add(pageTask);
			return pageTask;
		}else if(this.remoteLeasedPageTaskQueue.remove(pageTask)){
			this.submittedPageTaskQueue.add(pageTask);
			return pageTask;
		}else{
			return null;
		}
	}

	public PageTask pollSubmittedPageTask(){
		try{
			PageTask pageTask = this.submittedPageTaskQueue.poll(1000, TimeUnit.MILLISECONDS);
			return pageTask;
		}catch(InterruptedException ignore){
			return null;
		}
	}

	public Set<Map.Entry<Integer,String>> getErrorMapEntrySet(){
		return this.errorMap.entrySet();
	}

}