/*
 * 

 AbstractSession.java

 Copyright 2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */
package net.sqs2.exigrid.session;

import java.awt.Toolkit;
import java.io.File;
import java.io.IOException;
import java.rmi.NoSuchObjectException;
import java.rmi.RemoteException;
import java.rmi.server.UnicastRemoteObject;
import java.util.ArrayList;
import java.util.logging.Logger;

import net.sqs2.cache.PersistentCacheManager;
import net.sqs2.exigrid.event.ResultEvent;
import net.sqs2.exigrid.event.ResultEventManager;
import net.sqs2.exigrid.event.SessionMonitor;
import net.sqs2.exigrid.event.PageTaskSourceProducerMonitor;
import net.sqs2.exigrid.event.ResultManagerMonitor;
import net.sqs2.exigrid.event.ResultEventMonitor;
import net.sqs2.exigrid.master.PageMaster;
import net.sqs2.exigrid.master.PageMasterAccessor;
import net.sqs2.exigrid.master.PageMasterFactory;
import net.sqs2.exigrid.source.PageTask;
import net.sqs2.exigrid.source.PageTaskAccessor;
import net.sqs2.exigrid.source.PageTaskConfig;
import net.sqs2.exigrid.source.PageTaskConfigFactory;
import net.sqs2.exigrid.source.PageTaskException;
import net.sqs2.exigrid.source.SessionSource;
import net.sqs2.exigrid.source.SourceDirectoryScannerMonitor;
import net.sqs2.net.RMIRegistryMulticastAdvertisingService;

public abstract class AbstractSession implements PageTaskSourceProducerMonitor,SourceDirectoryScannerMonitor,SessionMonitor,ResultEventMonitor{

	public static final int SESSION_SOURCE_NEWFILE_IGNORE_THRESHOLD_IN_SEC = 10;
	public static final int EXTERNALIZER_WAKEUP_DELAY_IN_SEC = 1;

	protected ArrayList<ResultManagerMonitor> monitors;
	protected ResultEventManager resultEventManager = null;

	private PageTaskHolder pageTaskHolder;
	private boolean isNowProducingPageTasks = false;
	private SessionServiceImpl sessionService = null;
	private SessionThreadManager sessionThreadManager;

	private PageMasterFactory pageMasterFactory;
	private PageTaskConfigFactory pageTaskConfigFactory;
	private ResultEventManagerFactory resultEventManagerFactory;

	private long key;
	private RMIRegistryMulticastAdvertisingService advertisingService;
	private String serviceName;

	public AbstractSession(
			PageMasterFactory pageMasterFactory, PageTaskConfigFactory pageTaskConfigFactory, ResultEventManagerFactory resultEventManagerFactory,
			long key,
			RMIRegistryMulticastAdvertisingService advertisingService, 
			String serviceName) {
		super();
		this.pageMasterFactory= pageMasterFactory;
		this.pageTaskConfigFactory = pageTaskConfigFactory;
		this.resultEventManagerFactory = resultEventManagerFactory;
		this.key = key;
		this.advertisingService = advertisingService;
		this.serviceName = serviceName;
		this.pageTaskHolder = new PageTaskHolder();
		this.monitors = new ArrayList<ResultManagerMonitor>(1);
	}

	public void addMonitor(ResultManagerMonitor monitor) {
		monitors.add(monitor);
	}

	public void removeMonitor(ResultManagerMonitor monitor) {
		monitors.remove(monitor);
	}

	public void deleteMonitors() {
		monitors.clear();
	}

	long getKey() {
		return key;
	}

	private void startFanfare(File sourceDirectoryRoot) {
		Logger.getLogger("session").info("*********** START ************");
		Logger.getLogger("session").info("sourceDirectoryRoot:"+sourceDirectoryRoot.getAbsolutePath());
	}

	protected void finishFanfare() {
		Logger.getLogger("session").info("********* FINISHED ***********");
		Toolkit.getDefaultToolkit().beep();
				
		/*
		MidiPlayer player = new MidiPlayer(0);
		player.noteOn(60, 2);
		player.noteOn(62, 2);
		player.noteOn(64, 2);
		 */
	}

	public boolean isNoPageTask() {
		return this.pageTaskHolder.isEmpty();
	}

	private void exportSessionService() {
		try{
			if(this.advertisingService != null){
				this.advertisingService.export(this.sessionService, this.serviceName);
			}
		}catch(Exception ex){
			ex.printStackTrace();
		}
	}

	public void setNowProducingPageTasks(boolean isNowProducingPageTasks) {
		this.isNowProducingPageTasks = isNowProducingPageTasks; 
	}

	public boolean isNowProducingPageTasks() {
		return this.isNowProducingPageTasks;
	}

	private void resetCacheAndResult(final File sourceDirectoryRoot, final File resultDirectoryRoot) {
		try{
			PageMasterAccessor m = new PageMasterAccessor(sourceDirectoryRoot, resultDirectoryRoot);
			if(m != null){
				m.removeAll();
			}
		}catch(IOException ignore){}
		try{
			PageTaskAccessor p = new PageTaskAccessor(sourceDirectoryRoot, resultDirectoryRoot);
			if(p != null){
				p.removeAll();
			}
		}catch(IOException ignore){}

		//this.exteralizationManager.resetCacheAndResult(sourceDirectoryRoot, resultDirectoryRoot);

		//FileUtil.deleteDirectory(resultDirectoryRoot);
		//createresultDirectoryRoot(sourceDirectoryRoot);
	}

	public synchronized void start(File sourceDirectoryRoot, File resultDirectoryRoot, boolean doReset) throws PageTaskException {
		this.isStopped = false;
		if(doReset){
			resetCacheAndResult(sourceDirectoryRoot, resultDirectoryRoot);
		}

		startFanfare(resultDirectoryRoot);

		this.sessionThreadManager = new SessionThreadManager();
		this.sessionThreadManager.startPageTaskRecycleThread(this.pageTaskHolder);

		try {

			this.pageTaskHolder.reset();

			SessionSource sessionSource = new SessionSource(sourceDirectoryRoot, resultDirectoryRoot, 
					this.pageMasterFactory, this.pageTaskConfigFactory);
			this.sessionService = new SessionServiceImpl(this.key, sessionSource, this.pageTaskHolder);

			PageTaskProducer pageTaskProducer = new PageTaskProducer(this,
					sourceDirectoryRoot, resultDirectoryRoot,
					SESSION_SOURCE_NEWFILE_IGNORE_THRESHOLD_IN_SEC);

			//this.errorLogEventHandler = new ErrorLogEventHandler(createErrorLogFile(this.PathUtil));

			this.resultEventManager = 
				resultEventManagerFactory.create(sessionSource, sourceDirectoryRoot, resultDirectoryRoot);
			this.resultEventManager.setResultEventMonitor(this);

			PageTaskConsumer pageTaskConsumer = new PageTaskConsumer(this,
					sourceDirectoryRoot, resultDirectoryRoot,
					this.resultEventManager);

			pageTaskProducer.setMonitor(this);
			pageTaskConsumer.setMonitor(this);

			this.sessionThreadManager.startDynamicPageTaskSourceProducer(pageTaskProducer);
			this.sessionThreadManager.startPageTaskConsumer(pageTaskConsumer);

			exportSessionService();

		} catch (IOException ex){
			ex.printStackTrace();
		}
	}


	public PageTaskHolder getPageTaskHolder() {
		return this.pageTaskHolder;
	}

	private void unexportSessionService() {
		if(this.advertisingService != null){
			this.advertisingService.stop();
			//this.advertisingService.shutdown();
		}

		try{
			UnicastRemoteObject.unexportObject(this.sessionService, false);
		}catch(NoSuchObjectException ignore){
		}
		if(this.advertisingService != null){
			this.advertisingService.unexport(this.serviceName);
		}
	}

	boolean isStopped = true;

	public boolean isStopped(){
		return this.isStopped;
	}

	public synchronized void stop(File sourceDirectoryRoot, File resultDirectoryRoot) {
		this.isStopped = true;
		unexportSessionService();

		if(this.sessionThreadManager != null){
			this.sessionThreadManager.stop();
		}
		if(this.resultEventManager != null){
			this.resultEventManager.stop();
		}
		this.sessionService = null;
	}

	public void shutdown() {
		unexportSessionService();
		if(this.sessionThreadManager != null){
			this.sessionThreadManager.shutdown();
		}
		if(this.resultEventManager != null){
			this.resultEventManager.shutdown();
		}
		PersistentCacheManager.shutdownAll();
		this.sessionService = null;
	}

	public synchronized SessionService getSessionService() {
		return this.sessionService;
	}

	public synchronized SessionServiceHandler getSessionServiceHandler() {
		return this.sessionService;
	}

/*
	public synchronized int getNumRemoteExecutors() {
		try{
			return this.sessionService.getNumRemoteExecutors();
		}catch(RemoteException ex){
			return 0;
		}
	}
*/
	
	public void notifyProducePageTask(PageTask pageTask) {
		for(ResultManagerMonitor monitor: monitors){
			monitor.notifyProducePageTask(pageTask);
		}
		this.resultEventManager.updateLastPageTaskProducedTime();
	}

	public void notifyConsumePageTask(PageTask pageTask) {
		for(ResultManagerMonitor monitor: monitors){
			monitor.notifyConsumePageTask(pageTask);
		}
	}

	public void notifyFoundMaster(PageMaster master, boolean scanDescendant) {
		for(ResultManagerMonitor monitor: monitors){
			monitor.notifyFoundMaster(master, scanDescendant);
		}
	}

	public void notifyFoundConfig(PageTaskConfig config, boolean scanDescendant) {
		for(ResultManagerMonitor monitor: monitors){
			monitor.notifyFoundConfig(config, scanDescendant);
		}
	}

	public void notifyFoundImages(int numAddedImages, File sourceDirectory) {
		for(ResultManagerMonitor monitor: monitors){
			monitor.notifyFoundImages(numAddedImages, sourceDirectory);
		}
	}

	public void notifyScanningDone(File sourceDirectoryRootFile) {
		for(ResultManagerMonitor monitor: monitors){
			monitor.notifyScanningDone(sourceDirectoryRootFile);
		}
	}

	public void notifyResultEvent(ResultEvent ev) {
		for(ResultManagerMonitor monitor: monitors){
			monitor.notifyResultEvent(ev);
		}
	}

	public abstract boolean execute() throws IOException;
}