/**
 *  SessionExecutorResource.java

 Copyright 2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Created on 2007/01/31
 Author hiroya
 */

package net.sqs2.exigrid.execute;

import java.rmi.RemoteException;
import java.util.HashMap;
import java.util.Map;

import net.sqs2.exigrid.master.PageMaster;
import net.sqs2.exigrid.session.Session;
import net.sqs2.exigrid.session.SessionService;
import net.sqs2.exigrid.source.PageTask;
import net.sqs2.exigrid.source.PageTaskConfig;
import net.sqs2.util.FileResource;

import org.apache.commons.collections15.map.LRUMap;

public class SessionExecutorResource{

	public static final int SESSIONTASK_EXECUTOR_IMAGE_BYTEARRAY_CACHE_SIZE = 3; // size

	private long key;
	private Session session;
	private SessionService sessionService;

	private Map<String, byte[]> imageByteArrayCache = new LRUMap<String,byte[]>(SESSIONTASK_EXECUTOR_IMAGE_BYTEARRAY_CACHE_SIZE); 
	private Map<String, Long> imageLastModifiedCache = new LRUMap<String,Long>(SESSIONTASK_EXECUTOR_IMAGE_BYTEARRAY_CACHE_SIZE);
	private Map<FileResource, PageMaster> masterObjectCache = new HashMap<FileResource, PageMaster>();
	private Map<String, PageTaskConfig> pageTaskConfigCache = new HashMap<String, PageTaskConfig>();

	SessionExecutorResource(long key, Session session, SessionService sessionService){
		this.key = key;
		this.session = session;
		this.sessionService = sessionService;
	}

	long getKey(){
		return this.key;
	}

	boolean isRemoteSlaveSession(){
		return (this.sessionService != null);
	}

	boolean isLocalMasterSessionProducingPageTasks(){
		return this.session.isNowProducingPageTasks();
	}

	boolean isLocalMasterSessionRunning(){
		return (0 < this.session.getPageTaskHolder().getNumPreparedPages());
	}

	SessionService getSessionService(){
		if(this.sessionService != null){
			return this.sessionService;
		}else if(this.session != null){
			return this.session.getSessionService();
		}else{
			throw new RuntimeException("sessionService == null and sessionManager == null");
		}
	}

	public byte[] getImageByteArray(String path)throws RemoteException{
		synchronized(this.imageByteArrayCache){
			long imageLastModified;
			byte[] imageByteArray = null;
			Long cachedImageLastModified = null;

			imageLastModified = getSessionService().getLastModified(this.key, path);
			imageByteArray = this.imageByteArrayCache.get(path);
			cachedImageLastModified = ((Long)this.imageLastModifiedCache.get(path));

			if(imageByteArray == null || cachedImageLastModified == null ||
					cachedImageLastModified.longValue() != imageLastModified){

				imageLastModified = getSessionService().getLastModified(this.key, path);
				imageByteArray = this.imageByteArrayCache.get(path);
				cachedImageLastModified = ((Long)this.imageLastModifiedCache.get(path));

				if(imageByteArray == null || cachedImageLastModified == null || 
						cachedImageLastModified.longValue() != imageLastModified){
					imageByteArray = getSessionService().getImageByteArray(this.key, path, imageLastModified); 
					this.imageByteArrayCache.put(path, imageByteArray);
					this.imageLastModifiedCache.put(path, new Long(imageLastModified));
				}
			}
			return imageByteArray;
		}
	}

	public PageMaster getPageMaster(PageTask pageTask)throws RemoteException{
		return getPageMaster(pageTask.getMasterPath(), pageTask.getMasterLastModified());
	}

	public PageMaster getPageMaster(String masterPath, long masterLastModified)throws RemoteException{
		synchronized(this.masterObjectCache){
			PageMaster master = this.masterObjectCache.get(new FileResource(masterPath, masterLastModified)); 
			if(master != null){		    
				return master;
			}else{
				master = (PageMaster)getSessionService().getMaster(this.key, masterPath, masterLastModified);
				this.masterObjectCache.put(master.getFileResource(), master);
				return master;
			}
		}
	}

	public PageTaskConfig getDefaultPageTaskConfig()throws RemoteException{
		return getPageTaskConfig(null, 0L);
	}

	public PageTaskConfig getPageTaskConfig(String pageTaskConfigPath, long pageTaskConfigLastModified)throws RemoteException{
		synchronized(this.pageTaskConfigCache){
			PageTaskConfig pageTaskConfig = this.pageTaskConfigCache.get(pageTaskConfigPath); 
			if(pageTaskConfig != null && pageTaskConfig.lastModified() == pageTaskConfigLastModified){
				return pageTaskConfig;
			}else{
				//  rewrite cache 
				pageTaskConfig = getSessionService().getPageTaskConfig(this.key, pageTaskConfigPath, pageTaskConfigLastModified);
				this.pageTaskConfigCache.put(pageTaskConfigPath, pageTaskConfig);
				return pageTaskConfig;
			}
		}
	}

	void close(){
		this.session = null;
		this.sessionService = null;
	}
}