/**
 *  SessionExecutorImpl.java

 Copyright 2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Created on 2007/01/31
 Author hiroya
 */

package net.sqs2.exigrid.execute;

import java.rmi.RemoteException;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;

import net.sqs2.exigrid.session.Session;
import net.sqs2.exigrid.session.SessionService;
import net.sqs2.exigrid.source.PageTask;
import net.sqs2.lang.GroupThreadFactory;

public class SessionExecutor{

	public static final int SESSIONTASK_EXECUTOR_THREAD_PRIORIY = Thread.NORM_PRIORITY - 1;

	private static BlockingQueue<SessionExecutorTask> downloadingTaskQueue;
	private static BlockingQueue<SessionExecutorTask> uploadingTaskQueue;
	private static ExecutorService executorThreadPool;
	private static Future<?>[] executorFutures = null;
	private static ScheduledExecutorService uploaderThreadPool;
	private static Future<?>[] uploaderThreadFutures = null;
	private static boolean isRunning = true;

	private SessionExecutorResource sessionResource;
	private ScheduledExecutorService downloaderThreadPool;
	private Future<?>[] downloaderThreadFutures = null;
	private boolean isConnected = false;

	public SessionExecutor() {
		super();
	}

	public SessionExecutor(final long key, Session session, SessionService sessionService){
		if(SessionExecutor.executorFutures == null){
			throw new RuntimeException("plz call initialize first.");
		}
		this.sessionResource = new SessionExecutorResource(key, session, sessionService);
	}

	public static void initialize(int numExecutorThreads, SessionExecutorCore sessionExecutorCore) {
		synchronized(SessionExecutor.class){
			SessionExecutor.downloadingTaskQueue = new ArrayBlockingQueue<SessionExecutorTask>(numExecutorThreads);
			SessionExecutor.uploadingTaskQueue = new LinkedBlockingQueue<SessionExecutorTask>();

			if(SessionExecutor.executorFutures == null){
				GroupThreadFactory groupThreadFactory1 = new GroupThreadFactory("net.sqs2.exigrid.execute.SessionExecutorImpl#ExecutorThread", SESSIONTASK_EXECUTOR_THREAD_PRIORIY, true);
				SessionExecutor.isRunning = true;
				SessionExecutor.executorFutures = new Future[numExecutorThreads];
				SessionExecutor.executorThreadPool = Executors.newFixedThreadPool(numExecutorThreads, groupThreadFactory1);
				for(int i = 0; i < numExecutorThreads; i++){
					SessionExecutor.executorFutures[i] = SessionExecutor.executorThreadPool.submit(new SessionExecutorThread(sessionExecutorCore));
				}

				int numUploaderThread = 2;
				GroupThreadFactory groupThreadFactory2 = new GroupThreadFactory("net.sqs2.exigrid.execute.SessionExecutorImpl#UploaderThread", SESSIONTASK_EXECUTOR_THREAD_PRIORIY, true);
				SessionExecutor.uploaderThreadPool = Executors.newScheduledThreadPool(numUploaderThread, groupThreadFactory2);
				SessionExecutor.uploaderThreadFutures = new Future[numUploaderThread];
				for(int i = 0; i < numUploaderThread; i++){
					SessionExecutor.uploaderThreadFutures[i] = SessionExecutor.uploaderThreadPool.scheduleWithFixedDelay(new PageTaskUploader(),
							10,
							1,
							TimeUnit.MILLISECONDS);			
				}
			}
		}
	}

	public boolean isConnected(){
		try{
			this.sessionResource.getSessionService().ping(sessionResource.getKey());
			return this.isConnected;
		}catch(RemoteException ignore){
			stop();
			return false;
		}
	}

	public void setConnected(boolean isConnected){
		this.isConnected = isConnected;
	}

	public void start() {
		int numDownloaderThreads = 1;
		GroupThreadFactory groupThreadFactory = new GroupThreadFactory("net.sqs2.exigrid.exe.SessionExecutorImpl#DownloaderThread", 
				SESSIONTASK_EXECUTOR_THREAD_PRIORIY, true);
		this.downloaderThreadPool = Executors.newScheduledThreadPool(numDownloaderThreads, groupThreadFactory);
		this.downloaderThreadFutures = new Future[numDownloaderThreads];
		setConnected(true);
		for(int i = 0; i < numDownloaderThreads; i++){
			this.downloaderThreadFutures[i] = this.downloaderThreadPool.scheduleWithFixedDelay(new PageTaskDownloader(),
					50 * i,
					1,
					TimeUnit.MILLISECONDS);
		}
	}

	class PageTaskDownloader implements Runnable{
		public void run(){
			try{
				if(SessionExecutor.this.sessionResource.isRemoteSlaveSession()){
					// remote session
					if(SessionExecutor.this.sessionResource.isLocalMasterSessionRunning()){
						sleep(1000);
						return;
					}
				}else{
					// local session
					if(SessionExecutor.this.sessionResource.isLocalMasterSessionProducingPageTasks()){
						sleep(1000);
						return;
					}
					if(0 < SessionExecutor.this.sessionResource.getSessionService().getNumRemoteSlaveExecutors()){
						sleep(1000);
						return;
					}
				}
				try{
					PageTask pageTask = SessionExecutor.this.sessionResource.getSessionService().leasePageTask(SessionExecutor.this.sessionResource.getKey()); // block
					if(pageTask == null){
						Thread.yield();
						return;
					}
					while(! SessionExecutor.downloadingTaskQueue.offer(new SessionExecutorTask(pageTask, SessionExecutor.this.sessionResource))){
						Thread.yield();
					}
				}catch(RemoteException ignore){
					Logger.getLogger("executor").warning("RemoteSession closed.");
					stop();
				}
			}catch(Exception ex){
			}
		}

		private void sleep(int msec) {
	        try{
	        	Thread.sleep(msec);
	        }catch(InterruptedException ignore){}
        }
	}

	static class SessionExecutorThread implements Runnable{

		SessionExecutorCore worker;
		SessionExecutorThread(SessionExecutorCore worker){
			this.worker = worker;
		}

		public void run(){
			while(SessionExecutor.isRunning){
				SessionExecutorTask task = null;
				try{
					task = SessionExecutor.downloadingTaskQueue.take();
					this.worker.execute(task);
					SessionExecutor.uploadingTaskQueue.offer(task);
				}catch(InterruptedException ignore){
				}catch(RemoteException ignore){
					// ignore exception in uploading. we will drop the uploading task but never stop uploading thread itself.
					//ignore.printStackTrace();
				}
				Thread.yield();
			}
		}
	}

	protected static class PageTaskUploader implements Runnable{
		public void run(){
			SessionExecutorTask task = uploadingTaskQueue.poll(); // block
			try{
				if(task == null){
					Thread.yield();
					return;
				}
				task.getSessionExecutorResource().getSessionService().submitPageTask(task.getSessionExecutorResource().getKey(), task.getPageTask());
			}catch(RemoteException ignore){
				// ignore exception in uploading. we will drop the uploading task but never stop uploading thread itself.
			}
		}
	}

	public void shutdown() {
		for(int i = 0; i < SessionExecutor.executorFutures.length; i++){
			SessionExecutor.executorFutures[i].cancel(true);
		}
		if(this.downloaderThreadPool != null){
			SessionExecutor.executorThreadPool.shutdown();
		}
		isRunning = false;

		setConnected(false);
		for(int i = 0; i < SessionExecutor.uploaderThreadFutures.length; i++){
			SessionExecutor.uploaderThreadFutures[i].cancel(true);
		}
		SessionExecutor.uploaderThreadPool.shutdown();
		if(this.sessionResource != null){
			this.sessionResource.close();
			this.sessionResource = null;
		}
	}

	public void stop() {
		if(this.downloaderThreadFutures != null){
			for(int i = 0; i < this.downloaderThreadFutures.length; i++){
				if(this.downloaderThreadFutures != null && this.downloaderThreadFutures[i] != null){
					this.downloaderThreadFutures[i].cancel(true);
				}
			}
		}
		setConnected(false);
	}

}
