/*
 * 

 ExigridEngineFactory.java

 Copyright 2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */
package net.sqs2.exigrid.engine;

import java.io.IOException;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.util.Random;
import java.util.logging.Logger;

import net.sqs2.exigrid.execute.RemoteSessionManager;
import net.sqs2.exigrid.execute.SessionDiscoveryService;
import net.sqs2.exigrid.execute.SessionExecutor;
import net.sqs2.exigrid.execute.SessionExecutorCore;
import net.sqs2.exigrid.execute.SessionExecutor;
import net.sqs2.exigrid.master.PageMasterFactory;
import net.sqs2.exigrid.session.ResultEventManagerFactory;
import net.sqs2.exigrid.session.Session;
import net.sqs2.exigrid.source.PageTaskConfigFactory;
import net.sqs2.net.MulticastNetworkConnection;
import net.sqs2.net.NetworkUtil;
import net.sqs2.net.RMIRegistryMulticastAdvertisingService;

public abstract class ExigridEngineFactory {

    public static final int PAGETASK_POLLING_TIMEOUT_IN_SEC = 29;// sec

    public static final int SESSION_SOURCE_DIRECTORY_SCAN_DELAY_IN_SEC = 60 * 10;// sec

    public static final int SESSION_SOURCE_ADVERTISE_DELAY_IN_SEC = 3;// sec

    public static final int ADVERTISE_SERVICE_THREAD_PRIORITY = Thread.NORM_PRIORITY - 2;

    public static final int DISCOVERY_SERVICE_THREAD_PRIORITY = Thread.NORM_PRIORITY - 2;

    public static final int PAGETASK_EXECUTORS_MAX_EXECUTORS = 5;

    public static final int PAGETASK_EXECUORS_EXEC_DELAY_IN_SEC = 1; // sec

    public static final int SESSION_SOURCE_ADVERTISE_DATAGRAM_PACKET_BUFFER_LENGTH = 512;// len
    
    public static final String MULTICAST_ADDRESS = "239.0.0.100";
    public static final int MULTICAST_PORT = 1091;
    
    private static final Random RANDOM = new Random();

    public ExigridEngineFactory() {
        super();
    }

    public ExigridEngine create(int rmiPort) throws UnknownHostException, IOException,
            InstantiationException, ClassNotFoundException, IllegalAccessException {
        long key = RANDOM.nextLong();
        MulticastNetworkConnection multicastNetworkConnection = createMulticastNetworkConnection();

        RMIRegistryMulticastAdvertisingService advertiseService = null;
        SessionDiscoveryService discoveryService = null;
        String hostname = null;
        try {
            hostname = NetworkUtil.getInet4HostAddress();
        } catch (SocketException ex) {
            hostname = "localhost";
        }
        
        if (multicastNetworkConnection != null) {
            System.setProperty("java.rmi.server.hostname", hostname);

            advertiseService = new RMIRegistryMulticastAdvertisingService(
                    multicastNetworkConnection, key,
                    ADVERTISE_SERVICE_THREAD_PRIORITY, rmiPort,
                    SESSION_SOURCE_ADVERTISE_DELAY_IN_SEC);

        }

        Session session =
                new Session((PageMasterFactory) Class.forName(getPageMasterFactoryClassName()).newInstance(),
                        (PageTaskConfigFactory) Class.forName(getPageTaskConfigFactoryClassName()).newInstance(),
                        (ResultEventManagerFactory) Class.forName(getResultEventManagerFactoryClassName()).newInstance(),
                        key, advertiseService, getSessionServiceName());

        RemoteSessionManager remoteSessionManager = null;
        if (multicastNetworkConnection != null) {
            remoteSessionManager = new RemoteSessionManager(
                    session, PAGETASK_EXECUTORS_MAX_EXECUTORS);
            discoveryService = new SessionDiscoveryService(
                    multicastNetworkConnection,
                    DISCOVERY_SERVICE_THREAD_PRIORITY,
                    SESSION_SOURCE_ADVERTISE_DATAGRAM_PACKET_BUFFER_LENGTH,
                    remoteSessionManager);
        }

        SessionExecutor.initialize(Runtime.getRuntime().availableProcessors(),
                (SessionExecutorCore) Class.forName(getSessionExecutorCoreClassName()).newInstance());
        SessionExecutor localExecutors = new SessionExecutor(key, session, null);
        return new ExigridEngine(getTitle(), getVersionID(), rmiPort, session, localExecutors,
                discoveryService, remoteSessionManager);
    }

    protected MulticastNetworkConnection createMulticastNetworkConnection() {
        MulticastNetworkConnection multicastNetworkConnection = null;
        try {
            multicastNetworkConnection =
                    new MulticastNetworkConnection(
                            getSessionServiceMulticastAddress(),
                            getSessionServiceMulticastPort());
            Logger.getLogger("session").warning("multicast session has established.");
        } catch (IOException ignore) {
            Logger.getLogger("session").warning("multicast session has NOT established.");
        }
        return multicastNetworkConnection;
    }

    public abstract String getTitle();

    public abstract String getVersionID();

    public abstract String getPageMasterFactoryClassName();

    public abstract String getPageTaskConfigFactoryClassName();

    public abstract String getResultEventManagerFactoryClassName();

    public abstract String getSessionExecutorCoreClassName();

    public abstract String getSessionServiceName();

    public String getSessionServiceMulticastAddress() {
        return MULTICAST_ADDRESS;
    }

    public int getSessionServiceMulticastPort() {
        return MULTICAST_PORT;
    }

}