/*

 ExigridEngine.java

 Copyright 2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Created on 2007/01/11

 */
package net.sqs2.exigrid.engine;

import java.io.File;
import java.io.IOException;
import java.net.UnknownHostException;
import java.util.HashSet;
import java.util.Set;

import net.sqs2.exigrid.event.ResultEvent;
import net.sqs2.exigrid.event.ResultManagerMonitor;
import net.sqs2.exigrid.execute.RemoteSessionManager;
import net.sqs2.exigrid.execute.SessionDiscoveryService;
import net.sqs2.exigrid.execute.SessionExecutor;
import net.sqs2.exigrid.master.PageMaster;
import net.sqs2.exigrid.session.AbstractSession;
import net.sqs2.exigrid.session.PageTaskHolder;
import net.sqs2.exigrid.source.PageTask;
import net.sqs2.exigrid.source.PageTaskConfig;
import net.sqs2.exigrid.source.PageTaskException;

public class ExigridEngine implements ResultManagerMonitor {

	private String title;
	private String versionID;

	private int rmiPort;

	private AbstractSession session;

	private SessionExecutor localExecutor;

	private SessionDiscoveryService discoveryService;

	private RemoteSessionManager remoteSessionManager;

	private Set<ResultManagerMonitor> monitors = new HashSet<ResultManagerMonitor>();

	public ExigridEngine(String title, String versionID,
			int rmiPort,
			AbstractSession session,
			SessionExecutor localExecutor,
			SessionDiscoveryService discoveryService,
			RemoteSessionManager remoteSessionManager)
	throws UnknownHostException, IOException {
		this.title = title;
		this.versionID = versionID;

		this.rmiPort = rmiPort;
		this.localExecutor = localExecutor;
		this.discoveryService = discoveryService;
		this.session = session;
		this.remoteSessionManager = remoteSessionManager;

		this.session.addMonitor(this);
	}

	public static String getBaseURI(){
		return "class://net.sqs2.exigrid.engine.ExigridEngine/";
	}

	public String getTitle() {
		return this.title;
	}

	public String getVersionID() {
		return this.versionID;
	}

	public AbstractSession getSession() {
		return this.session;
	}

	public int getRMIPort() {
		return this.rmiPort;
	}

	public void start(File sourceDirectoryRoot, File resultDirectoryRoot,
			boolean doReset) throws PageTaskException {

		this.session.start(sourceDirectoryRoot, resultDirectoryRoot, doReset);
		this.localExecutor.start();
		notifyStarted();
	}

	public void stop(File sourceDirectoryRoot, File resultDirectoryRoot) {
		notifyStopped();
		this.localExecutor.stop();
		this.session.stop(sourceDirectoryRoot, resultDirectoryRoot);
	}

	public void addSessionManagerMonitor(ResultManagerMonitor monitor) {
		this.monitors.add(monitor);
	}

	public void removeSessionManagerMonitor(ResultManagerMonitor monitor) {
		this.monitors.remove(monitor);
	}

	public PageTaskHolder getPageTaskHolder() {
		return this.session.getPageTaskHolder();
	}

	public void shutdown() {
		if (this.discoveryService != null) {
			this.discoveryService.shutdown();
			this.remoteSessionManager.shutdown();
		}
		this.localExecutor.shutdown();
		this.session.shutdown();
	}

	public void notifyStarted() {
		for (ResultManagerMonitor monitors : this.monitors) {
			monitors.notifyStarted();
		}
	}

	public void notifyStopped() {
		for (ResultManagerMonitor monitor : this.monitors) {
			monitor.notifyStopped();
		}
	}

	public void notifyProducePageTask(PageTask pageTask) {
		for (ResultManagerMonitor monitor : this.monitors) {
			monitor.notifyProducePageTask(pageTask);
		}
	}

	public void notifyConsumePageTask(PageTask pageTask) {
		for (ResultManagerMonitor monitor : this.monitors) {
			monitor.notifyConsumePageTask(pageTask);
		}
	}

	public void notifyFoundMaster(PageMaster master, boolean scanDescendant) {
		for (ResultManagerMonitor monitor : this.monitors) {
			monitor.notifyFoundMaster(master, scanDescendant);
		}
	}

	public void notifyFoundConfig(PageTaskConfig config, boolean scanDescendant) {
		for (ResultManagerMonitor monitor : this.monitors) {
			monitor.notifyFoundConfig(config, scanDescendant);
		}
	}

	public void notifyFoundImages(int numAddedImages, File sourceDirectory) {
		for (ResultManagerMonitor monitor : this.monitors) {
			monitor.notifyFoundImages(numAddedImages, sourceDirectory);
		}
	}

	public void notifyScanningDone(File sourceDirectoryRootFile) {
		for (ResultManagerMonitor monitor : this.monitors) {
			monitor.notifyScanningDone(sourceDirectoryRootFile);
		}
	}

	public void notifyResultEvent(ResultEvent ev) {
		for (ResultManagerMonitor monitor : this.monitors) {
			monitor.notifyResultEvent(ev);
		}
	}

	public void notifyFinished() {
		for (ResultManagerMonitor monitor : this.monitors) {
			monitor.notifyFinished();
		}
	}
}
