/*

 ExportFileLogic.java
 
 Copyright 2004 KUBO Hiroya (hiroya@sfc.keio.ac.jp).
 
 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at
 
 http://www.apache.org/licenses/LICENSE-2.0
 
 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 
 Created on 2004/08/18

 */
package net.sf.sqs_xml.editor.sqs.swing;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import javax.swing.JOptionPane;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactoryConfigurationError;

import net.sf.sqs_xml.editor.base.swing.SourceEditorMediator;
import net.sqs2.browser.Browser;
import net.sqs2.editor.httpd.SQSHttpdManager;
import net.sqs2.exsed.module.sqs.source.SQSSource;
import net.sqs2.exsed.module.sqs.source.SQSToHTMLTranslator;
import net.sqs2.exsed.module.sqs.source.SQSToPDFTranslator;
import net.sqs2.exsed.module.sqs.source.SQSToPreviewTranslator;
import net.sqs2.swing.LoggerConsoleFrame;
import net.sqs2.translator.AbstractTranslator;
import net.sqs2.translator.TranslatorException;

public class ExportFileLogic{
    private SourceEditorMediator mediator;
    private Thread exportWorkerThread;
    //private SQSToPreviewTranslator sqsToPreviewTranslator;
    //private PreviewDialog previewDialog;
    
    public ExportFileLogic(SourceEditorMediator mediator){
        this.mediator = mediator;
    }
    
    private LoggerConsoleFrame createLogger(){
        final LoggerConsoleFrame logger = new LoggerConsoleFrame(mediator.getTitle()+" Log",  "Cancel",  "OK");
        logger.setFinished(false);
		logger.getCancelButton().addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ev){
				logger.setVisible(false);
				logger.setFinished(true);
				if(exportWorkerThread != null && exportWorkerThread.isAlive() && ! exportWorkerThread.isInterrupted()){
					exportWorkerThread.interrupt();
				}
			}
		});	
        logger.info("start translation.");
        logger.setSize(400, 300);
        logger.setVisible(true);
		return logger;
    }
    
    public synchronized void preview(String title)throws TranslatorException, IOException{
    	/*
    	if(this.sqsToPreviewTranslator == null){
    		this.sqsToPreviewTranslator = new SQSToPreviewTranslator();
    	}
    	if(this.previewDialog == null){    		
    		this.previewDialog = new PreviewDialog(SQSToPreviewTranslator.createFOUserAgent());    		
    	}*/
    	new SQSToPreviewTranslator(title).translate(createSQSInputStream(), null);
    }

    public void exportPDF()throws TranslatorException, IOException{    	
    	exportPDF(createTemporaryFile(".pdf"));
    }

    public void exportPDF(File exportingFile)throws TranslatorException, IOException{
        translate(new SQSToPDFTranslator(exportingFile), exportingFile);
    }
    
	public void exportHTML()throws TranslatorException, IOException{
		exportHTML(createTemporaryFile(".xhtml"));
	}
	
	public void exportHTML(File exportingFile)throws TranslatorException, IOException{
        translate(new SQSToHTMLTranslator(), exportingFile);
    }
	
    private File createTemporaryFile(String suffix)throws IOException{       
        File targetFile = File.createTempFile("sqs-draft-", suffix);
        targetFile.deleteOnExit();
        return targetFile;
    }

    public synchronized void translate(final AbstractTranslator translator, final File targetFile) throws TransformerFactoryConfigurationError {
    	ThreadGroup threadGroup = new ThreadGroup("translate");
    	threadGroup.setMaxPriority(Thread.MIN_PRIORITY);
    	this.exportWorkerThread = new Thread(threadGroup, "exportWorkerThread") {
            public void run() {
                try {
                	String systemId = null; //TODO
                    InputStream sqsInputStream = createSQSInputStream();
                    OutputStream outputStream = new BufferedOutputStream(new FileOutputStream(targetFile));
                    translate(translator, outputStream, systemId, sqsInputStream);
                    Browser.showDocument(SQSHttpdManager.getEXsedHttpd().createTempFileURL(targetFile));
                    /*
                    if(launcher != null && isBrowserAccessibleDirectory(tgtFile.getAbsolutePath())){
                        launcher.showDocument();
                    }                    
                    */
                } catch (InterruptedException ex){
                	Thread.currentThread().interrupt();
                } catch (IOException ex) {
                    ex.printStackTrace();
                    mediator.createSourceEditorMenuBarMediator().showError(ex, "エラー");
                } catch (TranslatorException ex) {
                    ex.printStackTrace();
                    mediator.createSourceEditorMenuBarMediator().showError(ex, "エラー");
                } catch (Exception ex) {
                    ex.printStackTrace();
                    mediator.createSourceEditorMenuBarMediator().showError(ex, "エラー");
                }
            }
        };
        this.exportWorkerThread.start();
    }
        
    private InputStream createSQSInputStream() throws IOException {
        return ((SQSSource)mediator.getSourceEditorTabbedPane().getCurrentEditingSource()).createInputStream();
    }

    private synchronized void translate(final AbstractTranslator translator, final OutputStream tmpOutputStream, final String systemId, final InputStream sqsInputStream) throws TranslatorException {
        mediator.getToolBar().setEnabled(false);
        final LoggerConsoleFrame logger = createLogger();
        try {
        	// TODO checking...
        	translator.execute(sqsInputStream, systemId, tmpOutputStream);
        } catch (TranslatorException ex) {
            if(ex.getCause() instanceof TransformerException){
                showTransformerException(logger, ex);
            }
            throw ex;
        } catch (IOException ex) {
            if(ex.getCause() instanceof TransformerException){
                // showTransformerException(logger, ex);
            }
            //throw ex;
            ex.printStackTrace();
        } finally {
            tearOffTranslation(tmpOutputStream, sqsInputStream, logger);
        }
    }

    private void tearOffTranslation(final OutputStream tmpOutputStream, final InputStream sqsInputStream, final LoggerConsoleFrame logger) {
        mediator.getToolBar().setEnabled(true);
        logger.setFinished(true);
        if(! logger.hasError()){
            logger.setVisible(false);
        }
        try {
            sqsInputStream.close();
            tmpOutputStream.close();
        } catch (IOException ignore) {
        }
    }

    private void showTransformerException(final LoggerConsoleFrame logger, TranslatorException ex) {
        TransformerException cause = (TransformerException)ex.getCause();
        logger.error(cause.getMessageAndLocation());
        JOptionPane.showMessageDialog(mediator.getFrame(),
                new Object[]{"ファイル形式が正しくないため，変換処理に失敗しました:",
                "位置:"+cause.getLocationAsString(),
                "内容:"+cause.getLocalizedMessage()
        		},
                "File Error", JOptionPane.ERROR_MESSAGE);
    }

}