/*

 SourceEditorMenuBar.java
 
 Copyright 2004 KUBO Hiroya (hiroya@sfc.keio.ac.jp).
 
 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at
 
 http://www.apache.org/licenses/LICENSE-2.0
 
 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 
 Created on 2004/08/03

 */
package net.sf.sqs_xml.editor.base.swing;


import java.awt.Cursor;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.List;

import javax.swing.JFileChooser;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JSeparator;
import javax.swing.KeyStroke;

import net.sqs2.exsed.source.DOMTreeSource;
import net.sqs2.exsed.source.Source;
import net.sqs2.exsed.source.SourceException;

import org.w3c.dom.Node;
import org.xml.sax.SAXParseException;


/**
 * @author hiroya
 *
 */
public abstract class SourceEditorMenuBarMediator{
	FileMenu fileMenu;
	JMenuBar menuBar;
	JMenu helpMenu;

	SourceEditorMediator mediator;
	private static final File USER_HOME = new File(System.getProperty("user.dir"));
    private List<Node> recentCutNode;
    
	public SourceEditorMenuBarMediator(SourceEditorMediator mediator) {
	    super();
	    this.mediator = mediator;
	    this.menuBar = new JMenuBar();
	    this.fileMenu = new FileMenu();
	    this.helpMenu = createHelpMenu();
	    
	    fileMenu.setMnemonic(KeyEvent.VK_F);
	    menuBar.add(fileMenu);
		menuBar.add(helpMenu);
	}
	
	public List<Node> getRecentCutNode(){
	    return recentCutNode;
	}
	
	public void setRecentCutNode(List<Node> list){
	    this.recentCutNode = list;
	}
	
	public JMenuBar getMenuBar(){
	    return menuBar;
	}
	
	public SourceEditorMediator getMediator(){
	    return mediator;
	}
	
	public class FileMenu extends JMenu{
        public static final long serialVersionUID = 0;
	    public JMenuItem openMenuItem;
	    public JMenuItem saveMenuItem;
	    public JMenuItem saveAsMenuItem;
	    public JMenuItem closeMenuItem;
	    public JMenuItem quitMenuItem;
	    public JMenuItem exportMenuItem;
	    
        /**
         * @return Returns the closeMenuItem.
         */
        public JMenuItem getCloseMenuItem() {
            return closeMenuItem;
        }
        /**
         * @return Returns the openMenuItem.
         */
        public JMenuItem getOpenMenuItem() {
            return openMenuItem;
        }
        /**
         * @return Returns the quitMenuItem.
         */
        public JMenuItem getQuitMenuItem() {
            return quitMenuItem;
        }
        /**
         * @return Returns the saveAsMenuItem.
         */
        public JMenuItem getSaveAsMenuItem() {
            return saveAsMenuItem;
        }
        /**
         * @return Returns the saveMenuItem.
         */
        public JMenuItem getSaveMenuItem() {
            return saveMenuItem;
        }

        public JMenuItem getExportMenuItem() {
            return exportMenuItem;
        }

        public FileMenu() {
	        setText("ファイル");

	        this.openMenuItem = new OpenMenuItem();
	        this.saveMenuItem = new SaveMenuItem();
	        this.saveAsMenuItem = new SaveAsMenuItem();
	        this.closeMenuItem = new CloseMenuItem();
	        this.quitMenuItem = new QuitMenuItem();
	        this.exportMenuItem = createExportMenuItem();
	        
	        add(createNewMenuItem());
	        add(this.openMenuItem);
	        add(new JSeparator());
	        add(this.saveMenuItem);
	        add(this.saveAsMenuItem);
	        add(this.exportMenuItem);
	        add(new JSeparator());
	        add(this.closeMenuItem);
	        add(new JSeparator());			
	        add(this.quitMenuItem);
	    }
				
	    private void save() {
	        Source source = getCurrentSource();
	        if(source.isDirty()){
	            if(avoidOverrideUpdatedFile(source)){
	                return;
	            }
	            try{
	                source.save();
	            }catch(IOException ex){
	                showError(ex, "ファイルの保存ができません:"+source.getFile());
	            }
	        }
	    }

        private void saveAs() {
            Source currentSource = getCurrentSource();
            JFileChooser saveAsFileChooser = createSaveAsFileChoser(currentSource);
	        int result = saveAsFileChooser.showSaveDialog(menuBar);
	        if(result == JFileChooser.APPROVE_OPTION){
	            File saveAsFile = saveAsFileChooser.getSelectedFile();
	            if(saveAsFile.equals(currentSource.getFile())){
	                save();
	                return;
	            }
	            
	            if(avoidOverrideOpenedBuffer(saveAsFile)){
	                return;
	            }
	            if(avoidOverwriteFile(saveAsFile)){
	                return;
	            }
	            currentSource.setFile(saveAsFile);
	            currentSource.setReadOnly(false);
	            try{
	                currentSource.save();
	            }catch(IOException ex){
	                showError(ex, "ファイルの保存ができません:"+currentSource.getFile());
	            }
	        }
	    }


		public class OpenMenuItem extends JMenuItem{
            public static final long serialVersionUID = 0;
		    public OpenMenuItem(){
		        setText("開く…");
		        setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_O, ActionEvent.CTRL_MASK));
		        addActionListener(createFileOpenActionListener());
		    }

	        private ActionListener createFileOpenActionListener() {
	            return new ActionListener(){
				  	public void actionPerformed(ActionEvent ev){
				  	    File defaultFolder = null;
				  	    try{
				  	    	defaultFolder = getCurrentSource().getFile().getParentFile();
				  	    }catch(NullPointerException ex){
				  	    	defaultFolder = USER_HOME;
				  	    }
			  	    	JFileChooser openFileChooser = createOpenFileChooser(defaultFolder);
			  	    	int result = openFileChooser.showOpenDialog(menuBar);
			  	    	if(result == JFileChooser.APPROVE_OPTION){
			  	    		open(openFileChooser.getSelectedFile());
			  	    	}

				    }
				};
	        }
		}

		public class CloseMenuItem extends JMenuItem{
            public static final long serialVersionUID = 0;
		    public CloseMenuItem(){
			    setText("閉じる");
			    setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_W, ActionEvent.CTRL_MASK));
				addActionListener(createFileCloseActionListener());
			}
		}	

		public class SaveMenuItem extends JMenuItem{
            public static final long serialVersionUID = 0;
			public SaveMenuItem(){
			    setText("保存する");
			    setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_S, ActionEvent.CTRL_MASK));
				addActionListener(createFileSaveActionListener());
			}

	        private ActionListener createFileSaveActionListener() {
	            return new ActionListener(){
				    public void actionPerformed(ActionEvent ev){
				        if(getCurrentSource().getFile() == null){
				            saveAs();
				        }else{
				            save();
				        }
				        mediator.getSourceEditorTabbedPane().updateCurrentTitle();
				        updateMenu();
				    }
				};
	        }
		}
	    
	    public class SaveAsMenuItem extends JMenuItem{
            public static final long serialVersionUID = 0;
	        public SaveAsMenuItem(){
			    setText("名前を変えて保存する…");
				addActionListener(createSaveAsActionListener());
	        }

	        private ActionListener createSaveAsActionListener() {
	            return new ActionListener(){
				    public void actionPerformed(ActionEvent ev){
				        saveAs();
				        mediator.getSourceEditorTabbedPane().updateCurrentTitle();
			            updateMenu();
				    }
				};
	        }
		}
		
		public class QuitMenuItem extends JMenuItem{
            public static final long serialVersionUID = 0;
			public QuitMenuItem() {
				super("終了", KeyEvent.VK_Q);
			    //setText();
				addActionListener(createQuitActionListener());
				setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Q, ActionEvent.CTRL_MASK));
			}	

	        private ActionListener createQuitActionListener() {
	            return new ActionListener(){
					   public void actionPerformed(ActionEvent ev){
					       if(avoidUnsavedExit(mediator.getSourceManager().countDirtySources())){
					           return;
					       }
					       System.exit(0);
					   }
				};
	        }
		}	
	}

	/*
	public class HelpMenu extends JMenu{
        public static final long serialVersionUID = 0;
	    public HelpMenu(){
	        setText("ヘルプ");
			add(new HelpMenuItem());
			add(createAboutMenuItem());
			setEnabled(false);
		}

	    public class HelpMenuItem extends JMenuItem{
            public static final long serialVersionUID = 0;
	        public HelpMenuItem() {
	            setText("ヘルプの目次");
			}	        
		}
	}*/

	public abstract File getExportingFile(File sourceFile, String suffix);
	public abstract JMenu createNewMenuItem();
	public abstract JMenuItem createExportMenuItem();
    public abstract JMenu createHelpMenu();
    
    public Source getCurrentSource() {
        int index = mediator.getSourceEditorTabbedPane().getSelectedIndex();
        return mediator.getSourceManager().get(index);
    }

    private ActionListener createFileCloseActionListener() {
        return new ActionListener(){
		    public void actionPerformed(ActionEvent ev){
		        close();
		    }
		};
    }
    
    protected boolean avoidNodeDelete(int n) {
        return JOptionPane.OK_OPTION != JOptionPane.showConfirmDialog(mediator.getFrame(),
        	        n+"個のノードを削除しようとしています．本当に削除しますか?", "Delete Confirmation",
        	        JOptionPane.OK_OPTION);
    }

    private boolean avoidDisposeBuffer(Source source) {
        return source.isDirty() && JOptionPane.OK_OPTION != JOptionPane.showConfirmDialog(mediator.getFrame(),
        	        "このバッファは保存されていません．このバッファを破棄しますか?", "Dispose Confirmation",
        	        JOptionPane.OK_OPTION);
    }

    /*
    private boolean avoidOverwriteBuffer(Source source) {
        return avoidOverrideOpenedBuffer(source.getFile());
    }
    */
    
    protected boolean avoidOverwriteFile(File file) {
        return file.exists() && JOptionPane.OK_OPTION != JOptionPane.showConfirmDialog(mediator.getFrame(),
        	        file+" はすでに存在します．このファイルを上書きしますか?", "Overwrite Confirmation",
        	        JOptionPane.OK_OPTION);
    }

    private boolean avoidOverrideUpdatedFile(Source source) {
        return source.isModified() && JOptionPane.OK_OPTION != JOptionPane.showConfirmDialog(mediator.getFrame(),
        	        "別のプロセスによってファイルが更新されています．このファイルを上書きしますか?", "Override Confirmation",
        	        JOptionPane.OK_OPTION);
    }

    private boolean avoidOverrideOpenedBuffer(File file) {
        int index = mediator.getSourceManager().getIndexOfFile(file);
        if(0 <= index){
            if(JOptionPane.OK_OPTION == JOptionPane.showConfirmDialog(mediator.getFrame(),
        	        "このファイルは既にバッファとして開かれています．このバッファを上書きしますか?", "Override Confirmation",
        	        JOptionPane.OK_OPTION)){
                mediator.getSourceManager().close(index);
                mediator.tabbedPane.removeTabAt(index);
                return false;
            }else{
                return true;
            }
        }else{
            return false;
        }
    }
    
    private boolean avoidUpdatedFileOpen(Source source){
        return source.isModified() && JOptionPane.OK_OPTION != JOptionPane.showConfirmDialog(mediator.getFrame(),
		        "別のプロセスによってファイルが更新されています．編集中の内容を破棄してファイルを読み直しますか?", "Updated File Open Confirmation",
		        JOptionPane.YES_OPTION);
    }
    
    private boolean avoidUnsavedExit(int count){
        return 0 < count && JOptionPane.showConfirmDialog(getMediator().getFrame(),
                "未保存のバッファが "+ count +"個あります．終了しますか?", "Exit Confirmation",
                JOptionPane.YES_OPTION) != JOptionPane.OK_OPTION;
    }
	/*
    public void showError(Exception ex) {
        showError(ex, "エラー:");
    }
    */
    public void showError(Exception ex, String message) {
        JOptionPane.showMessageDialog(mediator.getFrame(), new Object[]{message, ex}, "Error", JOptionPane.ERROR_MESSAGE);
    }

    private void showFileOpenErrorMessage(SourceException ex) {
        if(ex.getCause() instanceof SAXParseException){
            	SAXParseException saxParseException = ((SAXParseException)ex.getCause());
            	JOptionPane.showMessageDialog(getMediator().getFrame(),
            	        new Object[]{"ファイル形式が正しくないため，読み込むことができません:",
            	        "SystemID: "+saxParseException.getSystemId(),
            	        "PublicID: "+saxParseException.getPublicId(),
            	        "行: "+saxParseException.getLineNumber()+"  "+"列: "+saxParseException.getColumnNumber(),
            	        saxParseException.getLocalizedMessage()
        			},
        			"File Error", JOptionPane.ERROR_MESSAGE);
            	ex.printStackTrace();
        }else{
        	JOptionPane.showMessageDialog(getMediator().getFrame(),
        	        new Object[]{"ファイル形式が正しくないため，読み込むことができません:",
        	        ex.getLocalizedMessage()
    			},
    			"File Error", JOptionPane.ERROR_MESSAGE);
        	ex.printStackTrace();
        }
    }
    
    public DOMTreeSource createSource(File file)throws SourceException{
        if(file != null){
            return (DOMTreeSource) mediator.getSourceManager().createSource(file);
        }else{
            return (DOMTreeSource) mediator.getSourceManager().createSource();
        }
    }

    public DOMTreeSource createSource(URL url, boolean readonly, String title)throws SourceException{
        return (DOMTreeSource) mediator.getSourceManager().createSource(url, readonly, title);
    }

    public void open(URL url, boolean readonly, String title) {
        try {
            mediator.getFrame().setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
            DOMTreeSource source = createSource(url, readonly, title);
            mediator.tabbedPane.addComponent(source);
            updateMenu();
        } catch (SourceException ex) {
            showFileOpenErrorMessage(ex);
        }finally{
            mediator.getFrame().setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
        }
    }
    
    public void open(File file) {
        try {
            mediator.getFrame().setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
            DOMTreeSource source = null;
            if(file != null){
                int index = -1;
                index = mediator.getSourceManager().getIndexOfFile(file);
//                System.err.println(index+":"+mediator.getSourceManager().size()+":"+mediator.tabbedPane.getComponentCount());
                if(index == -1){
                    source = createSource(file);
                    mediator.tabbedPane.addComponent(source);
                }else{
                    source = (DOMTreeSource)mediator.getSourceManager().get(index);
	  	          	if(avoidUpdatedFileOpen(source)){
	  	          	    return;
	  	          	}else{
	  	          	    source.setFile(file);
	  	          	    source.initialize(file);
	  	          	    mediator.tabbedPane.setComponent(index, source);
	  	          	}
                }
            }
        } catch (SourceException ex) {
            showFileOpenErrorMessage(ex);
        }finally{
            updateMenu();
            mediator.getFrame().setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
        }
    }

    public void close() {
        Source source = getCurrentSource();
        int index = mediator.getSourceEditorTabbedPane().getSelectedIndex();
        if(avoidDisposeBuffer(source)){
            return;
        }
        mediator.getSourceEditorTabbedPane().removeTabAt(index);
        mediator.getSourceManager().close(index);
        updateMenu();
        if(mediator.getSourceManager().size()==0){
            fileMenu.closeMenuItem.setEnabled(false);
        }
    }

	public void updateMenu(){
        int selectedIndex = getMediator().getSourceEditorTabbedPane().getSelectedIndex();
        if (mediator.getSourceEditorTabbedPane().getComponentCount() == 0) {
            fileMenu.getSaveMenuItem().setEnabled(false);
            fileMenu.getSaveAsMenuItem().setEnabled(false);
            fileMenu.getExportMenuItem().setEnabled(false);
            fileMenu.getCloseMenuItem().setEnabled(false);
            mediator.toolBar.setEnabled(false);
        } else {
            Source source = getMediator().getSourceManager().get(selectedIndex);
            if (source.isDirty()) {
                fileMenu.getSaveMenuItem().setEnabled(true);
                fileMenu.getSaveAsMenuItem().setEnabled(true);
                fileMenu.getExportMenuItem().setEnabled(true);
            } else {
                fileMenu.getSaveMenuItem().setEnabled(false);
                fileMenu.getSaveAsMenuItem().setEnabled(true);
                fileMenu.getExportMenuItem().setEnabled(true);
            }
            fileMenu.getCloseMenuItem().setEnabled(true);
            mediator.toolBar.setEnabled(true);
        }
    }

	public abstract JFileChooser createSaveAsFileChoser(Source currentSource);
	public abstract JFileChooser createOpenFileChooser(File defaultFolder);

}
