package org.sqlite.text;

import java.io.UnsupportedEncodingException;
import java.sql.SQLException;
import java.util.Comparator;
import org.sqlite.callback.NamedCallback;
import org.sqlite.jdbc.JdbcSQLException;
import org.sqlite.swig.SQLite3;
import static org.sqlite.swig.SQLite3.sqlite3_create_collation;
import static org.sqlite.swig.SQLite3.sqlite3_drop_collation;
import org.sqlite.swig.SWIGTYPE_p_sqlite3;
import static org.sqlite.swig.SQLite3Constants.SQLITE_OK;

/**
 * User-Defined Collating Sequence class.
 * @author calico
 * @see <a href="http://sqlite.org/c3ref/create_collation.html">Define New Collating Sequences</a>
 * @see org.sqlite.jdbc.JdbcConnection#createCollationSequence(org.sqlite.text.Collator)
 * @see org.sqlite.jdbc.JdbcConnection#dropCollationSequence(org.sqlite.text.Collator)
 */
public abstract class Collator extends NamedCallback implements Comparator<String> {

    /**
     * invoke sqlite3_create_collation_v2() function and this object is registered in the database.<br/>
     * WARNING! Do not use this method because it is called internally.
     * @param db the database handle.
     * @throws java.sql.SQLException When the return value of the sqlite3_create_collation_v2() function is not SQLITE_OK.
     * @see org.sqlite.Database#createCollationSequence(Collator)
     */
    @Override
    public void register(SWIGTYPE_p_sqlite3 db) throws SQLException {
        final int ret = sqlite3_create_collation(db, this);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(ret, db);
        }
    }

    /**
     * Unregister this object from the database.<br/>
     * WARNING! Do not use this method because it is called internally.
     * @param db the database handle.
     * @throws java.sql.SQLException When the return value of the sqlite3_create_collation() function is not SQLITE_OK.
     * @see org.sqlite.Database#dropCollationSequence(Collator)
     */
    @Override
    public void unregister(SWIGTYPE_p_sqlite3 db) throws SQLException {
        final int ret = sqlite3_drop_collation(db, this);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(ret, db);
        }
    }
    
    /** text encoding name */
    private final String encoding;
    
    /**
     * Create new collation sequence with SQLITE_UTF8.
     * @param name the name of the new collation sequence.
     * @see NamedCallback#NamedCallback(java.lang.String)
     */
    public Collator(String name) {
        super(name);
        this.encoding = null;
    }
    
    /**
     * Create new collation sequence.
     * @param name the name of the new collation sequence.
     * @param enc the most desirable encoding may be one of the constants SQLITE_UTF8, SQLITE_UTF16LE, SQLITE_UTF16BE or SQLITE_UTF16_ALIGNED.
     * @see <a href="http://sqlite.org/c3ref/c_any.html">Text Encodings</a>
     * @see NamedCallback#NamedCallback(java.lang.String, int)
     */
    public Collator(String name, int enc) {
        super(name, enc);
        this.encoding = SQLite3.getEncodingName(enc);
    }
    
    /**
     * Called from the sqlite3_step() function.
     * @param source the source bytes to be compared.
     * @param target the target bytes to be compared.
     * @return a negative integer, zero, or a positive integer as the first argument is less than, equal to, or greater than the second.
     * @see java.util.Comparator#compare(Object, Object)
     */
    protected final int xCompare(byte[] source, byte[] target) throws UnsupportedEncodingException {
        // TODO Mysaifu JVMのBug#11980が解決したらアクセス修飾子をprivateに変更すること！
        // @see http://sourceforge.jp/tracker/index.php?func=detail&aid=11980&group_id=1890&atid=7027
        return compare(new String(source, encoding), new String(target, encoding));
    }
}
