/* File : sqlite3jni.i */
%module SQLite3
%{
#include "sqlite3.h"

/* declare function instead of #include "sqliteInt.h" for sqlite3_get_errmsg(int) */
const char *sqlite3ErrStr(int);

#ifdef __cplusplus
extern "C" {
#endif

/* Field ID of org.sqlite.callback.Callback._this member field */
static jfieldID fid_callback_this = 0;

/* org.sqlite.udf.Function class */
static jclass clsFunction = 0;

/* Field ID of org.sqlite.udf.Function member fields  */
static jfieldID fid_function_argc = 0;

/* org.sqlite.udf.AggregateFunction class */
static jclass clsAggregateFunction = 0;

/* org.sqlite.text.Collator class */
static jclass clsCollator = 0;

/* org.sqlite.auth.Authorizer class */
static jclass clsAuthorizer = 0;

/* org.sqlite.event.BusyHandler class */
static jclass clsBusyHandler = 0;

/* org.sqlite.event.CollationNeededHandler class */
static jclass clsCollationNeededHandler = 0;

/* org.sqlite.event.ProgressHandler class */
static jclass clsProgressHandler = 0;

/* org.sqlite.event.CommitHook class */
static jclass clsCommitHook = 0;

/* org.sqlite.event.RollbackHook class */
static jclass clsRollbackHook = 0;

/* org.sqlite.event.UpdateHook class */
static jclass clsUpdateHook = 0;

/* org.sqlite.profiler.Profiler class */
static jclass clsProfiler = 0;

/* org.sqlite.profiler.Tracer class */
static jclass clsTracer = 0;

/* org.sqlite.callback.ExecCallback class */
static jclass clsExecCallback = 0;

JNIEXPORT jint JNICALL JNI_OnLoad(JavaVM *vm, void *reserved) {
    JNIEnv *jenv = NULL;
    // require JNI ver 1.2 or later
    return ((*vm)->GetEnv(vm, (void **)&jenv, JNI_VERSION_1_2) == JNI_OK ? JNI_VERSION_1_2 : JNI_ERR);
}

/* delete references */
JNIEXPORT void JNICALL JNI_OnUnload(JavaVM *vm, void *reserved) {
    JNIEnv *jenv = NULL;
    if ((*vm)->AttachCurrentThread(vm, (void **)&jenv, 0) == JNI_OK) {
        jclass classes[]
                = {
                        clsFunction, clsAggregateFunction,
                        clsCollator,
                        clsAuthorizer,
                        clsBusyHandler, clsCollationNeededHandler, clsProgressHandler,
                        clsCommitHook, clsRollbackHook, clsUpdateHook,
                        clsProfiler, clsTracer,
                        clsExecCallback,
                  };
        int i = 0;
        int const last = sizeof(classes) / sizeof(classes[0]);
        for (; i < last; ++i) {
            if (classes[i]) {
                (*jenv)->DeleteGlobalRef(jenv, classes[i]);
                classes[i] = 0;
            }
        }
    }
}

/* convert pointer to jlong */
static jlong _jlong(void *p) {
    jvalue val;
    val.l = p;
    return val.j;
}

/* convert jlong to pointer */
static void * _voidp(jlong j) {
    jvalue val;
    val.j = j;
    return (void *)val.l;
}

/* throw new Exception */
static void JavaThrowException(JNIEnv *jenv, const char *clazz, const char *message) {
    if (jenv) {
        jclass ex;
        char *cls = (char *)clazz;
        char *msg = (char *)message;
        (*jenv)->ExceptionClear(jenv);
        ex = (*jenv)->FindClass(jenv, cls);
        if (ex) {
            (*jenv)->ThrowNew(jenv, ex, msg);
            (*jenv)->DeleteLocalRef(jenv, ex);
        }
    }
}

/* sqlite3_column_blob by byte[] */
JNIEXPORT jbyteArray JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1column_1blob_1by_1bytes(JNIEnv *jenv, jclass jcls, jlong stmt, jint col) {
    sqlite3_stmt *pStmt = (sqlite3_stmt *)_voidp(stmt);
    jsize len;
    jbyteArray result;
    jbyte *pSrc;
    const void *blob = sqlite3_column_blob(pStmt, col);

    if (!blob) {
        return NULL;
    }

    len = sqlite3_column_bytes(pStmt, col);
    result = (*jenv)->NewByteArray(jenv, len);
    if (!result) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        return NULL;
    }

    pSrc = (*jenv)->GetPrimitiveArrayCritical(jenv, result, 0);
    if (!pSrc) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        return NULL;
    }
    memcpy(pSrc, blob, len);
    (*jenv)->ReleasePrimitiveArrayCritical(jenv, result, pSrc, 0);

    return result;
}

/* sqlite3_value_blob by byte[] */
JNIEXPORT jbyteArray JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1value_1blob_1by_1bytes(JNIEnv *jenv, jclass jcls, jlong value) {
    sqlite3_value *pVal = (sqlite3_value *)_voidp(value);
    jsize len;
    jbyteArray result;
    jbyte *pSrc;
    const void *blob = sqlite3_value_blob(pVal);

    if (!blob) {
        return NULL;
    }

    len = sqlite3_value_bytes(pVal);
    result = (*jenv)->NewByteArray(jenv, len);
    if (!result) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        return NULL;
    }

    pSrc = (*jenv)->GetPrimitiveArrayCritical(jenv, result, 0);
    if (!pSrc) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        return NULL;
    }
    memcpy(pSrc, blob, len);
    (*jenv)->ReleasePrimitiveArrayCritical(jenv, result, pSrc, 0);

    return result;
}

/* sqlite3_prepare by String[] */
JNIEXPORT jint JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1prepare_1v1(JNIEnv *jenv, jclass jcls, jlong db, jstring sql, jint nByte, jlong ppStmt, jobjectArray tailArray) {
    const char *zSql = NULL;
    char *pzTail = NULL;
    int result;
    
    if (sql) {
        zSql = (const char *)(*jenv)->GetStringUTFChars(jenv, sql, 0);
        if (!zSql) {
            return 0;
        }
    }
    result = sqlite3_prepare((sqlite3 *)_voidp(db), zSql, (int)nByte, (sqlite3_stmt **)_voidp(ppStmt), (const char **)(tailArray != NULL ? &pzTail : NULL));
    if (result == SQLITE_OK
            && (tailArray != NULL && (*jenv)->GetArrayLength(jenv, tailArray) != 0)) {
        jstring tail = (*jenv)->NewStringUTF(jenv, pzTail);
        if (!tail) {
            return 0;
        }
        (*jenv)->SetObjectArrayElement(jenv, tailArray, 0, tail);
    }
    if (zSql) {
        (*jenv)->ReleaseStringUTFChars(jenv, sql, zSql);
    }
    return (jint)result;
}

/* sqlite3_prepare_v2 by String[] */
JNIEXPORT jint JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1prepare_1v2(JNIEnv *jenv, jclass jcls, jlong db, jstring sql, jint nByte, jlong ppStmt, jobjectArray tailArray) {
    const char *zSql = NULL;
    char *pzTail = NULL;
    int result;
    
    if (sql) {
        zSql = (const char *)(*jenv)->GetStringUTFChars(jenv, sql, 0);
        if (!zSql) {
            return 0;
        }
    }
    result = sqlite3_prepare_v2((sqlite3 *)_voidp(db), zSql, (int)nByte, (sqlite3_stmt **)_voidp(ppStmt), (const char **)(tailArray != NULL ? &pzTail : NULL));
    if (result == SQLITE_OK
            && (tailArray != NULL && (*jenv)->GetArrayLength(jenv, tailArray) != 0)) {
        jstring tail = (*jenv)->NewStringUTF(jenv, pzTail);
        if (!tail) {
            return 0;
        }
        (*jenv)->SetObjectArrayElement(jenv, tailArray, 0, tail);
    }
    if (zSql) {
        (*jenv)->ReleaseStringUTFChars(jenv, sql, zSql);
    }
    return (jint)result;
}

/* get FieldID */
static int GetFieldID(JNIEnv *jenv, jfieldID *pFieldID, const char *className, const char *fieldName, const char *signature) {
    jclass clazz = (*jenv)->FindClass(jenv, className);
    if (clazz) {
        *pFieldID = (*jenv)->GetFieldID(jenv, clazz, fieldName, signature);
        if (*pFieldID) {
            return 1;
        }
    }
    return 0;
}

/* load 'org.sqlite.callback.Callback' class */
static int LoadCallbackClass(JNIEnv *jenv) {
    return (fid_callback_this
            || GetFieldID(jenv, &fid_callback_this, "org/sqlite/callback/Callback", "_this", "J"));
}

/* load class */
static int LoadClass(JNIEnv *jenv, jclass *pClazz, const char* className) {
    if (*pClazz) {
        // already loaded
        return 1;
        
    } else {
        *pClazz = (*jenv)->FindClass(jenv, className);
        if (*pClazz) {
            *pClazz = (*jenv)->NewGlobalRef(jenv, *pClazz);
            return 1;
        }
    }
    return 0;
}

/* load 'org.sqlite.udf.Function' class */
static int LoadFunctionClass(JNIEnv *jenv) {
    if (fid_function_argc) {
        // alrealy loaded
        return 1;

    } else {
        if (LoadCallbackClass(jenv)
                && LoadClass(jenv, &clsFunction, "org/sqlite/udf/Function")) {
            fid_function_argc = (*jenv)->GetFieldID(jenv, clsFunction, "argc", "I");
            if (fid_function_argc) {
                return 1;
            }
        }
    }
    return 0;
}

/* load 'org.sqlite.udf.AggregateFunction' class */
static int LoadAggregateFunctionClass(JNIEnv *jenv) {
    return (clsAggregateFunction
            || (LoadCallbackClass(jenv)
                && LoadClass(jenv, &clsAggregateFunction, "org/sqlite/udf/AggregateFunction")));
}

/* load 'org.sqlite.text.Collator' class */
static int LoadCollatorClass(JNIEnv *jenv) {
    return (clsCollator
            || (LoadCallbackClass(jenv)
                && LoadClass(jenv, &clsCollator, "org/sqlite/text/Collator")));
}

/* load 'org.sqlite.auth.Authorizer' class */
static int LoadAuthorizerClass(JNIEnv *jenv) {
    return (clsAuthorizer
            || (LoadCallbackClass(jenv)
                && LoadClass(jenv, &clsAuthorizer, "org/sqlite/auth/Authorizer")));
}

/* load 'org.sqlite.event.BusyHandler' class */
static int LoadBusyHandlerClass(JNIEnv *jenv) {
    return (clsBusyHandler
            || (LoadCallbackClass(jenv)
                && LoadClass(jenv, &clsBusyHandler, "org/sqlite/event/BusyHandler")));
}

/* load 'org.sqlite.event.CollationNeededHandler' class */
static int LoadCollationNeededHandlerClass(JNIEnv *jenv) {
    return (clsCollationNeededHandler
            || (LoadCallbackClass(jenv)
                && LoadClass(jenv, &clsCollationNeededHandler, "org/sqlite/event/CollationNeededHandler")));
}

/* load 'org.sqlite.event.ProgressHandler' class */
static int LoadProgressHandlerClass(JNIEnv *jenv) {
    return (clsProgressHandler
            || (LoadCallbackClass(jenv)
                && LoadClass(jenv, &clsProgressHandler, "org/sqlite/event/ProgressHandler")));
}

/* load 'org.sqlite.event.CommitHook' class */
static int LoadCommitHookClass(JNIEnv *jenv) {
    return (clsCommitHook
            || (LoadCallbackClass(jenv)
                && LoadClass(jenv, &clsCommitHook, "org/sqlite/event/CommitHook")));
}

/* load 'org.sqlite.event.RollbackHook' class */
static int LoadRollbackHookClass(JNIEnv *jenv) {
    return (clsRollbackHook
            || (LoadCallbackClass(jenv)
                && LoadClass(jenv, &clsRollbackHook, "org/sqlite/event/RollbackHook")));
}

/* load 'org.sqlite.event.UpdateHook' class */
static int LoadUpdateHookClass(JNIEnv *jenv) {
    return (clsUpdateHook
            || (LoadCallbackClass(jenv)
                && LoadClass(jenv, &clsUpdateHook, "org/sqlite/event/UpdateHook")));
}

/* load 'org.sqlite.profiler.Profiler' class */
static int LoadProfilerClass(JNIEnv *jenv) {
    return (clsProfiler
            || (LoadCallbackClass(jenv)
                && LoadClass(jenv, &clsProfiler, "org/sqlite/profiler/Profiler")));
}

/* load 'org.sqlite.profiler.Tracer' class */
static int LoadTracerClass(JNIEnv *jenv) {
    return (clsTracer
            || (LoadCallbackClass(jenv)
                && LoadClass(jenv, &clsTracer, "org/sqlite/profiler/Tracer")));
}

/* load 'org.sqlite.callback.ExecCallback' class */
static int LoadExecCallbackClass(JNIEnv *jenv) {
    return (clsExecCallback
            || (LoadCallbackClass(jenv)
                && LoadClass(jenv, &clsExecCallback, "org/sqlite/callback/ExecCallback")));
}

/* structure for user-defined function and user-defined collating sequences */
typedef struct _JAVA_OBJECT {
    JavaVM *jvm;
    jobject obj;
} JAVA_OBJECT, *PJAVA_OBJECT;

/* allocate JAVA_OBJECT */
static PJAVA_OBJECT AllocJavaObject(JNIEnv *jenv, jobject obj) {
    PJAVA_OBJECT jobj = (PJAVA_OBJECT)calloc(1, sizeof(JAVA_OBJECT));
    if (jobj) {
        jobj->obj = (*jenv)->NewGlobalRef(jenv, obj);
        (*jenv)->GetJavaVM(jenv, &jobj->jvm);
    } else {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
    }
    return jobj;
}

/* delete JAVA_OBJECT */
static void DeleteJavaObject(JNIEnv *jenv, PJAVA_OBJECT jobj) {
    (*jenv)->DeleteGlobalRef(jenv, jobj->obj);
    free(jobj);
}

/* save JAVA_OBJECT pointer to org.sqlite.callback.Callback._this field */
static void SaveJavaObject(JNIEnv *jenv, PJAVA_OBJECT jobj) {
    (*jenv)->SetLongField(jenv, jobj->obj, fid_callback_this, _jlong(jobj));
}

/* save JAVA_OBJECT pointer to org.sqlite.callback.Callback._this field */
static void SaveOrDeleteJavaObject(int resultCode, JNIEnv *jenv, PJAVA_OBJECT jobj) {
    if (resultCode == SQLITE_OK) {
        SaveJavaObject(jenv, jobj);
    } else {
        DeleteJavaObject(jenv, jobj);
    }
}

/* load JAVA_OBJECT pointer from org.sqlite.callback.Callback._this field */
static PJAVA_OBJECT LoadJavaObject(JNIEnv *jenv, jobject obj) {
    jlong _this = (*jenv)->GetLongField(jenv, obj, fid_callback_this);
    if (_this) {
        return (PJAVA_OBJECT)_voidp(_this);
    } else {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "object is not registered yet.");
        return NULL;
    }
}

/* returns non-zero if the JAVA_OBJECT already saved */
static int IsAlreadySavedJavaObject(JNIEnv *jenv, jobject obj) {
    if ((*jenv)->GetLongField(jenv, obj, fid_callback_this)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "object is already registered.");
        return 1;
    }
    return 0;
}

/* delete JAVA_OBJECT and 0 clearing of the _this field.  */
static void DeleteCallback(JNIEnv *jenv, PJAVA_OBJECT jobj) {
    (*jenv)->SetLongField(jenv, jobj->obj, fid_callback_this, 0);
    DeleteJavaObject(jenv, jobj);
}

/* get callback name and text encoding code from org.sqlite.callback.NamedCallback fields */
static const char *GetCallbackNameAndEncoding(JNIEnv *jenv, jobject obj, jstring *jname, int *enc) {
    static jfieldID fid_name = 0;
    static jfieldID fid_enc = 0;
    if (!fid_name || !fid_enc) {
        jclass clazz = (*jenv)->FindClass(jenv, "org/sqlite/callback/NamedCallback");
        if (!clazz) {
            return NULL;
        }
        fid_name = (*jenv)->GetFieldID(jenv, clazz, "name", "Ljava/lang/String;");
        if (!fid_name) {
            return NULL;
        }
        fid_enc = (*jenv)->GetFieldID(jenv, clazz, "enc", "I");
        if (!fid_enc) {
            return NULL;
        }
    }

    // get enc value
    *enc = (int)(*jenv)->GetIntField(jenv, obj, fid_enc);
    // get name value
    *jname = (jstring)(*jenv)->GetObjectField(jenv, obj, fid_name);
    if (*jname) {
        return (const char *)(*jenv)->GetStringUTFChars(jenv, *jname, 0);
    } else {
        SWIG_JavaThrowException(jenv, SWIG_JavaNullPointerException, "name is null.");
    }
    return NULL;
}

/* AttachCurrentThread */
static int AttachCurrentThread(PJAVA_OBJECT jobj, JNIEnv **pjenv) {
    int result = (int)(*jobj->jvm)->AttachCurrentThread(jobj->jvm, (void **)pjenv, (void *)0);
    if (result != JNI_OK) {
        JavaThrowException(*pjenv, "java/lang/InternalError", "Failed AttachCurrentThread().");
    }
    return result;
}

/* User-Defined Function */
static void xFunc(sqlite3_context* ctx, int argc, sqlite3_value** value) {
    static jmethodID mid_xFunc = 0;
    JNIEnv *jenv = NULL;
    PJAVA_OBJECT jobj = (PJAVA_OBJECT)sqlite3_user_data(ctx);
    
    if (AttachCurrentThread(jobj, &jenv) != JNI_OK) {
        return;
    }

    if (!mid_xFunc) {
        mid_xFunc = (*jenv)->GetMethodID(jenv, clsFunction, "xFunc", "(JIJ)V");
        if (!mid_xFunc) {
            (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
            return;
        }
    }

    (*jenv)->CallVoidMethod(jenv, jobj->obj, mid_xFunc, _jlong(ctx), argc, _jlong(value));
    (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
}

static void xFinal(sqlite3_context* ctx) {
    static jmethodID mid_xFinal = 0;
    JNIEnv *jenv = NULL;
    PJAVA_OBJECT jobj = (PJAVA_OBJECT)sqlite3_user_data(ctx);

    if (AttachCurrentThread(jobj, &jenv) != JNI_OK) {
        return;
    }

    if (!mid_xFinal) {
        mid_xFinal = (*jenv)->GetMethodID(jenv, clsAggregateFunction, "xFinal", "(J)V");
        if (!mid_xFinal) {
            (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
            return;
        }
    }

    (*jenv)->CallVoidMethod(jenv, jobj->obj, mid_xFinal, _jlong(ctx));
    (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
}

/* sqlite3_create_function by org.sqlite.udf.Function */
JNIEXPORT jint JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1create_1function(JNIEnv *jenv, jclass jcls, jlong db, jobject func) {
    PJAVA_OBJECT jobj = NULL;
    jstring jname = NULL;
    const char *name = NULL;
    int argc = 0;
    int eTextRep = 0;
    int isAgFunc = 0;
    int result;

    if (!clsFunction && !LoadFunctionClass(jenv)) {
        // not found 'org.sqlite.udf.Function' class
        return 0;
    }

    // validate instance
    if (!(*jenv)->IsInstanceOf(jenv, func, clsFunction)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "object is not instanceof 'org.sqlite.udf.Function'.");
        return 0;
    }
    
    if (!clsAggregateFunction && !LoadAggregateFunctionClass(jenv)) {
        // not found 'org.sqlite.udf.AggregateFunction' class
        return 0;
    }

    // check already registered
    if (IsAlreadySavedJavaObject(jenv, func)) {
        return 0;
    }

    // get name and enc value
    name = GetCallbackNameAndEncoding(jenv, func, &jname, &eTextRep);
    if (!(name && jname)) {
        return 0;
    }
    
    // get org.sqlite.udf.Function.argc value
    argc = (int)(*jenv)->GetIntField(jenv, func, fid_function_argc);

    // allocate user data
    jobj = AllocJavaObject(jenv, func);
    if (!jobj) {
        return 0;
    }
    
    // func insetanceof org.sqlite.udf.AggregateFunction
    isAgFunc = (*jenv)->IsInstanceOf(jenv, func, clsAggregateFunction);
    
    // register function
    result = sqlite3_create_function((sqlite3 *)_voidp(db), name, argc, eTextRep, jobj, (isAgFunc ? NULL : &xFunc), (isAgFunc ? &xFunc : NULL), (isAgFunc ? &xFinal : NULL));
    
    // save JAVA_OBJECT to _this field
    SaveOrDeleteJavaObject(result, jenv, jobj);

    if (name) {
        (*jenv)->ReleaseStringUTFChars(jenv, jname, name);
    }

    return (jint)result;
}

/* sqlite3_create_function by org.sqlite.udf.Function */
JNIEXPORT jint JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1drop_1function(JNIEnv *jenv, jclass jcls, jlong db, jobject func) {
    PJAVA_OBJECT jobj = NULL;
    jstring jname = NULL;
    const char *name = NULL;
    int eTextRep = 0;
    int result;

    if (!clsFunction && !LoadFunctionClass(jenv)) {
        // not found 'org.sqlite.udf.Function' class
        return 0;
    }

    // validate instance
    if (!(*jenv)->IsInstanceOf(jenv, func, clsFunction)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "object is not instanceof 'org.sqlite.udf.Function'.");
        return 0;
    }

    // load JAVA_OBJECT from _this field
    jobj = LoadJavaObject(jenv, func);
    if (!jobj) {
        return 0;
    }

    // get name and enc value
    name = GetCallbackNameAndEncoding(jenv, func, &jname, &eTextRep);
    if (!(name && jname)) {
        return 0;
    }

    // unregister function
    result = sqlite3_create_function((sqlite3 *)_voidp(db), name, -1, eTextRep, NULL, NULL, NULL, NULL);
    
    // delete JAVA_OBJECT and 0 clearing of the _this field.
    if (result == SQLITE_OK) {
        DeleteCallback(jenv, jobj);
    }

    if (name) {
        (*jenv)->ReleaseStringUTFChars(jenv, jname, (const char *)name);
    }

    return (jint)result;
}

static int xCompare(void *p, int len1, const void *str1, int len2, const void *str2) {
    static jmethodID mid_compare = 0;
    PJAVA_OBJECT jobj = (PJAVA_OBJECT)p;
    JNIEnv *jenv = NULL;
    jstring source = NULL;
    jstring target = NULL;
    char *str;
    jint jresult;

    if (AttachCurrentThread(jobj, &jenv) != JNI_OK) {
        return 0;
    }

    if (!mid_compare) {
        mid_compare = (*jenv)->GetMethodID(jenv, clsCollator, "compare", "(Ljava/lang/Object;Ljava/lang/Object;)I");
        if (!mid_compare) {
            (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
            return 0;
        }
    }

    str = (char *)calloc(len1 + 1, sizeof(char));
    if (!str) {
        (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
        return 0;
    }
    strncpy(str, str1, len1);
    str[len1] = '\0';
    source = (*jenv)->NewStringUTF(jenv, (const char*)str);
    free(str);
    if (!source) {
        (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
        return 0;
    }

    str = (char *)calloc(len2 + 1, sizeof(char));
    if (!str) {
        (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
        return 0;
    }
    strncpy(str, str2, len2);
    str[len2] = '\0';
    target = (*jenv)->NewStringUTF(jenv, (const char*)str);
    free(str);
    if (!target) {
        (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
        return 0;
    }

    jresult = (*jenv)->CallIntMethod(jenv, jobj->obj, mid_compare, source, target);

    (*jobj->jvm)->DetachCurrentThread(jobj->jvm);

    return (int)jresult;
}

static int xCompare16(void *p, int len1, const void *str1, int len2, const void *str2) {
    static jmethodID mid_xCompare = 0;
    PJAVA_OBJECT jobj = (PJAVA_OBJECT)p;
    JNIEnv *jenv = NULL;
    jbyteArray source = NULL;
    jbyteArray target = NULL;
    jbyte *b;
    jint jresult;

    if (AttachCurrentThread(jobj, &jenv) != JNI_OK) {
        return 0;
    }

    if (!mid_xCompare) {
        mid_xCompare = (*jenv)->GetMethodID(jenv, clsCollator, "xCompare", "([B[B)I");
        if (!mid_xCompare) {
            (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
            return 0;
        }
    }

    source = (*jenv)->NewByteArray(jenv, (jsize)len1);
    if (!source) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
        return 0;
    }
    b = (*jenv)->GetPrimitiveArrayCritical(jenv, source, 0);
    if (!b) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
        return 0;
    }
    memcpy(b, str1, len1);
    (*jenv)->ReleasePrimitiveArrayCritical(jenv, source, b, 0);

    target = (*jenv)->NewByteArray(jenv, (jsize)len2);
    if (!target) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
        return 0;
    }
    b = (*jenv)->GetPrimitiveArrayCritical(jenv, target, 0);
    if (!b) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
        return 0;
    }
    memcpy(b, str2, len2);
    (*jenv)->ReleasePrimitiveArrayCritical(jenv, target, b, 0);

    jresult = (*jenv)->CallIntMethod(jenv, jobj->obj, mid_xCompare, source, target);

    (*jobj->jvm)->DetachCurrentThread(jobj->jvm);

    return (int)jresult;
}

/* delete org.sqlite.text.Collator object */
static void xDestroy(void *p) {
    PJAVA_OBJECT jobj = (PJAVA_OBJECT)p;
    JNIEnv *jenv = NULL;
    if (AttachCurrentThread(jobj, &jenv) == JNI_OK) {
        DeleteCallback(jenv, jobj);
    }
}

/* sqlite3_create_collation_v2 by org.sqlite.text.Collator */
JNIEXPORT jint JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1create_1collation(JNIEnv *jenv, jclass jcls, jlong db, jobject col) {
    PJAVA_OBJECT jobj = NULL;
    jstring jname = NULL;
    const char *name = NULL;
    int eTextRep = 0;
    int result;

    if (!clsCollator && !LoadCollatorClass(jenv)) {
        // not found 'org.sqlite.text.Collator' class
        return 0;
    }

    // validate instance
    if (!(*jenv)->IsInstanceOf(jenv, col, clsCollator)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "object is not instanceof 'org.sqlite.text.Collator'.");
        return 0;
    }

    // check already registered
    if (IsAlreadySavedJavaObject(jenv, col)) {
        return 0;
    }

    // get name and enc value
    name = GetCallbackNameAndEncoding(jenv, col, &jname, &eTextRep);
    if (!(name && jname)) {
        return 0;
    }

    // allocate user data
    jobj = AllocJavaObject(jenv, col);
    if (!jobj) {
        return 0;
    }
    
    // register collation sequence
    result = sqlite3_create_collation_v2((sqlite3 *)_voidp(db), name, eTextRep, jobj, (eTextRep == SQLITE_UTF8 ? &xCompare : &xCompare16), &xDestroy);

    // save JAVA_OBJECT to _this field
    SaveOrDeleteJavaObject(result, jenv, jobj);

    if (name) {
        (*jenv)->ReleaseStringUTFChars(jenv, jname, name);
    }

    return (jint)result;
}

/* sqlite3_create_collation by org.sqlite.text.Collator */
JNIEXPORT jint JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1drop_1collation(JNIEnv *jenv, jclass jcls, jlong db, jobject col) {
    PJAVA_OBJECT jobj = NULL;
    jstring jname = NULL;
    const char *name = NULL;
    int eTextRep = 0;
    int result;

    if (!clsCollator && !LoadCollatorClass(jenv)) {
        // not found 'org.sqlite.text.Collator' class
        return 0;
    }

    // validate instance
    if (!(*jenv)->IsInstanceOf(jenv, col, clsCollator)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "object is not instanceof 'org.sqlite.text.Collator'.");
        return 0;
    }

    // load JAVA_OBJECT from _this field
    jobj = LoadJavaObject(jenv, col);
    if (!jobj) {
        return 0;
    }

    // get name and enc value
    name = GetCallbackNameAndEncoding(jenv, col, &jname, &eTextRep);
    if (!(name && jname)) {
        return 0;
    }

    // unregister collation sequence
    result = sqlite3_create_collation((sqlite3 *)_voidp(db), name, eTextRep, NULL, NULL);

//    // delete JAVA_OBJECT and 0 clearing of the _this field.
//    if (result == SQLITE_OK) {
//        DeleteCallback(jenv, jobj);
//    }

    if (name) {
        (*jenv)->ReleaseStringUTFChars(jenv, jname, (const char *)name);
    }

    return (jint)result;
}

static int xAuth(void *p, int action, const char *param3, const char *param4, const char* database, const char* triggerOrView) {
    static jmethodID mid_xAuth = 0;
    PJAVA_OBJECT jobj = (PJAVA_OBJECT)p;
    JNIEnv *jenv = NULL;
    jstring jstr1 = NULL;
    jstring jstr2 = NULL;
    jstring jdatabase = NULL;
    jstring jtriggerOrView = NULL;    
    jint jresult;
    
    if (AttachCurrentThread(jobj, &jenv) != JNI_OK) {
        return 0;
    }

    if (!mid_xAuth) {
        mid_xAuth = (*jenv)->GetMethodID(jenv, clsAuthorizer, "xAuth", "(ILjava/lang/String;Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;)I");
        if (!mid_xAuth) {
            (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
            return 0;
        }
    }

    if (param3) {
        jstr1 = (*jenv)->NewStringUTF(jenv, param3);        
        if (!jstr1) {
            (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
            return 0;
        }
    }
    if (param4) {
        jstr2 = (*jenv)->NewStringUTF(jenv, param4);        
        if (!jstr2) {
            (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
            return 0;
        }
    }
    if (database) {
        jdatabase = (*jenv)->NewStringUTF(jenv, database);        
        if (!jdatabase) {
            (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
            return 0;
        }
    }
    if (triggerOrView) {
        jtriggerOrView = (*jenv)->NewStringUTF(jenv, triggerOrView);        
        if (!jtriggerOrView) {
            (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
            return 0;
        }
    }
    
    jresult = (*jenv)->CallIntMethod(jenv, jobj->obj, mid_xAuth, (jint)action, jstr1, jstr2, jdatabase, jtriggerOrView);
    
    (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
    
    return (int)jresult;
}

/* sqlite3_set_authorizer by org.sqlite.auth.Authorizer */
JNIEXPORT jint JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1set_1authorizer(JNIEnv *jenv, jclass jcls, jlong db, jobject auth) {
    PJAVA_OBJECT jobj = NULL;
    int result;

    if (!clsAuthorizer && !LoadAuthorizerClass(jenv)) {
        // not found 'org.sqlite.auth.Authorizer' class
        return 0;
    }

    // validate instance
    if (!(*jenv)->IsInstanceOf(jenv, auth, clsAuthorizer)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "object is not instanceof 'org.sqlite.auth.Authorizer'.");
        return 0;
    }

    // check already registered
    if (IsAlreadySavedJavaObject(jenv, auth)) {
        return 0;
    }

    // allocate user data
    jobj = AllocJavaObject(jenv, auth);
    if (!jobj) {
        return 0;
    }
    
    // set authorizer
    result = sqlite3_set_authorizer((sqlite3 *)_voidp(db), &xAuth, jobj);

    // save JAVA_OBJECT to _this field
    SaveOrDeleteJavaObject(result, jenv, jobj);

    return (jint)result;
}

/* sqlite3_set_authorizer by org.sqlite.auth.Authorizer */
JNIEXPORT jint JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1clear_1authorizer(JNIEnv *jenv, jclass jcls, jlong db, jobject auth) {
    PJAVA_OBJECT jobj = NULL;
    int result;

    if (!clsAuthorizer && !LoadAuthorizerClass(jenv)) {
        // not found 'org.sqlite.auth.Authorizer' class
        return 0;
    }

    // validate instance
    if (!(*jenv)->IsInstanceOf(jenv, auth, clsAuthorizer)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "object is not instanceof 'org.sqlite.auth.Authorizer'.");
        return 0;
    }

    // load JAVA_OBJECT from _this field
    jobj = LoadJavaObject(jenv, auth);
    if (!jobj) {
        return 0;
    }

    // clear authorizer
    result = sqlite3_set_authorizer((sqlite3 *)_voidp(db), NULL, NULL);

    // delete JAVA_OBJECT and 0 clearing of the _this field.
    if (result == SQLITE_OK) {
        DeleteCallback(jenv, jobj);
    }

    return (jint)result;
}

static int xBusy(void *p, int count) {
    static jmethodID mid_xBusy = 0;
    PJAVA_OBJECT jobj = (PJAVA_OBJECT)p;
    JNIEnv *jenv = NULL;
    jint jresult;
    
    if (AttachCurrentThread(jobj, &jenv) != JNI_OK) {
        return 0;
    }

    if (!mid_xBusy) {
        mid_xBusy = (*jenv)->GetMethodID(jenv, clsBusyHandler, "xBusy", "(I)I");
        if (!mid_xBusy) {
            (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
            return 0;
        }
    }

    jresult = (*jenv)->CallIntMethod(jenv, jobj->obj, mid_xBusy, (jint)count);
    
    (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
    
    return (int)jresult;
}

/* sqlite3_busy_handler by org.sqlite.event.BusyHandler */
JNIEXPORT jint JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1busy_1handler(JNIEnv *jenv, jclass jcls, jlong db, jobject busy) {
    PJAVA_OBJECT jobj = NULL;
    int result;

    if (!clsBusyHandler && !LoadBusyHandlerClass(jenv)) {
        // not found 'org.sqlite.event.BusyHandler' class
        return 0;
    }

    // validate instance
    if (!(*jenv)->IsInstanceOf(jenv, busy, clsBusyHandler)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "object is not instanceof 'org.sqlite.event.BusyHandler'.");
        return 0;
    }

    // check already registered
    if (IsAlreadySavedJavaObject(jenv, busy)) {
        return 0;
    }

    // allocate user data
    jobj = AllocJavaObject(jenv, busy);
    if (!jobj) {
        return 0;
    }
    
    // set busy handler
    result = sqlite3_busy_handler((sqlite3 *)_voidp(db), &xBusy, jobj);

    // save JAVA_OBJECT to _this field
    SaveOrDeleteJavaObject(result, jenv, jobj);

    return (jint)result;
}

/* sqlite3_busy_handler by org.sqlite.event.BusyHandler */
JNIEXPORT jint JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1clear_1busy_1handler(JNIEnv *jenv, jclass jcls, jlong db, jobject busy) {
    PJAVA_OBJECT jobj = NULL;
    int result;

    if (!clsBusyHandler && !LoadBusyHandlerClass(jenv)) {
        // not found 'org.sqlite.event.BusyHandler' class
        return 0;
    }

    // validate instance
    if (!(*jenv)->IsInstanceOf(jenv, busy, clsBusyHandler)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "object is not instanceof 'org.sqlite.event.BusyHandler'.");
        return 0;
    }

    // load JAVA_OBJECT from _this field
    jobj = LoadJavaObject(jenv, busy);
    if (!jobj) {
        return 0;
    }

    // clear busy handler
    result = sqlite3_busy_handler((sqlite3 *)_voidp(db), NULL, NULL);

    // delete JAVA_OBJECT and 0 clearing of the _this field.
    if (result == SQLITE_OK) {
        DeleteCallback(jenv, jobj);
    }

    return (jint)result;
}

static void xCollationNeeded(void *p, sqlite3 *pDb, int eTextRep, const char* name) {
    static jmethodID mid_xCollationNeeded = 0;
    PJAVA_OBJECT jobj = (PJAVA_OBJECT)p;
    JNIEnv *jenv = NULL;
    jstring jname = NULL;

    if (AttachCurrentThread(jobj, &jenv) != JNI_OK) {
        return;
    }

    if (!mid_xCollationNeeded) {
        mid_xCollationNeeded = (*jenv)->GetMethodID(jenv, clsCollationNeededHandler, "xCollationNeeded", "(JILjava/lang/String;)V");
        if (!mid_xCollationNeeded) {
            (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
            return;
        }
    }

    if (name) {
        jname = (*jenv)->NewStringUTF(jenv, name);        
        if (!jname) {
            (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
            return;
        }
    }

    (*jenv)->CallVoidMethod(jenv, jobj->obj, mid_xCollationNeeded, _jlong(pDb), (jint)eTextRep, jname);

    (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
}

/* sqlite3_collation_needed by org.sqlite.event.CollationNeededHandler */
JNIEXPORT jint JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1collation_1needed(JNIEnv *jenv, jclass jcls, jlong db, jobject needed) {
    PJAVA_OBJECT jobj = NULL;
    int result;

    if (!clsCollationNeededHandler && !LoadCollationNeededHandlerClass(jenv)) {
        // not found 'org.sqlite.event.CollationNeededHandler' class
        return 0;
    }

    // validate instance
    if (!(*jenv)->IsInstanceOf(jenv, needed, clsCollationNeededHandler)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "object is not instanceof 'org.sqlite.event.CollationNeededHandler'.");
        return 0;
    }

    // check already registered
    if (IsAlreadySavedJavaObject(jenv, needed)) {
        return 0;
    }

    // allocate user data
    jobj = AllocJavaObject(jenv, needed);
    if (!jobj) {
        return 0;
    }
    
    // set collation needed handler
    result = sqlite3_collation_needed((sqlite3 *)_voidp(db), jobj, &xCollationNeeded);

    // save JAVA_OBJECT to _this field
    SaveOrDeleteJavaObject(result, jenv, jobj);

    return (jint)result;
}

/* sqlite3_collation_needed by org.sqlite.event.CollationNeededHandler */
JNIEXPORT jint JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1clear_1collation_1needed(JNIEnv *jenv, jclass jcls, jlong db, jobject needed) {
    PJAVA_OBJECT jobj = NULL;
    int result;

    if (!clsCollationNeededHandler && !LoadCollationNeededHandlerClass(jenv)) {
        // not found 'org.sqlite.event.CollationNeededHandler' class
        return 0;
    }

    // validate instance
    if (!(*jenv)->IsInstanceOf(jenv, needed, clsCollationNeededHandler)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "object is not instanceof 'org.sqlite.event.CollationNeededHandler'.");
        return 0;
    }

    // load JAVA_OBJECT from _this field
    jobj = LoadJavaObject(jenv, needed);
    if (!jobj) {
        return 0;
    }

    // clear collation needed handler
    result = sqlite3_collation_needed((sqlite3 *)_voidp(db), NULL, NULL);

    // delete JAVA_OBJECT and 0 clearing of the _this field.
    if (result == SQLITE_OK) {
        DeleteCallback(jenv, jobj);
    }

    return (jint)result;
}

static int xProgress(void *p) {
    static jmethodID mid_xProgress = 0;
    PJAVA_OBJECT jobj = (PJAVA_OBJECT)p;
    JNIEnv *jenv = NULL;
    jint jresult;
    
    if (AttachCurrentThread(jobj, &jenv) != JNI_OK) {
        return 0;
    }

    if (!mid_xProgress) {
        mid_xProgress = (*jenv)->GetMethodID(jenv, clsProgressHandler, "xProgress", "()I");
        if (!mid_xProgress) {
            (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
            return 0;
        }
    }

    jresult = (*jenv)->CallIntMethod(jenv, jobj->obj, mid_xProgress);
    
    (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
    
    return (int)jresult;
}

/* sqlite3_progress_handler by org.sqlite.event.ProgressHandler */
JNIEXPORT void JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1progress_1handler(JNIEnv *jenv, jclass jcls, jlong db, jobject prog) {
    static jfieldID fid_progresshandler_opecodes = 0;
    jint opecodes = 0;
    PJAVA_OBJECT jobj = NULL;

    if (!clsProgressHandler) {
        if (!LoadProgressHandlerClass(jenv)) {
            // not found 'org.sqlite.event.ProgressHandler' class
            return;
        }
        fid_progresshandler_opecodes = (*jenv)->GetFieldID(jenv, clsProgressHandler, "opecodes", "I");
        if (!fid_progresshandler_opecodes) {
            return;
        }
    }

    // validate instance
    if (!(*jenv)->IsInstanceOf(jenv, prog, clsProgressHandler)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "object is not instanceof 'org.sqlite.event.ProgressHandler'.");
        return;
    }

    // check already registered
    if (IsAlreadySavedJavaObject(jenv, prog)) {
        return;
    }

    // get org.sqlite.event.ProgressHandler.opecodes value
    opecodes = (*jenv)->GetIntField(jenv, prog, fid_progresshandler_opecodes);
    
    // allocate user data
    jobj = AllocJavaObject(jenv, prog);
    if (!jobj) {
        return;
    }
    
    // set progress handler
    sqlite3_progress_handler((sqlite3 *)_voidp(db), (int)opecodes, &xProgress, jobj);

    // save JAVA_OBJECT to _this field
    SaveJavaObject(jenv, jobj);
}

/* sqlite3_progress_handler by org.sqlite.event.ProgressHandler */
JNIEXPORT void JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1clear_1progress_1handler(JNIEnv *jenv, jclass jcls, jlong db, jobject prog) {
    PJAVA_OBJECT jobj = NULL;

    if (!clsProgressHandler && !LoadProgressHandlerClass(jenv)) {
        // not found 'org.sqlite.event.ProgressHandler' class
        return;
    }

    // validate instance
    if (!(*jenv)->IsInstanceOf(jenv, prog, clsProgressHandler)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "object is not instanceof 'org.sqlite.event.ProgressHandler'.");
        return;
    }

    // load JAVA_OBJECT from _this field
    jobj = LoadJavaObject(jenv, prog);
    if (!jobj) {
        return;
    }

    // clear progress handler
    sqlite3_progress_handler((sqlite3 *)_voidp(db), 0, NULL, NULL);
    
    // delete JAVA_OBJECT and 0 clearing of the _this field.
    DeleteCallback(jenv, jobj);
}

static int xCommit(void *p) {
    static jmethodID mid_xCommit = 0;
    PJAVA_OBJECT jobj = (PJAVA_OBJECT)p;
    JNIEnv *jenv = NULL;
    jint jresult;
    
    if (AttachCurrentThread(jobj, &jenv) != JNI_OK) {
        return 0;
    }

    if (!mid_xCommit) {
        mid_xCommit = (*jenv)->GetMethodID(jenv, clsCommitHook, "xCommit", "()I");
        if (!mid_xCommit) {
            (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
            return 0;
        }
    }

    jresult = (*jenv)->CallIntMethod(jenv, jobj->obj, mid_xCommit);
    
    (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
    
    return (int)jresult;
}

/* sqlite3_commit_hook by org.sqlite.event.CommitHook */
JNIEXPORT void JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1commit_1hook(JNIEnv *jenv, jclass jcls, jlong db, jobject hook) {
    PJAVA_OBJECT jobj = NULL;
    void *pOld = NULL;

    if (!clsCommitHook && !LoadCommitHookClass(jenv)) {
        // not found 'org.sqlite.event.CommitHook' class
        return;
    }

    // validate instance
    if (!(*jenv)->IsInstanceOf(jenv, hook, clsCommitHook)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "object is not instanceof 'org.sqlite.event.CommitHook'.");
        return;
    }

    // check already registered
    if (IsAlreadySavedJavaObject(jenv, hook)) {
        return;
    }

    // allocate user data
    jobj = AllocJavaObject(jenv, hook);
    if (!jobj) {
        return;
    }
    
    // set commit hook
    pOld = sqlite3_commit_hook((sqlite3 *)_voidp(db), &xCommit, jobj);

    // save JAVA_OBJECT to _this field
    SaveJavaObject(jenv, jobj);

    // delete JAVA_OBJECT and 0 clearing of the _this field.
    if (pOld) {
        DeleteCallback(jenv, (PJAVA_OBJECT)pOld);
    }
}

/* sqlite3_commit_hook by org.sqlite.event.CommitHook */
JNIEXPORT void JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1clear_1commit_1hook(JNIEnv *jenv, jclass jcls, jlong db) {
    // clear commit hook
    void *pOld = sqlite3_commit_hook((sqlite3 *)_voidp(db), NULL, NULL);
    
    // delete JAVA_OBJECT and 0 clearing of the _this field.
    if (pOld) {
        DeleteCallback(jenv, (PJAVA_OBJECT)pOld);
    }
}

static void xRollback(void *p) {
    static jmethodID mid_xRollback = 0;
    PJAVA_OBJECT jobj = (PJAVA_OBJECT)p;
    JNIEnv *jenv = NULL;
    
    if (AttachCurrentThread(jobj, &jenv) != JNI_OK) {
        return;
    }

    if (!mid_xRollback) {
        mid_xRollback = (*jenv)->GetMethodID(jenv, clsRollbackHook, "xRollback", "()V");
        if (!mid_xRollback) {
            (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
            return;
        }
    }

    (*jenv)->CallIntMethod(jenv, jobj->obj, mid_xRollback);
    
    (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
}

/* sqlite3_rollback_hook by org.sqlite.event.RollbackHook */
JNIEXPORT void JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1rollback_1hook(JNIEnv *jenv, jclass jcls, jlong db, jobject hook) {
    PJAVA_OBJECT jobj = NULL;
    void *pOld = NULL;

    if (!clsRollbackHook && !LoadRollbackHookClass(jenv)) {
        // not found 'org.sqlite.event.RollbackHook' class
        return;
    }

    // validate instance
    if (!(*jenv)->IsInstanceOf(jenv, hook, clsRollbackHook)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "object is not instanceof 'org.sqlite.event.RollbackHook'.");
        return;
    }

    // check already registered
    if (IsAlreadySavedJavaObject(jenv, hook)) {
        return;
    }

    // allocate user data
    jobj = AllocJavaObject(jenv, hook);
    if (!jobj) {
        return;
    }
    
    // set rollback hook
    pOld = sqlite3_rollback_hook((sqlite3 *)_voidp(db), &xRollback, jobj);

    // save JAVA_OBJECT to _this field
    SaveJavaObject(jenv, jobj);

    // delete JAVA_OBJECT and 0 clearing of the _this field.
    if (pOld) {
        DeleteCallback(jenv, (PJAVA_OBJECT)pOld);
    }
}

/* sqlite3_rollback_hook by org.sqlite.event.RollbackHook */
JNIEXPORT void JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1clear_1rollback_1hook(JNIEnv *jenv, jclass jcls, jlong db) {
    // clear rollback hook
    void *pOld = sqlite3_rollback_hook((sqlite3 *)_voidp(db), NULL, NULL);
    
    // delete JAVA_OBJECT and 0 clearing of the _this field.
    if (pOld) {
        DeleteCallback(jenv, (PJAVA_OBJECT)pOld);
    }
}

static void xUpdate(void *p, int action, char const *dbName, char const *tableName, sqlite3_int64 rowid) {
    static jmethodID mid_xUpdate = 0;
    PJAVA_OBJECT jobj = (PJAVA_OBJECT)p;
    jstring jdbName = NULL;
    jstring jtableName = NULL;
    JNIEnv *jenv = NULL;
    
    if (AttachCurrentThread(jobj, &jenv) != JNI_OK) {
        return;
    }

    if (!mid_xUpdate) {
        mid_xUpdate = (*jenv)->GetMethodID(jenv, clsUpdateHook, "xUpdate", "(ILjava/lang/String;Ljava/lang/String;J)V");
        if (!mid_xUpdate) {
            (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
            return;
        }
    }

    if (dbName) {
        jdbName = (*jenv)->NewStringUTF(jenv, dbName);        
        if (!jdbName) {
            (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
            return;
        }
    }
    if (tableName) {
        jtableName = (*jenv)->NewStringUTF(jenv, tableName);        
        if (!jtableName) {
            (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
            return;
        }
    }

    (*jenv)->CallVoidMethod(jenv, jobj->obj, mid_xUpdate, (jint)action, jdbName, jtableName, (jlong)rowid);
    
    (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
}

/* sqlite3_update_hook by org.sqlite.event.UpdateHook */
JNIEXPORT void JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1update_1hook(JNIEnv *jenv, jclass jcls, jlong db, jobject hook) {
    PJAVA_OBJECT jobj = NULL;
    void *pOld = NULL;

    if (!clsUpdateHook && !LoadUpdateHookClass(jenv)) {
        // not found 'org.sqlite.event.UpdateHook' class
        return;
    }

    // validate instance
    if (!(*jenv)->IsInstanceOf(jenv, hook, clsUpdateHook)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "object is not instanceof 'org.sqlite.event.UpdateHook'.");
        return;
    }

    // check already registered
    if (IsAlreadySavedJavaObject(jenv, hook)) {
        return;
    }

    // allocate user data
    jobj = AllocJavaObject(jenv, hook);
    if (!jobj) {
        return;
    }
    
    // set update hook
    pOld = sqlite3_update_hook((sqlite3 *)_voidp(db), &xUpdate, jobj);

    // save JAVA_OBJECT to _this field
    SaveJavaObject(jenv, jobj);

    // delete JAVA_OBJECT and 0 clearing of the _this field.
    if (pOld) {
        DeleteCallback(jenv, (PJAVA_OBJECT)pOld);
    }
}

/* sqlite3_update_hook by org.sqlite.event.UpdateHook */
JNIEXPORT void JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1clear_1update_1hook(JNIEnv *jenv, jclass jcls, jlong db) {
    // clear update hook
    void *pOld = sqlite3_update_hook((sqlite3 *)_voidp(db), NULL, NULL);

    // delete JAVA_OBJECT and 0 clearing of the _this field.
    if (pOld) {
        DeleteCallback(jenv, (PJAVA_OBJECT)pOld);
    }
}

static void xProfile(void *p, char const *szSql, sqlite3_uint64 elapseTime) {
    static jmethodID mid_xProfile = 0;
    PJAVA_OBJECT jobj = (PJAVA_OBJECT)p;
    jstring jsql = NULL;
    JNIEnv *jenv = NULL;
    
    if (AttachCurrentThread(jobj, &jenv) != JNI_OK) {
        return;
    }

    if (!mid_xProfile) {
        mid_xProfile = (*jenv)->GetMethodID(jenv, clsProfiler, "xProfile", "(Ljava/lang/String;J)V");
        if (!mid_xProfile) {
            (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
            return;
        }
    }

    if (szSql) {
        jsql = (*jenv)->NewStringUTF(jenv, szSql);        
        if (!jsql) {
            (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
            return;
        }
    }

    (*jenv)->CallVoidMethod(jenv, jobj->obj, mid_xProfile, jsql, (jlong)elapseTime);
    
    (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
}

/* sqlite3_profile by org.sqlite.profiler.Profiler */
JNIEXPORT void JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1profile(JNIEnv *jenv, jclass jcls, jlong db, jobject prof) {
    PJAVA_OBJECT jobj = NULL;
    void *pOld = NULL;

    if (!clsProfiler && !LoadProfilerClass(jenv)) {
        // not found 'org.sqlite.profiler.Profiler' class
        return;
    }

    // validate instance
    if (!(*jenv)->IsInstanceOf(jenv, prof, clsProfiler)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "object is not instanceof 'org.sqlite.profiler.Profiler'.");
        return;
    }

    // check already registered
    if (IsAlreadySavedJavaObject(jenv, prof)) {
        return;
    }

    // allocate user data
    jobj = AllocJavaObject(jenv, prof);
    if (!jobj) {
        return;
    }
    
    // set profiler
    pOld = sqlite3_profile((sqlite3 *)_voidp(db), &xProfile, jobj);

    // save JAVA_OBJECT to _this field
    SaveJavaObject(jenv, jobj);

    // delete JAVA_OBJECT and 0 clearing of the _this field.
    if (pOld) {
        DeleteCallback(jenv, (PJAVA_OBJECT)pOld);
    }
}

/* sqlite3_profile by org.sqlite.profiler.Profiler */
JNIEXPORT void JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1clear_1profile(JNIEnv *jenv, jclass jcls, jlong db) {
    // clear profiler
    void *pOld = sqlite3_profile((sqlite3 *)_voidp(db), NULL, NULL);
    
    // delete JAVA_OBJECT and 0 clearing of the _this field.
    if (pOld) {
        DeleteCallback(jenv, (PJAVA_OBJECT)pOld);
    }
}

static void xTrace(void *p, char const *szSql) {
    static jmethodID mid_xTrace = 0;
    PJAVA_OBJECT jobj = (PJAVA_OBJECT)p;
    jstring jsql = NULL;
    JNIEnv *jenv = NULL;
    
    if (AttachCurrentThread(jobj, &jenv) != JNI_OK) {
        return;
    }

    if (!mid_xTrace) {
        mid_xTrace = (*jenv)->GetMethodID(jenv, clsTracer, "xTrace", "(Ljava/lang/String;)V");
        if (!mid_xTrace) {
            (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
            return;
        }
    }

    if (szSql) {
        jsql = (*jenv)->NewStringUTF(jenv, szSql);        
        if (!jsql) {
            (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
            return;
        }
    }

    (*jenv)->CallVoidMethod(jenv, jobj->obj, mid_xTrace, jsql);
    
    (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
}

/* sqlite3_trace by org.sqlite.profiler.Tracer */
JNIEXPORT void JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1trace(JNIEnv *jenv, jclass jcls, jlong db, jobject tracer) {
    PJAVA_OBJECT jobj = NULL;
    void *pOld = NULL;

    if (!clsTracer && !LoadTracerClass(jenv)) {
        // not found 'org.sqlite.profiler.Tracer' class
        return;
    }

    // validate instance
    if (!(*jenv)->IsInstanceOf(jenv, tracer, clsTracer)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "object is not instanceof 'org.sqlite.profiler.Tracer'.");
        return;
    }

    // check already registered
    if (IsAlreadySavedJavaObject(jenv, tracer)) {
        return;
    }

    // allocate user data
    jobj = AllocJavaObject(jenv, tracer);
    if (!jobj) {
        return;
    }
    
    // set tracer
    pOld = sqlite3_trace((sqlite3 *)_voidp(db), &xTrace, jobj);

    // save JAVA_OBJECT to _this field
    SaveJavaObject(jenv, jobj);

    // delete JAVA_OBJECT and 0 clearing of the _this field.
    if (pOld) {
        DeleteCallback(jenv, (PJAVA_OBJECT)pOld);
    }
}

/* sqlite3_trace by org.sqlite.profiler.Tracer */
JNIEXPORT void JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1clear_1trace(JNIEnv *jenv, jclass jcls, jlong db) {
    // clear profiler
    void *pOld = sqlite3_trace((sqlite3 *)_voidp(db), NULL, NULL);
    
    // delete JAVA_OBJECT and 0 clearing of the _this field.
    if (pOld) {
        DeleteCallback(jenv, (PJAVA_OBJECT)pOld);
    }
}

static void xDelete(void *p) {
    PJAVA_OBJECT jobj = (PJAVA_OBJECT)p;
    JNIEnv *jenv = NULL;
    if (AttachCurrentThread(jobj, &jenv) == JNI_OK) {
        DeleteJavaObject(jenv, jobj);
    }
}

/* sqlite3_set_auxdata by java.lang.Object */
JNIEXPORT void JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1set_1auxdata(JNIEnv *jenv, jclass jcls, jlong ctx, jint n, jobject obj) {
    // allocate user data
    PJAVA_OBJECT jobj = AllocJavaObject(jenv, obj);
    if (jobj) {
        // set auxiliary data
        sqlite3_set_auxdata((sqlite3_context *)_voidp(ctx), (int)n, jobj, &xDelete);
    }
}

/* sqlite3_get_auxdata by java.lang.Object */
JNIEXPORT jobject JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1get_1auxdata(JNIEnv *jenv, jclass jcls, jlong ctx, jint n) {
    // get auxiliary data
    PJAVA_OBJECT jobj = (PJAVA_OBJECT)sqlite3_get_auxdata((sqlite3_context *)_voidp(ctx), (int)n);

    return (jobj ? jobj->obj : NULL);
}

/* set sqlite3_temp_directory */
JNIEXPORT void JNICALL Java_org_sqlite_swig_SQLite3JNI_set_1sqlite3_1temp_1directory(JNIEnv *jenv, jclass jcls, jstring path) {
    const char *szPath = NULL;
    if (path) {
        szPath = (const char *)(*jenv)->GetStringUTFChars(jenv, path, 0);
        if (!szPath) {
            return;
        }
    }
    if (sqlite3_temp_directory) {
        free(sqlite3_temp_directory);
    }
    if (szPath) {
        sqlite3_temp_directory = (char *)calloc(strlen(szPath) + 1, sizeof(char));
        strcpy(sqlite3_temp_directory, szPath);
        (*jenv)->ReleaseStringUTFChars(jenv, path, szPath);
    } else {
        sqlite3_temp_directory = 0;
    }
}

static int xCallback(void *p, int cnt, char **pszValues, char **pszColumnNames) {
    static jmethodID mid_xCallback = 0;
    PJAVA_OBJECT jobj = (PJAVA_OBJECT)p;
    JNIEnv *jenv = NULL;
    jclass clsString = NULL;
    jobjectArray values;
    jobjectArray columnNames;
    int i;
    int result;

    if (AttachCurrentThread(jobj, &jenv) != JNI_OK) {
        return 0;
    }

    if (!mid_xCallback) {
        mid_xCallback = (*jenv)->GetMethodID(jenv, clsExecCallback, "xCallback", "([Ljava/lang/String;[Ljava/lang/String;)I");
        if (!mid_xCallback) {
            (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
            return 0;
        }
    }

    clsString = (*jenv)->FindClass(jenv, "java/lang/String");
    if (!clsString) {
        (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
        return 0;
    }

    values = (*jenv)->NewObjectArray(jenv, (jsize)cnt, clsString, NULL);
    columnNames = (*jenv)->NewObjectArray(jenv, (jsize)cnt, clsString, NULL);
    for (i = 0; i < cnt; ++i) {
        jstring val = (*jenv)->NewStringUTF(jenv, pszValues[i]);
        if (!val) {
            (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
            return 0;
        }
        (*jenv)->SetObjectArrayElement(jenv, values, i, val);

        val = (*jenv)->NewStringUTF(jenv, pszColumnNames[i]);
        if (!val) {
            (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
            return 0;
        }
        (*jenv)->SetObjectArrayElement(jenv, columnNames, i, val);
    }

    result = (int)(*jenv)->CallIntMethod(jenv, jobj->obj, mid_xCallback, values, columnNames);
    
    (*jobj->jvm)->DetachCurrentThread(jobj->jvm);

    return result;
}

/* sqlite3_exec by org.sqlite.callback.ExecCallback */
JNIEXPORT jint JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1exec(JNIEnv *jenv, jclass jcls, jlong db, jstring sql, jobject callback, jlong errmsg) {
    PJAVA_OBJECT jobj = NULL;
    const char *zSql = NULL;
    int result;

    if (callback) {
        if (!clsExecCallback && !LoadExecCallbackClass(jenv)) {
            // not found 'org.sqlite.callback.ExecCallback' class
            return 0;
        }

        // validate instance
        if (!(*jenv)->IsInstanceOf(jenv, callback, clsExecCallback)) {
            SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "object is not instanceof 'org.sqlite.callback.ExecCallback'.");
            return 0;
        }

        // check already registered
        if (IsAlreadySavedJavaObject(jenv, callback)) {
            return 0;
        }

        // allocate user data
        jobj = AllocJavaObject(jenv, callback);
        if (!jobj) {
            return 0;
        }

        // save JAVA_OBJECT to _this field
        SaveJavaObject(jenv, jobj);
    }

    if (sql) {
        zSql = (const char *)(*jenv)->GetStringUTFChars(jenv, sql, 0);
        if (!zSql) {
            return 0;
        }
    }

    // sqlite3_exec
    result = sqlite3_exec((sqlite3 *)_voidp(db), zSql, (callback ? &xCallback : NULL), jobj, (errmsg ? (char **)_voidp(errmsg) : NULL));

    if (zSql) {
        (*jenv)->ReleaseStringUTFChars(jenv, sql, zSql);
    }

    // delete JAVA_OBJECT and 0 clearing of the _this field.
    if (jobj) {
        DeleteCallback(jenv, jobj);
    }

    return (jint)result;
}

/* sqlite3_get_table by java.util.Collection */
JNIEXPORT jint JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1get_1table(JNIEnv *jenv, jclass jcls, jlong db, jstring sql, jobject collection, jlong errmsg) {
    static jmethodID mid_add = 0;
    jclass clsCollection = NULL;
    const char *zSql = NULL;
    char **pResult = NULL;
    int nrow;
    int ncolumn;
    int result;

    // load Collection interface
    if (!LoadClass(jenv, &clsCollection, "java/util/Collection")) {
        // not found 'java.util.Collection' interface
        return 0;
    }

    // validate instance
    if (!(*jenv)->IsInstanceOf(jenv, collection, clsCollection)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "object is not instanceof 'java.util.Collection'.");
        return 0;
    }

    if (!mid_add) {
        mid_add = (*jenv)->GetMethodID(jenv, clsCollection, "add", "(Ljava/lang/Object;)Z");
        if (!mid_add) {
            return 0;
        }
    }

    if (sql) {
        zSql = (const char *)(*jenv)->GetStringUTFChars(jenv, sql, 0);
        if (!zSql) {
            return 0;
        }
    }

    // sqlite3_get_table
    result = sqlite3_get_table((sqlite3 *)_voidp(db), zSql, &pResult, &nrow, &ncolumn, (errmsg ? (char **)_voidp(errmsg) : NULL));

    if (zSql) {
        (*jenv)->ReleaseStringUTFChars(jenv, sql, zSql);
    }

    if (result == SQLITE_OK) {
        jclass clsString = (*jenv)->FindClass(jenv, "java/lang/String");
        if (clsString) {
            int i;
            for (i = 0; i <= nrow; ++i) {
                int j;
                jobjectArray values = (*jenv)->NewObjectArray(jenv, (jsize)ncolumn, clsString, NULL);
                for (j = 0; j < ncolumn; ++j) {
                    jstring val = (*jenv)->NewStringUTF(jenv, (const char *)pResult[(i * ncolumn) + j]);
                    if (!val) {
                        break;
                    }
                    (*jenv)->SetObjectArrayElement(jenv, values, j, val);
                }

                (*jenv)->CallBooleanMethod(jenv, collection, mid_add, values);
                if ((*jenv)->ExceptionCheck(jenv) == JNI_TRUE) {
                    break;
                }
            }
        }
    }

    // free memory
    sqlite3_free_table(pResult);

    return (jint)result;
}

/* force delete Callback object */
JNIEXPORT void JNICALL Java_org_sqlite_swig_SQLite3JNI_delete_1callback(JNIEnv *jenv, jclass jcls, jobject callback) {
    if (LoadCallbackClass(jenv)) {
        PJAVA_OBJECT jobj = LoadJavaObject(jenv, callback);
        if (jobj) {
            DeleteCallback(jenv, jobj);
        }
    }
}

#ifdef __cplusplus
}
#endif
%}

/* typemap for 'const unsigned char *' to 'java.lang.String' */
%typemap(jni) const unsigned char *    "jstring"
%typemap(jtype) const unsigned char *  "String"
%typemap(jstype) const unsigned char * "String"
%typemap(out, noblock=1) const unsigned char * { if ($1) { $result = JCALL1(NewStringUTF, jenv, (const char *)$1); } }
%typemap(javaout) const unsigned char * {
    return $jnicall;
  }

/* Replace SWIG generic pointers */
%typemap(in) SWIGTYPE *, SWIGTYPE (CLASS::*) %{ $1 = ($1_ltype)_voidp($input); %}
%typemap(out) SWIGTYPE *, SWIGTYPE (CLASS::*) %{ $result = _jlong($1); %} 

/*
%pragma(java) jniclassclassmodifiers="public final class"
*/
%javamethodmodifiers sqlite3_bind_blob_by_bytes "private";
%javamethodmodifiers sqlite3_create_function "private";
%javamethodmodifiers sqlite3_drop_function "private";
%javamethodmodifiers sqlite3_create_collation "private";
%javamethodmodifiers sqlite3_drop_collation "private";
%javamethodmodifiers sqlite3_collation_needed "private";
%javamethodmodifiers sqlite3_clear_collation_needed "private";
%javamethodmodifiers sqlite3_set_authorizer "private";
%javamethodmodifiers sqlite3_clear_authorizer "private";
%javamethodmodifiers sqlite3_busy_handler "private";
%javamethodmodifiers sqlite3_clear_busy_handler "private";
%javamethodmodifiers sqlite3_progress_handler "private";
%javamethodmodifiers sqlite3_clear_progress_handler "private";
%javamethodmodifiers sqlite3_commit_hook "private";
%javamethodmodifiers sqlite3_rollback_hook "private";
%javamethodmodifiers sqlite3_update_hook "private";
%javamethodmodifiers sqlite3_profile "private";
%javamethodmodifiers sqlite3_trace "private";
%javamethodmodifiers sqlite3_exec "private";
%javamethodmodifiers sqlite3_get_table "private";
%javamethodmodifiers delete_callback "private";
%pragma(java) modulecode="
  /**
   * sqlite3** wrapper class.
   * @see <a href=\"http://sqlite.org/c3ref/sqlite3.html\">Database Connection Handle</a>
   */
  public static class SQLite3PtrPtr extends SWIGTYPE_p_p_sqlite3 {
    private boolean isDeleted = false;
    private SWIGTYPE_p_sqlite3 handle;

    /**
     * default constructor.
     */
    public SQLite3PtrPtr() {
      super(SQLite3JNI.new_p_p_sqlite3(), false);
    }

    public boolean isDeleted() {
      return isDeleted;
    }

    /**
     * Delete this object.
     * @see #delete_p_p_sqlite3(SWIGTYPE_p_p_sqlite3)
     */
    public synchronized void delete() {
      if (!isDeleted) {
        delete_p_p_sqlite3(this);
        isDeleted = true;
        handle = null;
      }
    }

    /**
     * Allocate sqlite3* value.
     * @see #get_p_sqlite3(SWIGTYPE_p_p_sqlite3)
     */
    public void allocateHandle() {
      if (!isDeleted) {
        if (handle == null) {
          handle = get_p_sqlite3(this);
        } else {
          throw new IllegalStateException(\"sqlite3 handle is already allocated.\");
        }
      } else {
        throw new IllegalStateException(\"SQLite3PtrPtr is already deleted.\");
      }
    }

    /**
     * Returns the sqlite3* value.</br>
     * It is necessary to invoke the allocateHandle() method beforehand.
     * @return the sqlite3* value
     * @see #allocateHandle()
     */
    public SWIGTYPE_p_sqlite3 getSQLite3Ptr() {
      if (!isDeleted) {
        return handle;
      }
      throw new IllegalStateException(\"SQLite3PtrPtr is already deleted.\");
    }

    /**
     * invoke delete() method.
     * @throws java.lang.Throwable
     * @see #delete()
     */
    @Override
    protected void finalize() throws Throwable {
      delete();
      super.finalize();
    }
  }

  /**
   * sqlite3_stmt** wrapper class.
   * @see <a href=\"http://sqlite.org/c3ref/stmt.html\">SQL Statement Object</a>
   */
  public static class SQLite3StmtPtrPtr extends SWIGTYPE_p_p_sqlite3_stmt {
    private boolean isDeleted = false;

    /**
     * default constructor.
     */
    public SQLite3StmtPtrPtr() {
      super(SQLite3JNI.new_p_p_sqlite3_stmt(), false);
    }

    public boolean isDeleted() {
      return isDeleted;
    }

    /**
     * Delete this object.
     * @see #delete_p_p_sqlite3_stmt(SWIGTYPE_p_p_sqlite3_stmt)
     */
    public synchronized void delete() {
      if (!isDeleted) {
        delete_p_p_sqlite3_stmt(this);
        isDeleted = true;
      }
    }

    /**
     * Returns the sqlite3_stmt* value.
     * @return the sqlite3_stmt* value.
     * @see #get_p_sqlite3_stmt(SWIGTYPE_p_p_sqlite3_stmt)
     */
    public SWIGTYPE_p_sqlite3_stmt getSQLite3StmtPtr() {
      if (!isDeleted) {
        return get_p_sqlite3_stmt(this);
      }
      throw new IllegalStateException(\"SQLite3StmtPtrPtr is already deleted.\");
    }

    /**
     * invoke delete() method.
     * @throws java.lang.Throwable
     * @see #delete()
     */
    @Override
    protected void finalize() throws Throwable {
      delete();
      super.finalize();
    }
  }

  /**
   * sqlite3_blob** wrapper class.
   * @see <a href=\"http://sqlite.org/c3ref/blob.html\">A Handle To An Open BLOB</a>
   */
  public static class SQLite3BlobPtrPtr extends SWIGTYPE_p_p_sqlite3_blob {
    private boolean isDeleted = false;

    /**
     * default constructor.
     */
    public SQLite3BlobPtrPtr() {
      super(SQLite3JNI.new_p_p_sqlite3_blob(), false);
    }

    public boolean isDeleted() {
      return isDeleted;
    }

    /**
     * Delete this object.
     * @see #delete_p_p_sqlite3_blob(SWIGTYPE_p_p_sqlite3_blob)
     */
    public synchronized void delete() {
      if (!isDeleted) {
        delete_p_p_sqlite3_blob(this);
        isDeleted = true;
      }
    }

    /**
     * Returns the sqlite3_blob* value.
     * @return the sqlite3_blob* value.
     * @see #get_p_sqlite3_blob(SWIGTYPE_p_p_sqlite3_blob)
     */
    public SWIGTYPE_p_sqlite3_blob getSQLite3BlobPtr() {
      if (!isDeleted) {
        return get_p_sqlite3_blob(this);
      }
      throw new IllegalStateException(\"SQLite3BlobPtrPtr is already deleted.\");
    }

    /**
     * invoke delete() method.
     * @throws java.lang.Throwable
     * @see #delete()
     */
    @Override
    protected void finalize() throws Throwable {
      delete();
      super.finalize();
    }
  }

  public static final int SQLITE_TRANSIENT = -1;

  /**
   * Returns in-memory filename.
   * @return \":memory:\"
   * @see <a href=\"http://sqlite.org/c3ref/open.html\">Opening A New Database Connection</a>
   */
  public static String getInMemoryFileName() {
    return \":memory:\";
  }

  /**
   * Returns date format pattern.
   * @return \"yyyy-MM-dd\"
   * @see <a href=\"http://sqlite.org/lang_createtable.html\">CREATE TABLE</a>
   */
  public static String getDateFormatPattern() {
    return \"yyyy-MM-dd\";
  }

  /**
   * Returns time format pattern.
   * @return \"HH:mm:ss\"
   * @see <a href=\"http://sqlite.org/lang_createtable.html\">CREATE TABLE</a>
   */
  public static String getTimeFormatPattern() {
    return \"HH:mm:ss\";
  }

  /**
   * Returns timestamp format pattern.
   * @return \"yyyy-MM-dd HH:mm:ss\"
   * @see <a href=\"http://sqlite.org/lang_createtable.html\">CREATE TABLE</a>
   */
  public static String getTimestampFormatPattern() {
    return \"yyyy-MM-dd HH:mm:ss\";
  }
  
  /**
   * invoke new java.text.SimpleDateFormat(String).format(Date) method.
   * @param pattern the pattern describing the date and time format
   * @param x the time value to be formatted into a time string
   * @return the formatted time string.
   * @see java.text.SimpleDateFormat#SimpleDateFormat(String)
   * @see java.text.DateFormat#format(java.util.Date)
   */
  public static String format(String pattern, java.util.Date x) {
    return new java.text.SimpleDateFormat(pattern).format(x);
  }
  
  /**
   * invoke format(getDateFormatPattern(), Date) method.
   * @param x the time value to be formatted into a time string
   * @return the formatted time string.
   * @see #getDateFormatPattern()
   * @see #format(String, java.util.Date x)
   */
  public static String format(java.sql.Date x) {
    return format(getDateFormatPattern(), x);
  }
  
  /**
   * invoke format(getTimeFormatPattern(), Date) method.
   * @param x the time value to be formatted into a time string
   * @return the formatted time string.
   * @see #getTimeFormatPattern()
   * @see #format(String, java.util.Date x)
   */
  public static String format(java.sql.Time x) {
    return format(getTimeFormatPattern(), x);
  }
  
  /**
   * invoke format(getTimestampFormatPattern(), Date) method.
   * @param x the time value to be formatted into a time string
   * @return the formatted time string.
   * @see #getTimestampFormatPattern()
   * @see #format(String, java.util.Date x)
   */
  public static String format(java.sql.Timestamp x) {
    return format(getTimestampFormatPattern(), x);
  }

  /**
   * invoke new java.text.SimpleDateFormat(String).parse(String, ParsePosition) method.
   * @param pattern the pattern describing the date and time format
   * @param x the date/time string to be parsed
   * @return value of Date#getTime()
   * @throws java.sql.SQLException
   * @see java.text.SimpleDateFormat#SimpleDateFormat(String)
   * @see java.text.SimpleDateFormat#parse(String, java.text.ParsePosition)
   * @see java.util.Date#getTime()
   */
  public static long parse(String pattern, String x) throws java.sql.SQLException {
    final java.text.DateFormat formatter = new java.text.SimpleDateFormat(pattern);
    formatter.setLenient(false);
    final java.text.ParsePosition position = new java.text.ParsePosition(0);
    final java.util.Date date = formatter.parse(x, position);
    if (position.getErrorIndex() != -1 || position.getIndex() != x.length()) {
        // parse failed
        throw new java.sql.SQLException(\"Format error.\", \"90J09\");
    }
    return date.getTime();
  }
  
  /**
   * invoke parse(getDateFormatPattern(), String) method.
   * @param x the date/time string to be parsed
   * @return value of Date#getTime()
   * @throws java.sql.SQLException
   * @see #getDateFormatPattern()
   * @see #parse(String, String)
   */
  public static long parseDate(String x) throws java.sql.SQLException {
    return parse(getDateFormatPattern(), x);
  }
  
  /**
   * invoke parse(getTimeFormatPattern(), String) method.
   * @param x the date/time string to be parsed
   * @return value of Date#getTime()
   * @throws java.sql.SQLException
   * @see #getTimeFormatPattern()
   * @see #parse(String, String)
   */
  public static long parseTime(String x) throws java.sql.SQLException {
    return parse(getTimeFormatPattern(), x);
  }
  
  /**
   * invoke parse(getTimestampFormatPattern(), String) method.
   * @param x the date/time string to be parsed
   * @return value of Date#getTime()
   * @throws java.sql.SQLException
   * @see #getTimestampFormatPattern()
   * @see #parse(String, String)
   */
  public static long parseTimestamp(String x) throws java.sql.SQLException {
    return parse(getTimestampFormatPattern(), x);
  }

  /**
   * Returns the unescaped string.
   * @param str single quoted string
   * @return unescaped string
   */
  public static String unescapeSingleQuotedString(String str) {
    if (str != null
        && str.length() > 1
        && str.charAt(0) == '\\''
        && str.charAt(str.length() - 1) == '\\'') {
      return str.substring(1, str.length() - 1).replaceAll(\"''\", \"'\");
    }
    return str;
  }

  /**
   * Returns the encoding name corresponding to the argument.
   * @param enc the text encoding code
   * @see <a href=\"http://sqlite.org/c3ref/c_any.html\">Text Encodings</a>
   */
  public static String getEncodingName(int enc) {
    switch (enc) {
      case SQLITE_UTF8:
        return \"UTF8\";

      case SQLITE_UTF16BE:
        return \"UnicodeBigUnmarked\";

      case SQLITE_UTF16LE:
        return \"UnicodeLittleUnmarked\";

      case SQLITE_UTF16:
      case SQLITE_UTF16_ALIGNED:
        return \"UTF-16\";
    }
    throw new IllegalArgumentException(\"Unsupported text encodings '\" + enc + \"'\");
  }

  /**
   * force delete Callback object.
   * @param callback the registered callback object.
   */
  public static void delete_callback(org.sqlite.callback.Callback callback) {
    SQLite3JNI.delete_callback(callback);
  }

  public static int sqlite3_exec(SWIGTYPE_p_sqlite3 db, String sql) {
    return SQLite3JNI.sqlite3_exec(SWIGTYPE_p_sqlite3.getCPtr(db), sql, null, 0);
  }

  public static int sqlite3_exec(SWIGTYPE_p_sqlite3 db, String sql, org.sqlite.callback.ExecCallback callback, SWIGTYPE_p_p_char errmsg) {
    return SQLite3JNI.sqlite3_exec(SWIGTYPE_p_sqlite3.getCPtr(db), sql, callback, SWIGTYPE_p_p_char.getCPtr(errmsg));
  }

  public static int sqlite3_get_table(SWIGTYPE_p_sqlite3 db, String sql, java.util.List<String[]> result, SWIGTYPE_p_p_char errmsg) {
    return SQLite3JNI.sqlite3_get_table(SWIGTYPE_p_sqlite3.getCPtr(db), sql, result, SWIGTYPE_p_p_char.getCPtr(errmsg));
  }

  /** invoke sqlite3_prepare_v2() method. */
  public static int sqlite3_prepare(SWIGTYPE_p_sqlite3 db, String zSql, int nBytes, SWIGTYPE_p_p_sqlite3_stmt ppStmt, String[] pzTail) {
    return SQLite3JNI.sqlite3_prepare_v2(SWIGTYPE_p_sqlite3.getCPtr(db), zSql, nBytes, SWIGTYPE_p_p_sqlite3_stmt.getCPtr(ppStmt), pzTail);
  }

  public static int sqlite3_bind_blob(SWIGTYPE_p_sqlite3_stmt stmt, int parameterIndex, byte[] val, int len) {
    return SQLite3JNI.sqlite3_bind_blob_by_bytes(SWIGTYPE_p_sqlite3_stmt.getCPtr(stmt), parameterIndex, val, len, SQLITE_TRANSIENT);
  }

  public static int sqlite3_bind_blob(SWIGTYPE_p_sqlite3_stmt stmt, int parameterIndex, SWIGTYPE_p_void val, int len) {
    return SQLite3JNI.sqlite3_bind_blob(SWIGTYPE_p_sqlite3_stmt.getCPtr(stmt), parameterIndex, SWIGTYPE_p_void.getCPtr(val), len, SQLITE_TRANSIENT);
  }

  public static int sqlite3_bind_text(SWIGTYPE_p_sqlite3_stmt stmt, int parameterIndex, String val) {
    return SQLite3JNI.sqlite3_bind_text(SWIGTYPE_p_sqlite3_stmt.getCPtr(stmt), parameterIndex, val, -1, SQLITE_TRANSIENT);
  }

  public static int sqlite3_create_function(SWIGTYPE_p_sqlite3 db, org.sqlite.udf.Function func) {
    return SQLite3JNI.sqlite3_create_function(SWIGTYPE_p_sqlite3.getCPtr(db), func);
  }

  public static int sqlite3_drop_function(SWIGTYPE_p_sqlite3 db, org.sqlite.udf.Function func) {
    return SQLite3JNI.sqlite3_drop_function(SWIGTYPE_p_sqlite3.getCPtr(db), func);
  }

  public static void sqlite3_result_blob(SWIGTYPE_p_sqlite3_context ctx, SWIGTYPE_p_void blob, int len) {
    SQLite3JNI.sqlite3_result_blob(SWIGTYPE_p_sqlite3_context.getCPtr(ctx), SWIGTYPE_p_void.getCPtr(blob), len, SQLITE_TRANSIENT);
  }

  public static void sqlite3_result_blob(SWIGTYPE_p_sqlite3_context ctx, byte[] val, int len) {
    SQLite3JNI.sqlite3_result_blob_by_bytes(SWIGTYPE_p_sqlite3_context.getCPtr(ctx), val, len, SQLITE_TRANSIENT);
  }

  public static void sqlite3_result_error(SWIGTYPE_p_sqlite3_context ctx, String message) {
    SQLite3JNI.sqlite3_result_error(SWIGTYPE_p_sqlite3_context.getCPtr(ctx), message, -1);
  }

  public static void sqlite3_result_text(SWIGTYPE_p_sqlite3_context ctx, String val) {
    SQLite3JNI.sqlite3_result_text(SWIGTYPE_p_sqlite3_context.getCPtr(ctx), val, -1, SQLITE_TRANSIENT);
  }

  public static int sqlite3_create_collation(SWIGTYPE_p_sqlite3 db, org.sqlite.text.Collator col) {
    return SQLite3JNI.sqlite3_create_collation(SWIGTYPE_p_sqlite3.getCPtr(db), col);
  }

  public static int sqlite3_drop_collation(SWIGTYPE_p_sqlite3 db, org.sqlite.text.Collator col) {
    return SQLite3JNI.sqlite3_drop_collation(SWIGTYPE_p_sqlite3.getCPtr(db), col);
  }

  public static SWIGTYPE_p_sqlite3 new_p_sqlite3(long cPtr) {
    return new SWIGTYPE_p_sqlite3(cPtr, false);
  }

  public static SWIGTYPE_p_sqlite3_context new_p_sqlite3_context(long cPtr) {
    return new SWIGTYPE_p_sqlite3_context(cPtr, false);
  }

  public static SWIGTYPE_p_p_Mem new_p_p_sqlite3_value(long cPtr) {
    return new SWIGTYPE_p_p_Mem(cPtr, false);
  }

  public static int sqlite3_collation_needed(SWIGTYPE_p_sqlite3 db, org.sqlite.event.CollationNeededHandler needed) {
    return SQLite3JNI.sqlite3_collation_needed(SWIGTYPE_p_sqlite3.getCPtr(db), needed);
  }

  public static int sqlite3_clear_collation_needed(SWIGTYPE_p_sqlite3 db, org.sqlite.event.CollationNeededHandler needed) {
    return SQLite3JNI.sqlite3_clear_collation_needed(SWIGTYPE_p_sqlite3.getCPtr(db), needed);
  }

  public static int sqlite3_set_authorizer(SWIGTYPE_p_sqlite3 db, org.sqlite.auth.Authorizer auth) {
    return SQLite3JNI.sqlite3_set_authorizer(SWIGTYPE_p_sqlite3.getCPtr(db), auth);
  }

  public static int sqlite3_clear_authorizer(SWIGTYPE_p_sqlite3 db, org.sqlite.auth.Authorizer auth) {
    return SQLite3JNI.sqlite3_clear_authorizer(SWIGTYPE_p_sqlite3.getCPtr(db), auth);
  }

  public static int sqlite3_busy_handler(SWIGTYPE_p_sqlite3 db, org.sqlite.event.BusyHandler busy) {
    return SQLite3JNI.sqlite3_busy_handler(SWIGTYPE_p_sqlite3.getCPtr(db), busy);
  }

  public static int sqlite3_clear_busy_handler(SWIGTYPE_p_sqlite3 db, org.sqlite.event.BusyHandler busy) {
    return SQLite3JNI.sqlite3_clear_busy_handler(SWIGTYPE_p_sqlite3.getCPtr(db), busy);
  }

  public static void sqlite3_progress_handler(SWIGTYPE_p_sqlite3 db, org.sqlite.event.ProgressHandler prog) {
    SQLite3JNI.sqlite3_progress_handler(SWIGTYPE_p_sqlite3.getCPtr(db), prog);
  }

  public static void sqlite3_clear_progress_handler(SWIGTYPE_p_sqlite3 db, org.sqlite.event.ProgressHandler prog) {
    SQLite3JNI.sqlite3_clear_progress_handler(SWIGTYPE_p_sqlite3.getCPtr(db), prog);
  }

  public static void sqlite3_commit_hook(SWIGTYPE_p_sqlite3 db, org.sqlite.event.CommitHook hook) {
    SQLite3JNI.sqlite3_commit_hook(SWIGTYPE_p_sqlite3.getCPtr(db), hook);
  }

  public static void sqlite3_rollback_hook(SWIGTYPE_p_sqlite3 db, org.sqlite.event.RollbackHook hook) {
    SQLite3JNI.sqlite3_rollback_hook(SWIGTYPE_p_sqlite3.getCPtr(db), hook);
  }

  public static void sqlite3_update_hook(SWIGTYPE_p_sqlite3 db, org.sqlite.event.UpdateHook hook) {
    SQLite3JNI.sqlite3_update_hook(SWIGTYPE_p_sqlite3.getCPtr(db), hook);
  }

  public static void sqlite3_profile(SWIGTYPE_p_sqlite3 db, org.sqlite.profiler.Profiler prof) {
    SQLite3JNI.sqlite3_profile(SWIGTYPE_p_sqlite3.getCPtr(db), prof);
  }

  public static void sqlite3_trace(SWIGTYPE_p_sqlite3 db, org.sqlite.profiler.Tracer tracer) {
    SQLite3JNI.sqlite3_trace(SWIGTYPE_p_sqlite3.getCPtr(db), tracer);
  }
"

/* Replace SQLite C Interface Functions by Java */
/* sqlite3_prepare by String[] */
%ignore sqlite3_prepare(sqlite3*, const char*, int, sqlite3_stmt**, const char**);
%native(sqlite3_prepare_v1) jint sqlite3_prepare_v1(sqlite3*, jstring, jint, sqlite3_stmt**, jobjectArray);

/* sqlite3_prepare_v2 by String[] */
%ignore sqlite3_prepare_v2(sqlite3*, const char*, int, sqlite3_stmt**, const char**);
%native(sqlite3_prepare_v2) jint sqlite3_prepare_v2(sqlite3*, jstring, jint, sqlite3_stmt**, jobjectArray);

/* sqlite3_create_function and sqlite3_drop_function */
%ignore sqlite3_create_function(sqlite3*, const char*, int, int, void*, void(*)(sqlite3_context*, int, sqlite3_value**), void(*)(sqlite3_context*, int, sqlite3_value**), void(*)(sqlite3_context*));
%native(sqlite3_create_function) jint sqlite3_create_function(sqlite3*, jobject);
%native(sqlite3_drop_function) jint sqlite3_drop_function(sqlite3*, jobject);

/* sqlite3_create_collation and sqlite3_drop_collation */
%ignore sqlite3_create_collation(sqlite3*, const char*, int, void*, int(*)(void*, int, const void*, int, const void*));
%ignore sqlite3_create_collation_v2(sqlite3*, const char*, int, void*, int(*)(void*, int, const void*, int, const void*), void(*)(void*));
%native(sqlite3_create_collation) jint sqlite3_create_collation(sqlite3*, jobject);
%native(sqlite3_drop_collation) jint sqlite3_drop_collation(sqlite3*, jobject);

/* sqlite3_set_authorizer and sqlite3_clear_authorizer */
%ignore sqlite3_set_authorizer(sqlite3*, int(*)(void*, int, const char*, const char*, const char*, const char*), void*);
%native(sqlite3_set_authorizer) jint sqlite3_set_authorizer(sqlite3*, jobject);
%native(sqlite3_clear_authorizer) jint sqlite3_clear_authorizer(sqlite3*, jobject);

/* sqlite3_busy_handler and sqlite3_clear_busy_handler */
%ignore sqlite3_busy_handler(sqlite3*, int(*)(void*, int), void*);
%native(sqlite3_busy_handler) jint sqlite3_busy_handler(sqlite3*, jobject);
%native(sqlite3_clear_busy_handler) jint sqlite3_clear_busy_handler(sqlite3*, jobject);

/* sqlite3_collation_needed and sqlite3_clear_collation_needed */
%ignore sqlite3_collation_needed(sqlite3*, void*, void(*)(void*, sqlite3*, int, const char*));
%native(sqlite3_collation_needed) jint sqlite3_collation_needed(sqlite3*, jobject);
%native(sqlite3_clear_collation_needed) jint sqlite3_clear_collation_needed(sqlite3*, jobject);

/* sqlite3_progress_handler and sqlite3_clear_progress_handler */
%ignore sqlite3_progress_handler(sqlite3*, int, int(*)(void*), void*);
%native(sqlite3_progress_handler) void sqlite3_progress_handler(sqlite3*, jobject);
%native(sqlite3_clear_progress_handler) void sqlite3_clear_progress_handler(sqlite3*, jobject);

/* sqlite3_commit_hook and sqlite3_clear_commit_hook */
%ignore sqlite3_commit_hook(sqlite3*, int(*)(void*), void*);
%native(sqlite3_commit_hook) void sqlite3_commit_hook(sqlite3*, jobject);
%native(sqlite3_clear_commit_hook) void sqlite3_clear_commit_hook(sqlite3*);

/* sqlite3_rollback_hook and sqlite3_clear_rollback_hook */
%ignore sqlite3_rollback_hook(sqlite3*, void(*)(void *), void*);
%native(sqlite3_rollback_hook) void sqlite3_rollback_hook(sqlite3*, jobject);
%native(sqlite3_clear_rollback_hook) void sqlite3_clear_rollback_hook(sqlite3*);

/* sqlite3_update_hook and sqlite3_clear_update_hook */
%ignore sqlite3_update_hook(sqlite3*, void(*)(void*, int, char const*, char const*, sqlite3_int64), void*);
%native(sqlite3_update_hook) void sqlite3_update_hook(sqlite3*, jobject);
%native(sqlite3_clear_update_hook) void sqlite3_clear_update_hook(sqlite3*);

/* sqlite3_profile and sqlite3_clear_profile */
%ignore sqlite3_profile(sqlite3*, void(*)(void*, const char*, sqlite3_uint64), void*);
%native(sqlite3_profile) void sqlite3_profile(sqlite3*, jobject);
%native(sqlite3_clear_profile) void sqlite3_clear_profile(sqlite3*);

/* sqlite3_trace and sqlite3_clear_trace */
%ignore sqlite3_trace(sqlite3*, void(*)(void*, const char*), void*);
%native(sqlite3_trace) void sqlite3_trace(sqlite3*, jobject);
%native(sqlite3_clear_trace) void sqlite3_clear_trace(sqlite3*);

/* sqlite3_set_auxdata and sqlite3_get_auxdata by Object */
%ignore sqlite3_set_auxdata(sqlite3_context*, int, void*, void(*)(void*));
%native(sqlite3_set_auxdata) void sqlite3_set_auxdata(sqlite3_context*, jint, jobject);
%ignore sqlite3_get_auxdata(sqlite3_context*, int);
%native(sqlite3_get_auxdata) jobject sqlite3_get_auxdata(sqlite3_context*, jint);

/* sqlite3_column_blob by byte[] */
%native(sqlite3_column_blob_by_bytes) jbyteArray sqlite3_column_blob_by_bytes(sqlite3_stmt*, jint);

/* sqlite3_value_blob by byte[] */
%native(sqlite3_value_blob_by_bytes) jbyteArray sqlite3_value_blob_by_bytes(sqlite3_value*);

/* set_sqlite3_temp_directory */
%native(set_sqlite3_temp_directory) void set_sqlite3_temp_directory(jstring);

/* sqlite3_exec with org.sqlite.callback.ExecCallback */
%ignore sqlite3_exec(sqlite3*, const char*, int(*)(void*, int, char**, char**), void*, char**);
%native(sqlite3_exec) jint sqlite3_exec(sqlite3*, jstring, jobject, jlong);

/* sqlite3_get_table by java.util.Collection */
%ignore sqlite3_get_table(sqlite3*, const char*, char***, int*, int*, char**);
%ignore sqlite3_free_table(char **);
%native(sqlite3_get_table) jint sqlite3_get_table(sqlite3*, jstring, jobject, jlong);

/* force delete Callback object */
%native(delete_callback) void delete_callback(jobject);

/* sqlite3_blob_read by byte[] */
%ignore sqlite3_blob_read(sqlite3_blob*, void*, int, int);
%rename (sqlite3_blob_read) sqlite3_blob_read_bytes;

/* sqlite3_blob_read by byte[] */
%ignore sqlite3_blob_write(sqlite3_blob*, const void*, int, int);
%rename (sqlite3_blob_write) sqlite3_blob_write_bytes;

/* ignore Obsolete Functions */
%ignore sqlite3_aggregate_count(sqlite3_context*);
%ignore sqlite3_expired(sqlite3_stmt*);
%ignore sqlite3_transfer_bindings(sqlite3_stmt*, sqlite3_stmt*);
%ignore sqlite3_global_recover(void);
%ignore sqlite3_thread_cleanup(void);
%ignore sqlite3_memory_alarm(void(*)(void*, sqlite3_int64, int), void*, sqlite3_int64);

/* Not supported functions */
%ignore sqlite3_key(sqlite3*, const void*, int);
%ignore sqlite3_rekey(sqlite3*, const void*, int);
%ignore sqlite3_complete16(const void *);
%ignore sqlite3_mprintf(const char*,...);
%ignore sqlite3_vmprintf(const char*, va_list);
%ignore sqlite3_free(char *);
%ignore sqlite3_snprintf(int, char*, const char*, ...);
%ignore sqlite3_open16(const void *, sqlite3**);
%ignore sqlite3_errmsg16(sqlite3*);
%ignore sqlite3_prepare16(sqlite3*, const void*, int, sqlite3_stmt**, const void**);
%ignore sqlite3_prepare16_v2(sqlite3*, const void*, int, sqlite3_stmt**, const void**);
%ignore sqlite3_bind_text16(sqlite3_stmt*, int, const void*, int, void(*)(void*));
%ignore sqlite3_column_name16(sqlite3_stmt*, int);
%ignore sqlite3_column_database_name16(sqlite3_stmt*, int);
%ignore sqlite3_column_table_name16(sqlite3_stmt*, int);
%ignore sqlite3_column_origin_name16(sqlite3_stmt*, int);
%ignore sqlite3_column_decltype16(sqlite3_stmt*, int);
%ignore sqlite3_column_text16(sqlite3_stmt*, int);
%ignore sqlite3_create_function16(sqlite3*, const void *, int, int, void*, void(*)(sqlite3_context*, int, sqlite3_value**), void(*)(sqlite3_context*, int, sqlite3_value**), void(*)(sqlite3_context*));
%ignore sqlite3_value_bytes16(sqlite3_value*);
%ignore sqlite3_value_bytes16(sqlite3_value*);
%ignore sqlite3_value_text16(sqlite3_value*);
%ignore sqlite3_value_text16le(sqlite3_value*);
%ignore sqlite3_value_text16be(sqlite3_value*);
%ignore sqlite3_result_error16(sqlite3_context*, const void*, int);
%ignore sqlite3_result_text16(sqlite3_context*, const void*, int, void(*)(void*));
%ignore sqlite3_result_text16le(sqlite3_context*, const void*, int, void(*)(void*));
%ignore sqlite3_result_text16be(sqlite3_context*, const void*, int, void(*)(void*));
%ignore sqlite3_create_collation16(sqlite3*, const char*, int, void*, int(*)(void*, int, const void*, int, const void*));
%ignore sqlite3_collation_needed16(sqlite3*, void*, void(*)(void*, sqlite3*, int, const void*));
%ignore sqlite3_release_memory(int);
%ignore sqlite3_soft_heap_limit(int);
%ignore sqlite3_aggregate_context(sqlite3_context*, int);
%ignore sqlite3_user_data(sqlite3_context*);
%ignore sqlite3_test_control(int, ...);

/* ignore virtual table interface and functions */
%ignore sqlite3_module;
%ignore sqlite3_vtab;
%ignore sqlite3_vtab_cursor;
%ignore sqlite3_index_info;
%ignore sqlite3_index_info_aConstraint;
%ignore sqlite3_index_info_aOrderBy;
%ignore sqlite3_index_info_aConstraintUsage;
%ignore sqlite3_create_module(sqlite3*, const char*, const sqlite3_module*, void*);
%ignore sqlite3_declare_vtab(sqlite3*, const char *);
%ignore sqlite3_overload_function(sqlite3*, const char *, int);
%ignore sqlite3_create_module_v2(sqlite3*, const char*, const sqlite3_module*, void*, void(*)(void*));

/* ignore virtual file system objects */
%ignore sqlite3_io_methods;
%ignore sqlite3_file;
%ignore sqlite3_vfs;

/* ignore variable */
%ignore sqlite3_version;
%ignore sqlite3_temp_directory;

/* ignore constants */
%ignore SQLITE_VERSION;
%ignore SQLITE_VERSION_NUMBER;
%ignore SQLITE_TEXT;

%javaconst(1);
%include "sqlite3.h"
%include "various.i"
%include "cpointer.i"

/*******************************************************************/
%exception new_p_p_sqlite3 {
    $action
    if (!result) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        return $null;
    }
}

%exception new_p_p_sqlite3_stmt {
    $action
    if (!result) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        return $null;
    }
}

%exception new_p_p_char {
    $action
    if (!result) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        return $null;
    }
}

%exception new_p_int {
    $action
    if (!result) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        return $null;
    }
}

%exception new_p_p_sqlite3_blob {
    $action
    if (!result) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        return $null;
    }
}

/* Create some functions for working with "sqlite3**" */
%ignore copy_p_p_sqlite3(sqlite3 *value);
%ignore p_p_sqlite3_assign(sqlite3 **self, sqlite3 *value);
%rename(get_p_sqlite3) p_p_sqlite3_value;
%pointer_functions(sqlite3*, p_p_sqlite3);

/* Create some functions for working with "sqlite3_stmt**" */
%ignore copy_p_p_sqlite3_stmt(sqlite3_stmt *value);
%ignore p_p_sqlite3_stmt_assign(sqlite3_stmt **self, sqlite3_stmt *value);
%rename(get_p_sqlite3_stmt) p_p_sqlite3_stmt_value;
%pointer_functions(sqlite3_stmt*, p_p_sqlite3_stmt);

/* Create some functions for working with "char*" */
%ignore copy_p_p_char(char *value);
%ignore p_p_char_assign(char **self, char *value);
%rename(get_p_char) p_p_char_value;
%pointer_functions(char*, p_p_char);

/* Create some functions for working with "int*" */
%ignore copy_p_int(int value);
%ignore p_int_assign(int *self, int value);
%rename(get_int) p_int_value;
%pointer_functions(int, p_int);

/* Create some functions for working with "sqlite3_blob**" */
%ignore copy_p_p_sqlite3_blob(sqlite3_blob *value);
%ignore p_p_sqlite3_blob_assign(sqlite3_blob **self, sqlite3_blob *value);
%rename(get_p_sqlite3_blob) p_p_sqlite3_blob_value;
%pointer_functions(sqlite3_blob*, p_p_sqlite3_blob);

%inline %{
/* sqlite3_bind_blob by byte[] */
static int sqlite3_bind_blob_by_bytes(sqlite3_stmt *pStmt, int parameterIndex, char *BYTE, int length, void (*xDel)(void*)) {
    return sqlite3_bind_blob(pStmt, parameterIndex, BYTE, length, xDel);
}

/* sqlite3_column_blob by java.sql.Blob */
static void read_blob(const void *blob, long long pos, char *BYTE, int offset, int len) {
    memcpy(&BYTE[offset], &((const char *)blob)[pos], len);
}

/* get (sqlite3_value*)sqlite3_value**[i] */
static sqlite3_value* get_p_sqlite3_value(sqlite3_value** value, int i) {
    return value[i];
}

/* sqlite3_result_blob by byte[] */
static void sqlite3_result_blob_by_bytes(sqlite3_context *pCtx, char *BYTE, int length, void (*xDel)(void*)) {
    sqlite3_result_blob(pCtx, BYTE, length, xDel);
}

/* sqlite3_errmsg by int */
static const char *sqlite3_get_errmsg(int rc) {
    return sqlite3ErrStr(rc);
}

/* sqlite3_blob_read by byte[] */
static int sqlite3_blob_read_bytes(sqlite3_blob *pBlob, char *BYTE, int n, int iOffset) {
	return sqlite3_blob_read(pBlob, BYTE, n, iOffset);
}

/* sqlite3_blob_read by byte[] */
static int sqlite3_blob_write_bytes(sqlite3_blob *pBlob, char *BYTE, int n, int iOffset) {
	return sqlite3_blob_write(pBlob, BYTE, n, iOffset);
}

%}
/*******************************************************************/
