package com.ozacc.blog.trackback;

import java.io.UnsupportedEncodingException;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * トラックバックPingを扱うユーティリティメソッド群。
 * 
 * @since 1.0
 * @author Tomohiro Otsuka
 * @version $Id: TrackBackUtils.java 201 2005-12-08 14:32:18Z otsuka $
 */
public class TrackBackUtils {

	private static Log log = LogFactory.getLog(TrackBackUtils.class);

	/**
	 * 受信したトラックバックにcharsetパラメータで文字コードがセットされていなかった場合に
	 * 使用するデフォルトの文字コード。
	 */
	private static final String DEFAULT_CHARSET = "UTF-8";

	/**
	 * コンストラクタ。通常使用する必要はありません。
	 */
	private TrackBackUtils() {}

	/**
	 * HttpServletRequestにセットされたパラメータ値からTrackBackPingオブジェクトを生成します。
	 * <p>
	 * url、excerpt、blog_name、titleの各パラメータがリクエストにセットされていない場合でも、
	 * 各値がnullのままのTrackBackPingインスタンスを生成して返します。
	 * 
	 * @param request TrackBackPingデータを持ったHttpServletRequest
	 * @return TrackBackPing
	 */
	public static TrackBackPing getTrackBackPingFromRequest(HttpServletRequest request) {
		String charset = request.getParameter("charset");
		String url = request.getParameter("url");
		String excerpt = request.getParameter("excerpt");
		String blogName = request.getParameter("blog_name");
		String title = request.getParameter("title");

		TrackBackPing ping = new TrackBackPing(title, excerpt, url, blogName);
		ping.setCharset(charset);
		return ping;
	}

	/**
	 * 指定された文字列をISO-8859-1から指定された文字コードに変換して返します。
	 * 
	 * @param charset 変換する文字コード
	 * @param str 変換する文字列
	 * @return 変換後の文字列
	 * @throws UnsupportedEncodingException 指定された文字コードがサポートされていない場合
	 */
	public static String convertCharset(String charset, String str)
																	throws UnsupportedEncodingException {
		if (str != null) {
			log.debug("文字コードを変換します。[string='" + str + "', charset='" + charset + "']");
			String result = new String(str.getBytes("8859_1"), charset);
			log.debug("文字コードを変換しました。[string='" + result + "', charset='" + charset + "']");
			return result;
		}
		return str;
	}

	/**
	 * トラックバックPingの送信成功レスポンスXMLを返します。
	 * 
	 * @return トラックバックPingの送信成功レスポンスXML
	 */
	public static String getSuccessResponse() {
		StringBuffer buf = new StringBuffer();
		buf.append("<?xml version=\"1.0\" encoding=\"" + DEFAULT_CHARSET + "\"?>\n");
		buf.append("<response>\n");
		buf.append("<error>0</error>\n");
		buf.append("</response>");
		return buf.toString();
	}

	/**
	 * トラックバックPingの送信失敗レスポンスXMLを返します。
	 * 
	 * @param message レスポンスに含めるエラーメッセージ
	 * @return トラックバックPingの送信失敗レスポンスXML
	 */
	public static String getErrorResponse(String message) {
		StringBuffer buf = new StringBuffer();
		buf.append("<?xml version=\"1.0\" encoding=\"" + DEFAULT_CHARSET + "\"?>\n");
		buf.append("<response>\n");
		buf.append("<error>1</error>\n");
		buf.append("<message>").append(message).append("</message>\n");
		buf.append("</response>");
		return buf.toString();
	}

	/**
	 * リクエストメソッドがGETで、 __mode=rss パラメータがリクエストに設定されているかどうか判定します。
	 * 
	 * @param request
	 * @return __mode=rss パラメータが設定されている場合 true
	 */
	public static boolean isRSSMode(HttpServletRequest request) {
		if ("GET".equalsIgnoreCase(request.getMethod())) {
			String mode = request.getParameter("__mode");
			return "rss".equals(mode);
		}
		return false;
	}

}