package com.ozacc.blog.ping.impl;

import java.io.IOException;
import java.util.Vector;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.ozacc.blog.ping.AsyncUpdatePingClient;
import com.ozacc.blog.ping.ConnectionException;
import com.ozacc.blog.ping.FailedUpdatePingException;
import com.ozacc.blog.ping.UpdatePing;
import com.ozacc.blog.ping.UpdatePingCallbackHandler;
import com.ozacc.blog.ping.UpdatePingClient;
import com.ozacc.blog.ping.UpdatePingException;

/**
 * UpdatePingClient インターフェースの実装クラス。
 * 
 * commons-httpclient を使用した実装です。
 * 
 * @author Tomohiro Otsuka
 * @version $Id: UpdatePingClientImpl.java 180 2005-07-22 09:26:25Z otsuka $
 */
public class UpdatePingClientImpl implements UpdatePingClient, AsyncUpdatePingClient {

	/**  更新PingのRPCメソッド名。<code>weblogUpdates.ping</code> */
	public static String PING_METHOD_NAME = "weblogUpdates.ping";

	/** デフォルトの接続タイムアウト時間、5,000ミリ秒。*/
	public static final int DEFAULT_CONNECTION_TIMEOUT = 5000;

	/** デフォルトの読込タイムアウト時間、5,000ミリ秒。*/
	public static final int DEFAULT_READ_TIMEOUT = 5000;

	public static final String DEFAULT_USER_AGENT = "ozacc-blog library";

	private static Log log = LogFactory.getLog(UpdatePingClientImpl.class);

	private int connectionTimeout = DEFAULT_CONNECTION_TIMEOUT;

	private int readTimeout = DEFAULT_READ_TIMEOUT;

	private String userAgent = DEFAULT_USER_AGENT;

	private Pattern p = Pattern
			.compile(
					"<member><name>flerror</name><value><boolean>(\\d)</boolean></value></member><member><name>message</name><value><string>(.+?)</string></value></member>",
					Pattern.CASE_INSENSITIVE);

	/**
	 * コンストラクタ。
	 */
	public UpdatePingClientImpl() {
		super();
	}

	public int getConnectionTimeout() {
		return connectionTimeout;
	}

	public void setConnectionTimeout(int connectionTimeout) {
		this.connectionTimeout = connectionTimeout;
	}

	public int getReadTimeout() {
		return readTimeout;
	}

	public void setReadTimeout(int readTimeout) {
		this.readTimeout = readTimeout;
	}

	public String getUserAgent() {
		return userAgent;
	}

	public void setUserAgent(String userAgent) {
		this.userAgent = userAgent;
	}

	/**
	 * @see com.ozacc.blog.ping.UpdatePingClient#ping(java.lang.String, java.lang.String, java.lang.String)
	 */
	public String ping(String pingUrl, String blogName, String blogUrl) throws UpdatePingException {
		log.debug("更新Pingを送信します。[pingUrl='" + pingUrl + "', blogName='" + blogName + "', blogUrl='"
				+ blogUrl + "']");
		HttpClient client = new HttpClient();
		log.debug("接続タイムアウトを設定します。[" + connectionTimeout + "]");
		client.setConnectionTimeout(connectionTimeout);
		log.debug("読込タイムアウトを設定します。[" + readTimeout + "]");
		client.setTimeout(readTimeout);

		PostMethod method = new PostMethod(pingUrl);
		method.addRequestHeader("Content-Type", "text/xml; charset=UTF-8");
		method.addRequestHeader("User-Agent", userAgent);
		method.setRequestBody(createRequestBody(blogName, blogUrl));
		try {
			client.executeMethod(method);
			String response = method.getResponseBodyAsString();
			return parseResponse(response);
		} catch (HttpException e) {
			throw new UpdatePingException("サーバがエラーを返しました。", e);
		} catch (IOException e) {
			throw new ConnectionException(
					"サーバとの接続に失敗しました。サーバのURLが正しいか、またサーバがRPCをサポートしているか確認してください。", e);
		} finally {
			log.debug("HTTP接続を閉じます。");
			method.releaseConnection();
		}
	}

	private String parseResponse(String response) {
		log.debug("更新Pingレスポンス\n" + response);
		response = response.replaceAll("[\r\n]", "");
		Matcher m = p.matcher(response);
		if (m.find()) {
			String flag = m.group(1);
			String message = m.group(2);
			if ("0".equals(flag)) {
				return message;
			}
			throw new FailedUpdatePingException(message);
		}
		throw new UpdatePingException("サーバがエラーを返しました。");
	}

	private String createRequestBody(String blogName, String blogUrl) {
		StringBuffer buf = new StringBuffer();
		buf.append("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
		buf.append("<methodCall>");
		buf.append("<methodName>").append(PING_METHOD_NAME).append("</methodName>");
		buf.append("<params>");
		buf.append("<param><value>").append(blogName).append("</value></param>");
		buf.append("<param><value>").append(blogUrl).append("</value></param>");
		buf.append("</params>");
		buf.append("</methodCall>");
		return buf.toString();
	}

	/**
	 * @see com.ozacc.blog.ping.UpdatePingClient#ping(java.lang.String, com.ozacc.blog.ping.UpdatePing)
	 */
	public String ping(String pingUrl, UpdatePing ping) throws UpdatePingException {
		Vector v = ping.getParameters();
		return ping(pingUrl, (String)v.get(0), (String)v.get(1));
	}

	/**
	 * @see com.ozacc.blog.ping.AsyncUpdatePingClient#ping(java.lang.String, com.ozacc.blog.ping.UpdatePing, com.ozacc.blog.ping.UpdatePingCallbackHandler)
	 */
	public void ping(String pingUrl, UpdatePing ping, UpdatePingCallbackHandler handler) {
		Vector v = ping.getParameters();
		ping(pingUrl, (String)v.get(0), (String)v.get(1), handler);
	}

	/**
	 * @see com.ozacc.blog.ping.AsyncUpdatePingClient#ping(java.lang.String[], com.ozacc.blog.ping.UpdatePing, com.ozacc.blog.ping.UpdatePingCallbackHandler)
	 */
	public void ping(String[] pingUrls, UpdatePing ping, UpdatePingCallbackHandler handler) {
		for (int i = 0; i < pingUrls.length; i++) {
			String pingUrl = pingUrls[i];
			ping(pingUrl, ping, handler);
		}
	}

	/**
	 * @see com.ozacc.blog.ping.AsyncUpdatePingClient#ping(java.lang.String, java.lang.String, java.lang.String, com.ozacc.blog.ping.UpdatePingCallbackHandler)
	 */
	public void ping(final String pingUrl, final String blogName, final String blogUrl,
						final UpdatePingCallbackHandler handler) {
		Thread t = new Thread() {

			public void run() {
				try {
					String message = ping(pingUrl, blogName, blogUrl);
					handler.handleResult(message, pingUrl);
				} catch (UpdatePingException e) {
					handler.handleError(e, pingUrl);
				}
			}
		};
		t.start();
	}
}