/*
 * Copyright (c) 2008 TAKAHASHI Tomoyuki. All rights reserved.
 */
package jp.sourceforge.sowba.dtd2java;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;
import java.util.StringTokenizer;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import jp.sourceforge.sowba.dtd2java.dtd.AttributeObject;
import jp.sourceforge.sowba.dtd2java.dtd.ContextObject;
import jp.sourceforge.sowba.dtd2java.dtd.ElementObject;

import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.VelocityEngine;

/**
 * dtd2javãCNXłB<br/>
 *
 * @author TAKAHASHI Tomoyuki
 *
 * <table border="1" cellpadding="3" cellspacing="0">
 * <tr bgcolor="#ccccff"><th>Ver.</th><th>t</th><th>ڍ</th></tr>
 * <tr><td>0.01</td><td>2008/01/05</td><td>VK쐬</td>
 * </table>
 */
public class Dtd2Java implements Runnable {

    private static final Pattern C_P1 = Pattern.compile("<!ELEMENT\\s+([^\\s]+)\\s+(.*)>");

    private static final Pattern C_P2 = Pattern.compile("<!ATTLIST\\s+([^\\s]+)\\s+(.*)>");

    private static final Pattern C_P3 = Pattern.compile("(.*)(\\()(.+?)(\\))(.*)");

    public static void main(String[] args) {
        try {
            Dtd2Java me = new Dtd2Java();
            Recipe recipe = new Recipe();
            for (int i = 0; i < args.length; i++) {
                if (args[i].equalsIgnoreCase("-i") && i <= args.length - 2) {
                    recipe.setDtdPath(args[++i]);
                } else if (args[i].equalsIgnoreCase("-d") && i <= args.length - 2) {
                    recipe.setOutputDir(args[++i]);
                } else if (args[i].equalsIgnoreCase("-p") && i <= args.length - 2) {
                    recipe.setPackageName(args[++i]);
                } else if (args[i].equalsIgnoreCase("-t") && i <= args.length - 2) {
                    recipe.setTemplateDir(args[++i]);
                } else if (args[i].equalsIgnoreCase("-bv") && i <= args.length - 2) {
                    recipe.setBeansVmName(args[++i]);
                } else if (args[i].equalsIgnoreCase("-fv") && i <= args.length - 2) {
                    recipe.setFactoryVmName(args[++i]);
                } else if (args[i].equalsIgnoreCase("-f") && i <= args.length - 2) {
                    recipe.setFactoryName(args[++i]);
                } else if (args[i].equalsIgnoreCase("-k") && i <= args.length - 2) {
                    String expression = args[++i];
                    String[] tokens = expression.split("=");
                    if (tokens.length != 2) {
                        recipe.setDtdPath(null);
                        break;
                    }
                    recipe.getKeyMap().put(tokens[0], tokens[1]);
                } else if (args[i].equalsIgnoreCase("-ra") && i <= args.length - 2) {
                    String expression = args[++i];
                    String[] tokens = expression.split("=");
                    if (tokens.length != 2) {
                        recipe.setDtdPath(null);
                        break;
                    }
                    recipe.getReplacingAttributeMap().put(tokens[0], tokens[1]);
                } else if (args[i].equalsIgnoreCase("-cp") && i <= args.length - 2) {
                    recipe.setClassPrefix(args[++i]);
                } else if (args[i].equalsIgnoreCase("-cs") && i <= args.length - 2) {
                    recipe.setClassSuffix(args[++i]);
                } else if (args[i].equalsIgnoreCase("-cache")) {
                    recipe.setCache(true);
                } else if (args[i].equalsIgnoreCase("-generics")){
                    recipe.setGenerics(true);
                }
            }

            me.setRecipe(recipe);
            if (!me.validateParameter()) {
                System.out.println("usage:");
                System.out.println("java jp.sourceforge.sowba.dtd2java.Dtd2Java -i DTDpX -d \[XpX -p pbP[W");
                System.out.println("  -t ev[gfBNg -bv r[ev[g -fv t@Ngev[g");
                System.out.println("  [-f t@Ng] [-k vf=]...");
                System.out.println("");
                System.out.println("    -i  DTD̃tpXw肵ĂB");
                System.out.println("");
                System.out.println("    -d  Java\[X̏o͐fBNg̃pXw肵ĂB");
                System.out.println("        Ŏw肵fBNgȉɃpbP[W̃fBNg쐬܂B");
                System.out.println("");
                System.out.println("    -p  pbP[Ww肵ĂB");
                System.out.println("");
                System.out.println("    -t  Velocityev[gi[fBNg̃pXw肵ĂB");
                System.out.println("");
                System.out.println("    -bv r[𐶐ׂ̃ev[gt@Cw肵ĂB");
                System.out.println("");
                System.out.println("    -fv t@Ng𐶐ׂ̃ev[gt@Cw肵ĂB");
                System.out.println("");
                System.out.println("    -f  t@Ng̃NXw肵ĂB");
                System.out.println("        pbP[W͕svłB");
                System.out.println("        w肳Ȃꍇ́AXmlObjectFactoryƂ܂B");
                System.out.println("");
                System.out.println("    -k  vf̃L[w肵ĂB`\łB");
                System.out.println("");
                System.out.println("    -ra ɑΉvpeBw肵ĂB`\łB");
                System.out.println("");
                System.out.println("    -cp r[NX̃vtBbNXłB");
                System.out.println("");
                System.out.println("    -cs r[NX̃TtBbNXłB");
                System.out.println("");
                System.out.println("    -cache r[LbV悤ɂ܂B");
                System.out.println("");
                System.out.println("    -generics \[XR[hgenericsg悤ɂ܂B");
                return;
            }

            me.run();

            Dtd2JavaUtil util = new Dtd2JavaUtil(recipe);
            System.out.println("NX𐶐܂BpX[" + recipe.getOutputDir() + "]");
            ElementObject[] elems = me.getContext().getElements();
            for (int i = 0; i < elems.length; i++) {
                System.out.println("    " + recipe.getPackageName() + "." + util.getClassName(elems[i].getName()));
            }
            System.out.println("    " + recipe.getPackageName() + "." + recipe.getFactoryName());
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private ContextObject m_context;

    private Recipe m_recipe;

    public Dtd2Java() {}

    private void analyzeChildElements(TreeSet<String> childmap, ContextObject context, String text) {
        int nameidx = 0;
        Map<String, String> map = new HashMap<String, String>();

        while (true) {
            Matcher m1 = C_P3.matcher(text);
            if (m1.matches()) {
                String name = "__P__" + nameidx;
                nameidx++;
                text = m1.group(1) + name + m1.group(5);
                String v = m1.group(3);
                map.put(name, v);
            } else {
                break;
            }
        }

        analyzeChildElementsSub01(childmap, context, text, map, false);

    }

    private void analyzeChildElementsSub01(TreeSet<String> childmap, ContextObject context, String s, Map placeHolderMap, boolean forceMultiple) {
        Pattern p1 = Pattern.compile("(.*)([\\*\\?\\+])");
        String[] tokens = s.split("[,|]");
        for (int i = 0; i < tokens.length; i++) {
            String token = tokens[i].trim();
            String indicator = (forceMultiple ? "*" : "");
            Matcher m1 = p1.matcher(token);
            if (m1.matches()) {
                token = m1.group(1);
                indicator = m1.group(2);
                if (indicator.matches("[\\*\\+]")) {
                    indicator = "*";
                }
            }
            String value = (String)placeHolderMap.get(token);
            if (value != null) {
                // ċA
                analyzeChildElementsSub01(childmap, context, value, placeHolderMap, (indicator.equals("*")));
                continue;
            }

            childmap.add(token);
            ElementObject elem = context.getElement(token);
            if (indicator.equals("*")) {
                elem.setIndicator("*");
            }
        }
    }

    private void analyzeDtd(ContextObject context, String dtdpath) throws FileNotFoundException, IOException {
        String content = createContent(dtdpath);
        BufferedReader reader = new BufferedReader(new StringReader(content));
        String line;
        while ((line = reader.readLine()) != null) {
            line = line.trim();
            Matcher m1 = C_P1.matcher(line);
            if (m1.matches()) {
                // vf̏ꍇ
                String elementDef = m1.group(1);
                String childrenDef = m1.group(2);
                ElementObject elem = context.getElement(elementDef);
                String ct = childrenDef.trim();
                if (ct.equalsIgnoreCase("EMPTY")) {
                    // qvfȂ
                } else if (ct.matches(".*\\(#PCDATA\\).*") || ct.equalsIgnoreCase("ANY")) {
                    // {fBvf
                    elem.setHasBody(true);
                } else {
                    TreeSet<String> childmap = new TreeSet<String>();
                    analyzeChildElements(childmap, context, childrenDef);
                    for (Iterator k = childmap.iterator(); k.hasNext();) {
                        ElementObject child = context.getElement((String)k.next());
                        elem.addChild(child.getName(), child);
                    }
                }
                continue;
            }
            Matcher m2 = C_P2.matcher(line);
            if (m2.matches()) {
                // ̏ꍇ
                String elementDef = m2.group(1);
                String attlistDef = m2.group(2);
                ElementObject element = context.getElement(elementDef);
                attlistDef = attlistDef.replaceAll("\\\".*?\\\"", "_DQUOT_");
                attlistDef = attlistDef.replaceAll("\\(.*?\\)", "_PAREN_");
                StringTokenizer tokenizer = new StringTokenizer(attlistDef, " ");
                while (tokenizer.hasMoreTokens()) {
                    String token = tokenizer.nextToken();
                    if (token.equalsIgnoreCase("CDATA") || token.equalsIgnoreCase("#REQUIRED") || token.equalsIgnoreCase("#FIXED") || token.equalsIgnoreCase("#IMPLIED") || token.equalsIgnoreCase("_DQUOT_") || token.equalsIgnoreCase("_PAREN_")) {
                        // ȂɂȂ
                    } else {
                        element.addAttribute(token, new AttributeObject(elementDef, token));
                    }
                }
                continue;
            }
        }
        reader.close();
    }

    private String createContent(String dtd) throws FileNotFoundException, IOException {
        File f = new File(dtd);
        BufferedReader reader = new BufferedReader(new FileReader(f));
        StringWriter sw = new StringWriter();
        PrintWriter writer = new PrintWriter(sw);
        int data;
        while ((data = reader.read()) != -1) {
            writer.write(data);
        }
        writer.close();
        reader.close();
        String s = sw.toString();
        s = s.replaceAll("\\s+", " ");
        s = s.replaceAll("<!--.*?-->", "");
        s = s.replaceAll("\\s+", " ");
        s = s.replaceAll(">\\s+<", "><");
        s = s.trim();
        s = s.replaceAll(">", ">\n");
        return s;
    }

    private void generateBeans(ContextObject context, VelocityEngine engine, VelocityContext vc, Dtd2JavaUtil util) throws IOException, Exception {
        String newdir = this.getRecipe().getOutputDir() + "/" + this.getRecipe().getPackageName();
        newdir = newdir.replaceAll("\\.", "/");
        newdir = newdir.replaceAll("\\\\", "/");
        ElementObject[] elems = context.getElements();
        for (int i = 0; i < elems.length; i++) {
            ElementObject elem = elems[i];
            String classname = util.getClassName(elem.getName());
            File newfile = new File(newdir + "/" + classname + ".java");
            if (!newfile.getParentFile().mkdirs()) {
                if (!newfile.getParentFile().exists()) {
                    throw new RuntimeException("pX̐Ɏs܂B[" + newfile.getParent() + "]");
                }
            }
            BufferedWriter writer = new BufferedWriter(new FileWriter(newfile));
            vc.put("element", elem);
            engine.mergeTemplate(this.getRecipe().getBeansVmName(), vc, writer);
            writer.close();
        }
    }

    private void generateFactory(ContextObject context, VelocityEngine engine, VelocityContext vc, Dtd2JavaUtil util) throws IOException, Exception {
        String newdir = this.getRecipe().getOutputDir() + "/" + this.getRecipe().getPackageName();
        newdir = newdir.replaceAll("\\.", "/");
        newdir = newdir.replaceAll("\\\\", "/");
        File newfile = new File(newdir + "/" + context.getFactoryName() + ".java");
        if (!newfile.getParentFile().mkdirs()) {
            if (!newfile.getParentFile().exists()) {
                throw new RuntimeException("pX̐Ɏs܂B[" + newfile.getParent() + "]");
            }
        }
        BufferedWriter writer = new BufferedWriter(new FileWriter(newfile));
        engine.mergeTemplate(this.getRecipe().getFactoryVmName(), vc, writer);
        writer.close();
    }

    public ContextObject getContext() {
        return m_context;
    }

    public Recipe getRecipe() {
        return m_recipe;
    }

    public void run() {
        try {
            Properties veloprop = new Properties();
            veloprop.setProperty("runtime.log.logsystem.class", "org.apache.velocity.runtime.log.NullLogChute");
            veloprop.setProperty("input.encoding", "Windows-31J");
            veloprop.setProperty("output.encoding", "Windows-31J");
            veloprop.setProperty("resource.loader", "FILE");
            veloprop.setProperty("FILE.resource.loader.class", "org.apache.velocity.runtime.resource.loader.FileResourceLoader");
            veloprop.setProperty("FILE.resource.loader.cache", "false");
            veloprop.setProperty("FILE.resource.loader.path", this.getRecipe().getTemplateDir());

            m_context = new ContextObject();
            m_context.setPackageName(this.getRecipe().getPackageName());
            if (this.getRecipe().getFactoryName() == null || this.getRecipe().getFactoryName().length() == 0) {
                this.getRecipe().setFactoryName("XmlObjectFactory");
            }
            m_context.setFactoryName(this.getRecipe().getFactoryName());
            for (Iterator i = this.getRecipe().getKeyMap().keySet().iterator(); i.hasNext();) {
                String key = (String)i.next();
                String value = (String)this.getRecipe().getKeyMap().get(key);
                m_context.addKey(key, value);
            }
            this.analyzeDtd(m_context, this.getRecipe().getDtdPath());

            VelocityEngine engine = new VelocityEngine();
            engine.init(veloprop);
            VelocityContext vc = new VelocityContext();
            Dtd2JavaUtil util = new Dtd2JavaUtil(this.getRecipe());
            vc.put("context", m_context);
            vc.put("recipe", this.getRecipe());
            vc.put("util", util);

            this.generateBeans(m_context, engine, vc, util);
            this.generateFactory(m_context, engine, vc, util);

        } catch (RuntimeException e) {
            throw e;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public void setRecipe(Recipe recipe) {
        m_recipe = recipe;
    }

    public boolean validateParameter() {
        boolean b = true;
        b &= this.getRecipe().getBeansVmName() != null;
        b &= this.getRecipe().getDtdPath() != null;
        b &= this.getRecipe().getFactoryVmName() != null;
        b &= this.getRecipe().getOutputDir() != null;
        b &= this.getRecipe().getPackageName() != null;
        b &= this.getRecipe().getTemplateDir() != null;

        return b;
    }

}
