package jp.sourceforge.sos.cytoq.colorView;

import java.awt.Image;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.awt.geom.Point2D.Double;
import java.util.Arrays;

import javax.swing.JComponent;

import jp.sourceforge.sos.cytoq.AbstractAlgorithm;
import jp.sourceforge.sos.cytoq.colorFinal.ModelColorFinal;
import jp.sourceforge.sos.cytoq.message.IAnimation;
import jp.sourceforge.sos.cytoq.message.IInitSection;
import jp.sourceforge.sos.framework.model.AbstractModel;
import jp.sourceforge.sos.framework.pattern.IObserver;
import jp.sourceforge.sos.lib.color.ColorModel;
import jp.sourceforge.sos.lib.image.ImagePixelMediator;
import jp.sourceforge.sos.lib.math.MathMatrix;
import jp.sourceforge.sos.lib.math.MathVector;
import jp.sourceforge.sos.lib.math.Statistics;

/*
 * ModelColorView.java
 *
 * Created on 2004/08/02, 0:17
 */

/**
 * @author Scientific Open Source Project (Gaku Tanaka)
 */
public class ModelColorView extends AbstractModel<ModelColorFinal> implements IInitSection, IAnimation {
	private double[][] viewMx = null;
	
	private ColoredVector colorPoints;
	
	private ColoredVector unitRects;
	
	private Rectangle2D.Double canvasView;

	private Double ratio;
	
	private AbstractCanvas[] canvases;

	public ModelColorView() {
		colorPoints = new ColoredVector();
		unitRects = new ColoredVector();
		
		canvasView = new Rectangle2D.Double();
		ratio = new Point2D.Double();
		
		canvases = new AbstractCanvas[2];
		canvases[0] = new CanvasDistribution();
		canvases[1] = new CanvasSegmentation();
		for (AbstractCanvas c: canvases){
			c.setModel(this);
		}
	}

	@Override
	protected void initModel() {
		modelParent.addObserverInitColor((IObserver<ColorModel>) canvases[0]);
	}

	Image getImage(int w, int h) {
		int[] pixels = new int[w * h];
		Arrays.fill(pixels, 0xff000000);

		updatePixels(w, h, pixels);
		return ImagePixelMediator.toImage(pixels, w, h);
	}

	/**
	 * @param w
	 * @param h
	 * @param pixels
	 */
	void updatePixels(int w, int h, int[] pixels) {
		double rw = (w - 1) / canvasView.width;
		double rh = (h - 1) / canvasView.height;
		ratio.setLocation(rw, rh);
		
		double[][] location = colorPoints.getLocation();
		int[] color = colorPoints.getColor();
		for (int in = 0; in < location.length; in++) {
			int x = (int) (ratio.x * location[in][0]);
			int y = (int) (ratio.y * location[in][1]);
			int index = x + y * w;
			pixels[index] = (0xff000000) + color[in];
		}
	}

	int[] convertRect(int index) {
		double[] p = unitRects.getLocation(index);
		double x = MathVector.dot(viewMx[0], p) - canvasView.x;
		double y = MathVector.dot(viewMx[1], p) - canvasView.y;

		int[] result = new int[2];
		result[0] = (int) (ratio.x * x);
		result[1] = (int) (ratio.y * y);
		return result;
	}

	int getColorNumber() {
		return modelParent.getColorNumber();
	}

	int getHeight(int width) {
		return (int) (width * canvasView.height / canvasView.width);
	}

	void setPointColor(int[] color) {
		colorPoints.setColor(color);
	}

	public void setRectLocation(double[][] u) {
		unitRects.setLocation(u);
		modelParent.setColorData(u);
	}

	int[] createColors(int[] result, int[] labels) {
		int[] colors = modelParent.getColor();
		for (int i = 0; i < result.length; i++) {
			result[i] = colors[labels[i]];
		}
		return result;
	}

	public void observeInitSection(AbstractAlgorithm changed) {
		setPointLocation(changed.getInput());
		setPointColor(changed.getIntColors());
		
		setRectLocation(changed.getCopyOfUnitD());
		
		for (AbstractCanvas view: canvases){
			view.observeInitSection(changed);
		}
		
	}

	private void setPointLocation(double[][] points) {
		Statistics stat = new Statistics(points);
		double[][] covMx = stat.getCovariance();
		viewMx = new double[covMx.length][covMx.length];
		MathMatrix.eigenJacobi(covMx, viewMx);

		// Convert to the 2D feature space
		colorPoints.createLocation(points.length);
		double[][] location = colorPoints.getLocation();
		for (int in = 0; in < points.length; in++) {
			location[in][0] = MathVector.dot(viewMx[0], points[in]);
			location[in][1] = MathVector.dot(viewMx[1], points[in]);
		}
		// find min and max in the feature space
		Statistics statLoc = new Statistics(location);
		double[] minPoints = statLoc.getMin();
		canvasView.x = minPoints[0];
		canvasView.y = minPoints[1];
		for (int in = 0; in < points.length; in++) {
			location[in][0] -= canvasView.x;
			location[in][1] -= canvasView.y;
		}
		// Calculate the size of the feature space
		double[] maxPoints = statLoc.getMax();
		canvasView.width = maxPoints[0] - canvasView.x;
		canvasView.height = maxPoints[1] - canvasView.y;
	}

	public void observeAnimation(AbstractAlgorithm changed) {
		setRectLocation(changed.getCopyOfUnitD());
		unitRects.setColor(modelParent.getColor());
		
		for (AbstractCanvas view: canvases){
			view.observeAnimation(changed);
		}
	}

	public ColoredVector getUnitRects() {
		return unitRects;
	}

	public JComponent getCanvases(int i) {
		return canvases[i];
	}

}
