package jp.sourceforge.sos.cytoq.analysis.imageColor;

import java.awt.Color;
import java.awt.Image;
import java.util.Arrays;

import jp.sourceforge.sos.cytoq.analysis.IView;
import jp.sourceforge.sos.cytoq.analysis.ModelAnalysis;
import jp.sourceforge.sos.cytoq.main.IModel;
import sos.image.ImagePixelMediator;
import sos.math.MathMatrix;
import sos.math.MathVector;
import sos.math.Statistics;
import sos.pac.IPACAbstraction;

/*
 * ModelImageColor.java
 *
 * Created on 2004/08/02, 0:17
 */

/**
 * @author Scientific Open Source Project (Gaku Tanaka)
 */
public class ModelImageColor implements IModel, IView{
	private double[][]					viewMx			= null;

	private double[][]					pointLocation	= null;

	private int[]						colorsForPoints	= null;

	private double						minX;

	private double						minY;

	private double						colorWidth;

	private double						colorHeight;

	private double						rw;

	private double						rh;

	/**
	 * @uml.property name="rectLocation"
	 */
	private double[][]					rectLocation;

	private ModelAnalysis				model;

	/**
	 */
	public ModelImageColor() {
	}

	void setLabelColors(Color[] colors) {
		int[] labels = model.getColorLabel();
		for (int in = 0; in<labels.length; in++) {
			colorsForPoints[in] = colors[labels[in]].getRGB();
		}
	}
	
	

	Image getImage(int w, int h) {
		rw = (w-1)/colorWidth;
		rh = (h-1)/colorHeight;

		int[] pixels = new int[w*h];
		Arrays.fill(pixels, 0xff000000);
		int x, y;
		for (int in = 0; in<pointLocation.length; in++) {
			x = (int) (rw*pointLocation[in][0]);
			y = (int) (rh*pointLocation[in][1]);
			pixels[x+y*w] = (0xff000000)+colorsForPoints[in];
		}

		return ImagePixelMediator.convertToImage(pixels, w, h);
	}

	int[] convertRect(int index) {
		double[] p = rectLocation[index];
		double x = MathVector.dot(viewMx[0], p)-minX;
		double y = MathVector.dot(viewMx[1], p)-minY;

		int[] result = new int[2];
		result[0] = (int) (rw*x);
		result[1] = (int) (rh*y);
		return result;
	}

	int getRectNumber() {
		return rectLocation.length;
	}

	public void setAbstraction(IPACAbstraction abstraction) {
		model = (ModelAnalysis)abstraction;
		model.addViewer(this);
	}

	int getHeight(int width) {
		return (int)(width*colorHeight/colorWidth);
	}

	String getName() {
		return model.getFileName();
	}

	void addViewer(IView view) {
		model.addViewer(view);
	}

	Color[] getFinalColor() {
		return model.getFinalColor();
	}

	public void init() {
		double[][] points = model.getInput();
		Statistics stat = new Statistics(points);
		double[][] covMx = stat.getCovariance();
		viewMx = new double[covMx.length][covMx.length];
		MathMatrix.eigenJacobi(covMx, viewMx);
		
		// Convert to the 2D feature space
		pointLocation = new double[points.length][2];
		for (int in = 0; in<points.length; in++) {
			pointLocation[in][0] = MathVector.dot(viewMx[0], points[in]);
			pointLocation[in][1] = MathVector.dot(viewMx[1], points[in]);
		}
		// find min and max in the feature space
		Statistics statLoc = new Statistics(pointLocation);
		double[] minPoints = statLoc.getMin();
		minX = minPoints[0];
		minY = minPoints[1];
		for (int in = 0; in<points.length; in++) {
			pointLocation[in][0] -= minX;
			pointLocation[in][1] -= minY;
		}
		// Calculate the size of the feature space
		double[] maxPoints = statLoc.getMax();
		colorWidth = maxPoints[0]-minX;
		colorHeight = maxPoints[1]-minY;
		
		rectLocation = model.getUnits();
		colorsForPoints = new int[points.length];
	}
	
	public void update(Color[] obj) {
		
	}	

	public void post() {
		
	}

	void setLabelColors() {
		int[] colors = model.getUsedColor();
		System.arraycopy(colors,0,colorsForPoints,0,colors.length);
	}

	void updateImage(Animation animation) {
		int w = animation.getWidth();
		int h = animation.getHeight();
		rw = (w-1)/colorWidth;
		rh = (h-1)/colorHeight;

		int x, y;
		int[] pixels = animation.getPixels();
		for (int in = 0; in<pointLocation.length; in++) {
			x = (int) (rw*pointLocation[in][0]);
			y = (int) (rh*pointLocation[in][1]);
			pixels[x+y*w] = (0xff000000)+colorsForPoints[in];
		}
		animation.update();
	}



}
