package jp.sourceforge.sos.cytoq.analysis;

import java.awt.Color;
import java.awt.Image;
import java.awt.image.BufferedImage;
import java.io.File;
import java.util.ArrayList;

import javax.swing.ComboBoxModel;
import javax.swing.SpinnerModel;
import javax.swing.SpinnerNumberModel;

import jp.sourceforge.sos.cytoq.analysis.imageColor.Animation;
import jp.sourceforge.sos.cytoq.color.ModelColor;
import jp.sourceforge.sos.cytoq.color.PickupRandom;
import jp.sourceforge.sos.cytoq.main.AbstractVariable;
import jp.sourceforge.sos.cytoq.main.IModel;
import jp.sourceforge.sos.cytoq.main.ModelMain;
import jp.sourceforge.sos.cytoq.main.SegmentationAlgorithm;
import jp.sourceforge.sos.cytoq.main.VariableList;
import sos.color.ColorModel;
import sos.designPattern.observer.ConcreteSubject;
import sos.designPattern.observer.IObserver;
import sos.designPattern.observer.ISubject;
import sos.image.ImageInfo;
import sos.io.Extension;
import sos.io.SOSImageIO;
import sos.pac.IPACAbstraction;

/**
 * @uml.dependency supplier="jp.sourceforge.sos.cytoq.analysis.ModelAnalysis"
 */
public class ModelAnalysis implements IModel, IObserver<ColorModel>, ISubject<Color[]> {

	AbstractViewImage[]					resultImages	= new AbstractViewImage[3];

	private ImageInfo					imageInfo		= new ImageInfo();

	private ModelColor					modelColor;

	private SegmentationAlgorithm		algorithm;

	private AbstractVariable			itemArea;

	private AbstractVariable			itemName;

	private ConcreteSubject<Color[]>	subject			= new ConcreteSubject<Color[]>();

	private ArrayList<IView>			viewers			= new ArrayList<IView>();

	private double[][]					input;

	private SpinnerModel				magnificationModel;
	
	private boolean visible;

	private Animation	animation	= new Animation();

	public ModelAnalysis() {
		super();
		itemArea = ModelMain.getItem("Area");
		itemName = ModelMain.getItem("File");

		magnificationModel = new SpinnerNumberModel(1.0, 0.1, 2.0, 0.1);
	}

	public void saveResultImages(String dir) {
		if (!dir.endsWith(File.separator)) {
			dir += File.separator;
		}
		String savingName = Extension.replace(imageInfo.getName(), "png");

		for (int i = 0; i<resultImages.length; i++) {
			if (resultImages[i].isVisible()) {
				String path = dir+resultImages[i].toString()+savingName;
				File file = new File(path);
				BufferedImage img = (BufferedImage)resultImages[i].createImage(imageInfo);
				SOSImageIO.writeImage(file, SOSImageIO.TYPE_PNG, img);
			}
		}
	}

	public void showResultImages() {
		for (IView viewer: viewers) {
			viewer.post();
		}
	}

	/**
	 * @param algorithm
	 *            The algorithm to set.
	 * @uml.property name="algorithm"
	 */
	public void setAlgorithm(SegmentationAlgorithm plugin) {
		algorithm = plugin;
	}

	/**
	 * @param file
	 */
	public void initFile(File file) {
		imageInfo.setFile(file);
		itemName.setValue(imageInfo.getName());
		
		modelColor.setInitColor();
		input = modelColor.convertToData(imageInfo.getIntRGB());
		initAlgorithm();
		
		animation.setSize(imageInfo.getWidth(), imageInfo.getHeight());
		
		for (IView viewer: viewers) {
			viewer.init();
		}
	}

	private void initAlgorithm() {
		algorithm.setInput(input);
		algorithm.setUnit(modelColor.toData());
		algorithm.setParameters(imageInfo);
		algorithm.resetEvaluatedValue();
	}

	public void doMainProcess() {
		algorithm.calculate();
		algorithm.setPixelsLabel(imageInfo);
		modelColor.update(algorithm.getMeanD());
		subject.inform(modelColor.getFinalColor());
	}

	public boolean isConvoluted() {
		return algorithm.isConvoluted();
	}

	public void updateResultData() {
		int[] order = modelColor.setFinalColorOrder();
		if (0<order.length){
			algorithm.sortUnits(order);
		}
		itemArea.setValue(calculateSegmentedArea());		
	}

	/**
	 * @return
	 */
	private String calculateSegmentedArea() {
		int[] area = new int[modelColor.getColorNumber()];
		int[] label = imageInfo.getPixelsLabel();
		for (int i = 0; i<label.length; i++) {
			if (0<=label[i]){
				area[label[i]]++;
			}
		}
		StringBuilder builder = new StringBuilder();
		for (int i = 0; i<area.length; i++) {
			builder.append(area[i]);
			builder.append(" ");
		}
		return builder.toString();
	}

	public void setAbstraction(IPACAbstraction abstraction) {
		modelColor = (ModelColor)abstraction;
		PickupRandom randomPickup = new PickupRandom();
		randomPickup.setImageInfo(imageInfo);
		modelColor.addObserver(randomPickup);
		modelColor.addToPickups(randomPickup, 2);

		resultImages[0] = new OriginalImage("Original");
		resultImages[1] = new ColorPanelImage("Initial", modelColor.initModel);
		resultImages[2] = new ColorPanelImage("Final", modelColor.finalModel);
	}

	ComboBoxModel getColorModels() {
		return modelColor.getColorModels();
	}

	public void update(ColorModel obj) {
		VariableList list = (VariableList)itemArea;
		if (obj==null) {
			list.remove();
		} else {
			list.add();
		}
	}

	public void addObserver(IObserver<Color[]> observer) {
		subject.addObserver(observer);
	}

	public void addViewer(IView viewer) {
		subject.addObserver(viewer);
		viewers.add(viewer);
	}

	int resultImagesNumber() {
		return resultImages.length;
	}

	Image getImage(int i) {
		return resultImages[i].createImage(imageInfo);
	}

	public Color[] getFinalColor() {
		return modelColor.getFinalColor();
	}

	public int[] getColorLabel() {
		return algorithm.getCluster();
	}

	public double[][] getInput() {
		return input;
	}

	public String getFileName() {
		return imageInfo.getName();
	}

	public double[][] getUnits() {
		return algorithm.getMeanD();
	}

	public int getCanvasWidth() {
		return (int) (imageInfo.getWidth()*(Double)magnificationModel.getValue());
	}

	public int getCanvasHeight() {
		return (int) (imageInfo.getHeight()*(Double)magnificationModel.getValue());
	}

	public void setEPS(double value) {
		algorithm.setDesiredValue(value);
	}

	public int[] getUsedColor() {
		return imageInfo.getIntRGB();
	}

	public SpinnerModel getMagnificationModel() {
		return magnificationModel;
	}

	public void setVisible(boolean b) {
		visible = b;
	}

	public boolean isVisible() {
		return visible;
	}

	public void updateAnimation() {
		int[] colorInts = modelColor.getAnimationColor();
		int[] pixels = animation.getPixels();
		int[] labels = imageInfo.getPixelsLabel();
		for (int pn = 0; pn<pixels.length; pn++) {
			pixels[pn] = (0<=labels[pn]) ? colorInts[labels[pn]]:0;
		}
		animation.update();
	}

	public Animation getAnimation() {
		return animation;
	}

}
