/*
 * CytoQViewer.java
 *
 * Created on 2004/12/05, 18:05
 */

package cytoq;

import java.awt.GridBagConstraints;
import java.awt.Image;
import java.awt.image.BufferedImage;
import java.io.File;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Random;
import java.util.Vector;
import sos.graphics.SOSRectangle;
import sos.image.ImageInfo;
import sos.image.ImagePixelMediator;
import sos.io.Extension;
import sos.io.PluginClassLoader;
import sos.io.SOSImageIO;
import sos.math.Clustering;
import sos.math.MathMatrix;
import sos.math.MathVector;
import sos.math.RevertPolish;
import sos.util.ColorVector;
import sos.util.Oval;
import sos.util.Sets;
import sos.util.SortInteger;
import sos.window.JImageStack;
import sos.window.JImageFrame;
import sos.window.JTextFrame;

/**
 *
 * @author Scientific Open Source projects (Gaku Tanaka)
 */
public class CytoQViewer extends javax.swing.JFrame implements Runnable{
	
	static private final String[] cpTitle = new String[]{"SOURCE","INITIAL","FINAL","MERGE"};
	static private final String	sep = System.getProperty("file.separator");
	static private final String pathColorPresets = System.getProperty("user.dir")+sep+"ColorPresets.dat";
	static private String pathAlgorithm;
	static private Random r = new Random();
	static private final ArrayList<String> headerBase;	
	static private JTextFrame twResult = new JTextFrame();
	
	private int
		iteration,
		maxIteration,
		epsDecimal;
	private Image[] imgResult = new Image[cpTitle.length];
	private Vector<String> headings = new Vector();
	private String strStatus;
	private double
		eps,
		betaIncrement,
		radius = 2.0;
	private double[][] input;
	private int[] area;
	
	private Clustering clustering;
	private ImageInfo currentImageInfo = new ImageInfo();
	private JImageStack stackFraction;
	private PointsToImage pti = null;
	private Oval confidenceEllipse = new Oval(24);
	
	private volatile boolean threadSuspended;
	private long requiredTime;
	
	private SOSImageIO io = new SOSImageIO();
	private JDialogAlgebla algeblaDialog;
	private DecimalFormat answerFormat = new DecimalFormat();
	
	private ColorSpaceViewer colorDialog = new ColorSpaceViewer();
	private ColorSpaceViewer segDialog = new ColorSpaceViewer();
	private JImageFrame processFrame = new JImageFrame();
	private SOSRectangle[] rectFraction;
	
	static{
		headerBase = new ArrayList<String>();
		headerBase.add("File");
		headerBase.add("Calculation");
		headerBase.add("Time(s)");
		headerBase.add("Iteration");
		headerBase.add("Colors");
		headerBase.add("Likelihood");
	}
	
	/** Creates new form CytoQViewer */
	public CytoQViewer() {
		
		initComponents();
		
		algeblaDialog = new JDialogAlgebla(this, true);
		setEquationFromDialog();
		
		loadAlgorithmPlugin();
		
		panelInit.init(true,3, "Color");
		panelFinal.init(false,3,"Color");
		jSpinnerColorNumber.setValue(3);
		pack();
		
		setVisible(true);
		
		optionDialog.pack();
		optionDialog.setSize(optionDialog.getWidth()+10,optionDialog.getHeight()+10);
		
		colorDialog.setCanvasSize(this.getWidth()*2/3, this.getWidth()/2);
		colorDialog.setVisible(true);
		
		segDialog.setSize(colorDialog.getWidth(), colorDialog.getHeight());
		
		twResult.setTitle("Results");
		twResult.setSize(350,150);
		
		// after frames and dialogs were shown
		jComboPresets.loadItem(pathColorPresets);
		panelInit.updateColor(jComboPresets.getSelectedRGB());
		
		jLabelStatus.setText("Welcome to CytoQ !");
		
	}
	
	private void loadAlgorithmPlugin(){
		pathAlgorithm = System.getProperty("user.dir");
		pathAlgorithm += sep+ "plugins";
		File file = new File(pathAlgorithm);
		File[] fileList = file.listFiles();
		for (int i=0; i<fileList.length; i++){
			String algorithmName = Extension.remove(fileList[i].getName());
			jComboAlgorithm.addItem(algorithmName);
		}
	}
	
	public void run() {
		// MainProcess
		threadSuspended = false;
		stackFraction = null;
		switch(jComboAnalysis.getSelectedIndex()){
			case 0:
				analyzeSingle();
				break;
			case 1:
				analyzeBatch();
				break;
			case 2:
				analyzeEnsemble();
				break;
		}
		
		jLabelStatus.setText("The analysis was finished!");
	}
	
	private void analyzeSingle() {
		File file = io.getOpenFile();
		if (file!=null){
			showResultWindow("");
			currentImageInfo.setName(file.getName());
			strStatus = file.getName();
			Image img = io.readImage(file);
			analyzePre(img);
			analyzeMain();
			analyzePost(img);
			
			jLabelStatus.setText("Showing...");
			if (!jCheckSource.isSelected() && !jCheckResultInit.isSelected() && !jCheckResultFinal.isSelected()){
				return;
			}
			JImageStack stack = new JImageStack();
			stack.setTitle("Result images");
			stack.setCanvasSize(currentImageInfo.getWidth(),currentImageInfo.getHeight());
			for (int i=0; i<cpTitle.length; i++){
				if (imgResult[i]!=null){
					stack.addImage(imgResult[i]);
				}
			}
			stack.setLocation(this.getWidth(),stack.getY());
			stack.setVisible(true);
		}
	}
	
	private final void analyzeBatch(){
		// setup previous analysis images
		int batchMax = (Integer)jSpinnerBatchIteration.getValue();
		
		// setup loading directory
		File file = io.getOpenDirectory();
		if (file==null){
			jLabelStatus.setText("The analysis was canceled!");
			return;
		}
		File[] fileList = file.listFiles();		
		Arrays.sort(fileList);
		
		// setup saving directory
		String saveDir = null;
		if (jCheckSource.isSelected() || jCheckResultInit.isSelected() || jCheckResultFinal.isSelected()){
			saveDir = io.getSaveDirectory().getPath();
			if (saveDir==null){
				jLabelStatus.setText("The analysis was canceled!");
				return;
			}
		}
		
		// initialize
		stackFraction.setCanvasSize(canvasFractFinal.getWidth(), canvasFractFinal.getHeight());
		showResultWindow("("+(String)jComboPresets.getSelectedItem()+")");
		
		int nImage = 0;
		int[][]	sumColor = new int[panelInit.getColorNumber()][ColorVector.dim];
		int cNumber = -1;
		int[] intRGB = null;
		for (int t=0; t<batchMax; t++){
			for (int i=0; i<fileList.length; i++){
				currentImageInfo.setName(fileList[i].getName());
				cNumber = i+t*fileList.length;
				strStatus = "("+(cNumber+1)+"/"+(batchMax*fileList.length)+") "+fileList[i];
				Image img = io.readImage(fileList[i]);
				if (img!=null){
					analyzePre(img);
					analyzeMain();
					analyzePost(img);
					if (saveDir!=null){
						jLabelStatus.setText("Saving images...");
						saveResultImages(saveDir, (t+1));
					}
					MathMatrix.add(sumColor, panelFinal.getColorArray());
					nImage++;
				}
			}
		}
		if (jCheckRandom.isSelected()){
			panelInit.clearColor();
		}
		MathMatrix.scaleDivide(sumColor,nImage);
		panelFinal.updateColor(sumColor);
		
		stackFraction.setVisible(true);
	}
	
	private final void analyzeEnsemble(){
		/*
		// setup loading directory
		File file = io.getOpenDirectory();
		File[] fileList = file.listFiles();
		if (fileList==null){
			jLabelStatus.setText("The analysis was canceled!");
			return;
		}
		Arrays.sort(fileList);
		
		// initialize
		String strTab = "";
		for (int i=0; i<13; i++){
			strTab += "\t";
		}
		showResultWindow("("+(String)jComboPresets.getSelectedItem()+")"+strTab+panelInit.getColorString(" "));
		
		int[] rgb = new int[16777216];
		int index;
		for (int i=0; i<fileList.length; i++){
			strStatus = "("+(i+1)+"/"+(fileList.length)+") "+fileList[i].getName();
			Image currentImage = io.loadImage(fileList[i]);
			if (currentImage!=null){
				jLabelStatus.setText(strStatus+": Correcting background...");
				currentImageInfo.setSize(currentImage.getWidth(null),currentImage.getHeight(null));
				ImageProcessor currentIp = correctBackground(currentImage, jComboBack.getSelectedIndex());
				
				jLabelStatus.setText(strStatus+": Extracting colors...");
				int[] pixels = (int[])currentIp.getPixels();
				for (int in=0; in<pixels.length; in++){
					index = pixels[in]&0xffffff;
					rgb[index]++;
				}
			}
		}
		
		jLabelStatus.setText("Ensemble colors...");
		int n=0;
		for (int cn=0; cn<rgb.length; cn++){
			if (rgb[cn]>0){
				n++;
			}
		}
		int[] intRGB = new int[n];
		int[] histogram = new int[n];
		n=0;
		for (int cn=0; cn<rgb.length; cn++){
			if (rgb[cn]>0){
				intRGB[n] = cn;
				histogram[n] = rgb[cn];
				n++;
			}
		}
		strStatus = "Ensemble: t=";
		rgb = null;
		currentImageInfo.setIntRGB(intRGB);
		currentImageInfo.setColorHistogram(histogram);
		analyzeMain();*/
	}
	
	private final void showResultWindow(String data){	// show result window
		if (!twResult.isVisible()){
			headings.clear();
			headings.addAll(headerBase);
			for (int i=0; i<panelInit.getColorNumber(); i++){
				headings.add("#"+(i+1)+"(area)");
			}
			for (int i=0; i<panelInit.getColorNumber(); i++){
				headings.add("#"+(i+1)+"(color)");
			}
			for (int i=0; i<panelInit.getColorNumber(); i++){
				headings.add("#"+(i+1)+"(volume)");
			}
			for (int i=0; i<panelInit.getColorNumber(); i++){
				headings.add("\tRGB"+(i+1));
			}
			
			twResult.setHeadings(headings);
			twResult.setLocation(colorDialog.getX(),colorDialog.getY()+colorDialog.getHeight());
			twResult.setVisible(true);
		}
	}
	
	private void analyzePre(Image img){
		jLabelStatus.setText(strStatus+": Correcting background...");
		currentImageInfo.setImage(img);
		currentImageInfo.clearLabel();
		//ImageProcessor currentIp = correctBackground(imp.getProcessor(), jComboBack.getSelectedIndex());
		
		jLabelStatus.setText(strStatus+": Extracting colors...");
		int[] pixels = (int[])currentImageInfo.getPixels();
		currentImageInfo.setColor(pixels);
	}
	
	private void analyzeMain(){
		if (jComboColorSpace.getSelectedIndex()==0){
			input = currentImageInfo.getRGB();
		}else{
			input = currentImageInfo.getLAB();
		}
		showColorDialogs();
		
		// setting parameters
		iteration = 0;
		area = new int[panelFinal.getColorNumber()];
		for (int i=0; i<cpTitle.length; i++){
			imgResult[i] = null;
		}
		
		if (jCheckRandom.isSelected()){
			panelInit.setRandomColor(currentImageInfo);
		}else{
			panelInit.updateColor(jComboPresets.getSelectedRGB());
		}
		
		// set algorithm
		requiredTime = System.currentTimeMillis();
		selectAlgorithm();
		requiredTime -= System.currentTimeMillis();
		
		// sort final colors if initial colors are random
		jLabelStatus.setText(strStatus+": Sorting colors...");
		switch (jComboSort.getSelectedIndex()){
			case 0: // do nothing
				break;
			case 1:	// sort by luminance
				int[][] unit = panelFinal.getColorArray();
				int[] luminance = new int[panelInit.getColorNumber()];
				for (int un=0; un<panelInit.getColorNumber(); un++){
					luminance[un] = ColorVector.RGBtoL(unit[un]);
				}
				SortInteger si = new SortInteger(luminance);
				panelFinal.updateColor((int[][])si.sort(unit));
				panelInit.updateColor((int[][])si.sort(panelInit.getColorArray()));
				
				for (int i=0; i<luminance.length; i++){
					System.out.println(i+"="+si.getOrder(i));
				}
				
				clustering.sortUnits(si.getOrder());
				break;
			case 2:	// match into color set
				matchColorSet();
				break;
		}
	}
	
	private final void analyzePost(Image img){
		if (stackFraction!=null){
			stackFraction.addImage(canvasFractFinal.getScreenImage());
		}
		
		if (currentImageInfo.getPixelsLabel(0)==-1){
			currentImageInfo.setPixelsLabel(clustering.getCluster());
		}
		
		// set result images
		if (jCheckSource.isSelected()){
			imgResult[0] = img;
		}
		if (jCheckResultInit.isSelected()){
			if (jCheckRandom.isSelected()){
				imgResult[1] = drawSegmentation(jComboPresets.getSelectedRGBint());
			}else{
				imgResult[1] = drawSegmentation(panelInit.getColorInt());
			}
		}
		if (jCheckResultFinal.isSelected()){
			imgResult[2] = drawSegmentation(panelFinal.getColorInt());
		}
		
		twResult.append(makeResultString());
	}
	
	/*
	private ImageProcessor correctBackground(Image img, int type){
		ColorProcessor cp = new ColorProcessor(img);
		
		if (type==2){	// withoud background correnction
			return cp;
		}else{
			byte[] H = new byte[currentImageInfo.getImageSize()];
			byte[] S = new byte[currentImageInfo.getImageSize()];
			byte[] B = new byte[currentImageInfo.getImageSize()];
			cp.getHSB(H, S, B);
			ByteProcessor brightness = new ByteProcessor(currentImageInfo.getWidth(), currentImageInfo.getHeight(), B, null);
			if (type==0){	// white background
				brightness.invert();
				new BackgroundSubtracter().subtractBackround(brightness, currentImageInfo.getWidth()/4);
				brightness.invert();
			}else{			// black background
				new BackgroundSubtracter().subtractBackround(brightness, currentImageInfo.getWidth()/4);
			}
			cp.setHSB(H, S, (byte[])brightness.getPixels());
		}
		
		return cp;
	}*/
	
	private void showColorDialogs(){
		if (jCheckProcess.isSelected()){
			double m = Double.parseDouble(jTextMagnification.getText());
			int w = (int)(currentImageInfo.getWidth()*m);
			int h = (int)(currentImageInfo.getHeight()*m);
			processFrame.setCanvasSize(w,h);
			if (!processFrame.isVisible()){
				processFrame.setLocation(getWidth(),getY());
				processFrame.setVisible(true);
			}			
		}
		
		pti = new PointsToImage(input);
		pti.setColors(currentImageInfo.getIntRGB());		
		colorDialog.setPointsToImage(pti);
		colorDialog.setTitle("Distribution map ("+currentImageInfo.getName()+")");
		
		if (segDialog.isShowing()){
			segDialog.setPointsToImage(pti);
			segDialog.setTitle("Segmentation map ("+currentImageInfo.getName()+")");
			segDialog.pack();
		}
	}
	
	private final double[][] colorToUnit(int[][] data){
		double[][] unit = new double[panelFinal.getColorNumber()][ColorVector.dim];
		if (jComboColorSpace.getSelectedIndex()==0){
			for (int un=0; un<panelInit.getColorNumber(); un++){
				for (int d=0; d<ColorVector.dim; d++){
					unit[un][d] = data[un][d];
				}
			}
		}else{
			for (int un=0; un<panelInit.getColorNumber(); un++){
				ColorVector.RGBtoLAB(data[un],unit[un]);
			}
		}
		
		if (unit.length>panelInit.getColorNumber()){
			int tempNumber = panelInit.getColorNumber();
			int index;
			do{
				index = tempNumber%panelInit.getColorNumber();
				System.arraycopy(unit[index],0, unit[tempNumber],0, ColorVector.dim);
				tempNumber++;
			}while(tempNumber<unit.length);
		}
		
		return unit;
	}
	
	private void selectAlgorithm(){
		clustering.setInput(input);
		clustering.setUnit(colorToUnit(panelInit.getColorArray()));
		clustering.setParameters(currentImageInfo);
		
		epsDecimal = Integer.parseInt(jTextEPS.getText());
		eps = Math.pow(0.1, epsDecimal);
		clustering.setDesiredValue(eps);
		
		//radius = Math.sqrt(-2*Math.log(1.0-Double.parseDouble (jTextConfidence.getText())));
		maxIteration = Integer.parseInt(jTextMaxIteration.getText());
		
		do{
			clustering.calculate();
			((AnalyzePluginGUI)clustering).setPixelsLabel(currentImageInfo);
			updateComponentsForUnit();
			iteration++;
			
			try{
				Thread.currentThread().sleep(10);
				if (threadSuspended) {
					synchronized(this) {
						while (threadSuspended)
							wait();
					}
				}
			} catch (InterruptedException e){
				e.printStackTrace();
			}
		}while(!clustering.isConvoluted() && iteration<maxIteration);
		
	}
	
	private final boolean hasUnchangedColor(){
		
		int[] clusterOfColor = clustering.getCluster();
		int[] size = new int[panelInit.getColorNumber()];
		for (int pn=0; pn<clusterOfColor.length; pn++){
			size[clusterOfColor[pn]]++;
		}
		
		for (int i=0; i<panelInit.getColorNumber(); i++){
			if (size[i]==0){
				return true;
			}
		}
		
		return false;
	}
	
	private void updateComponentsForUnit(){
		int[][] newColor = new int[panelFinal.getColorNumber()][ColorVector.dim];
		switch (jComboColorSpace.getSelectedIndex()){
			case 0:	// RGB
				newColor = clustering.getMean();
				break;
			case 1:	// CIE L*a*b*
				for (int un=0; un<panelFinal.getColorNumber(); un++){
					ColorVector.LABtoRGB(clustering.getMeanD(un),newColor[un]);
				}
				break;
		}
		panelFinal.updateColor(newColor);
		
		colorDialog.drawRects(clustering.getMeanD(), panelFinal.getColor());
		segDialog.drawLabel(clustering.getCluster(), panelFinal.getColorInt());
		
		drawFraction();
		
		if (processFrame.isShowing()){
			int[] color;
			if (jComboAnimationColor.getSelectedIndex()==0){
				color = panelInit.getColorInt();
			}else{
				color = panelFinal.getColorInt();
			}
			Image img = drawSegmentation(color);
			processFrame.setImage(img);
		}
	}
	
	private final void matchColorSet(){
		// get color array from the color set
		double[][] unitD = colorToUnit(jComboPresets.getSelectedRGB());
		if (unitD.length!=panelInit.getColorNumber()){
			jLabelStatus.setText("Could NOT sort because of the invalid preset!");
		}else{
			int[] order = Sets.match(input,currentImageInfo.getHistogram(),unitD,clustering.getMeanD(),clustering.getCluster());
			clustering.sortUnits(order);
			
			int[][] orderedUnit = new int[panelInit.getColorNumber()][];
			for (int i=0; i<panelInit.getColorNumber(); i++){
				orderedUnit[i] = panelFinal.getColorArray(order[i]);
			}
			panelFinal.updateColor(orderedUnit);
			
			if (jCheckRandom.isSelected()){
				for (int i=0; i<panelInit.getColorNumber(); i++){
					orderedUnit[i] = panelInit.getColorArray(order[i]);
				}
				panelInit.updateColor(orderedUnit);
			}
		}
	}
	
	private final void drawFraction(){
		Arrays.fill(area,0);
		if (currentImageInfo.getPixelsLabel(0)==-1){
			int[] histogram = currentImageInfo.getHistogram();
			int[] cluster = clustering.getCluster();
			for (int i=0; i<histogram.length; i++){
				area[cluster[i]] += histogram[i];
			}
		}else{
			for (int pn=0; pn<currentImageInfo.getImageSize(); pn++){
				area[currentImageInfo.getPixelsLabel(pn)]++;
			}
		}
		
		// percentage of each area
		int cumX = 1;
		int fract;
		for (int un=0; un<panelFinal.getColorNumber(); un++){
			fract = canvasFractFinal.getWidth()*area[un]/currentImageInfo.getImageSize();
			rectFraction[un].setColor(panelFinal.getColor(un));
			rectFraction[un].setBounds(cumX,0, fract+1,canvasFractFinal.getHeight());
			cumX += fract;
		}
		canvasFractFinal.repaint();
	}
	
	private final Vector<String> makeResultString(){
		Vector<String> result = new Vector<String>();
		
		result.add(currentImageInfo.getName());
		
		// calculation
		String equation = jLabelEquation.getText();
		double calc = RevertPolish.calc(RevertPolish.convert(equation),area);
		result.add(Double.toString(calc));
		jLabelResult.setText(answerFormat.format(calc));
		
		result.add(Long.toString(-requiredTime/1000));
		result.add(Integer.toString(iteration));
		result.add(Integer.toString(currentImageInfo.getUsedColorNumber()));
		result.add(Double.toString(clustering.getEvaluatedValue()));
		
		// the number of PIXELS in the cluster
		for (int un=0; un<area.length; un++){
			result.add(Integer.toString(area[un]));
		}
		// the number of COLORS in the cluster
		int[] clusterOfColor = clustering.getCluster();
		int[] colorSize = new int[panelFinal.getColorNumber()];
		for (int in=0; in<clusterOfColor.length; in++){
			colorSize[clusterOfColor[in]]++;
		}
		for (int un=0; un<panelFinal.getColorNumber(); un++){
			result.add(Integer.toString(colorSize[un]));
		}
		// Volume of confidence ellipse
		double[][][] intraCovMx = MathVector.covariance(input,clusterOfColor,panelFinal.getColorNumber());
		double[][] w = new double[ColorVector.dim][ColorVector.dim];
		double volume;
		for (int i=0; i<panelFinal.getColorNumber(); i++){
			MathMatrix.eigenJacobi(intraCovMx[i],w);
			volume = 4*Math.PI/3;
			for (int d=0; d<ColorVector.dim; d++){
				volume *= 2*Math.sqrt(intraCovMx[i][d][d]);
			}
			result.add(Double.toString(volume));
		}
		// RGB value of each final color
		String argColor = panelFinal.getColorString(" ");
		String[] strColors = argColor.split(" ");
		for (int i=0; i<strColors.length; i++){
			result.add(strColors[i]);
		}
		
		return result;
	}
	
	private final Image drawSegmentation(int[] rgb){
		int[] pixels = new int[currentImageInfo.getImageSize()];
		for (int pn=0; pn<currentImageInfo.getImageSize(); pn++){
			if (currentImageInfo.getPixelsLabel(pn)>=0){
				pixels[pn] = rgb[currentImageInfo.getPixelsLabel(pn)];
			}else{
				pixels[pn] = 0;
			}
		}
		
		return ImagePixelMediator.convertToImage(pixels,currentImageInfo.getWidth(),currentImageInfo.getHeight());
	}
	
	private void saveResultImages(String dir, int t){
		// remove the extension from imi.getName()
		String savingName = Extension.remove(currentImageInfo.getName());
		savingName += ".png";
		
		for (int i=0; i<cpTitle.length; i++){
			if (imgResult[i]!=null){
				io.writeImage(io.TYPE_PNG,(BufferedImage)imgResult[i]);
			}
		}
	}
	
	private void setNewColorPanel(int n){
		panelInit.setColorNumber(n);
		panelFinal.setColorNumber(n);
		this.pack();
	}
	
	private void setEquationFromDialog(){
		jLabelEquation.setText(algeblaDialog.getEquation());
		jLabelUnit.setText(algeblaDialog.getUnit());
		answerFormat.setMaximumFractionDigits(algeblaDialog.getDecimalPlace());
	}
	
	private String getPresetsName(){
		presetsDialog.pack();
		presetsDialog.setVisible(true);
		return jTextPresetsName.getText();
	}
	
	/** This method is called from within the constructor to
	 * initialize the form.
	 * WARNING: Do NOT modify this code. The content of this method is
	 * always regenerated by the Form Editor.
	 */
    // <editor-fold defaultstate="collapsed" desc=" Generated Code ">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        optionDialog = new javax.swing.JDialog();
        jLabel7 = new javax.swing.JLabel();
        jLabel9 = new javax.swing.JLabel();
        jLabel10 = new javax.swing.JLabel();
        jComboColorSpace = new javax.swing.JComboBox();
        jLabel13 = new javax.swing.JLabel();
        jTextEPS = new javax.swing.JTextField();
        jLabel18 = new javax.swing.JLabel();
        jComboSort = new javax.swing.JComboBox();
        jLabel12 = new javax.swing.JLabel();
        jTextMaxIteration = new javax.swing.JTextField();
        jPanelOption = new javax.swing.JPanel();
        jLabel4 = new javax.swing.JLabel();
        jLabel11 = new javax.swing.JLabel();
        jLabel25 = new javax.swing.JLabel();
        jSpinnerBatchIteration = new javax.swing.JSpinner();
        presetsDialog = new javax.swing.JDialog();
        jLabel14 = new javax.swing.JLabel();
        jTextPresetsName = new javax.swing.JTextField();
        jButtonPresetsNameOK = new javax.swing.JButton();
        jMenuBar1 = new javax.swing.JMenuBar();
        jMenu1 = new javax.swing.JMenu();
        jMenuSaveImage = new javax.swing.JMenuItem();
        jFrameTextResult = new javax.swing.JFrame();
        jPanelMain = new javax.swing.JPanel();
        jLabel20 = new javax.swing.JLabel();
        jComboAnalysis = new javax.swing.JComboBox();
        jLabel21 = new javax.swing.JLabel();
        jCheckSource = new javax.swing.JCheckBox();
        jCheckResultInit = new javax.swing.JCheckBox();
        jCheckResultFinal = new javax.swing.JCheckBox();
        jLabel22 = new javax.swing.JLabel();
        jCheckColor = new javax.swing.JCheckBox();
        jCheckSeg = new javax.swing.JCheckBox();
        jButtonOption = new javax.swing.JButton();
        jButtonAnalysis = new javax.swing.JButton();
        jLabel5 = new javax.swing.JLabel();
        jComboAlgorithm = new javax.swing.JComboBox();
        jCheckProcess = new javax.swing.JCheckBox();
        jLabel26 = new javax.swing.JLabel();
        jComboAnimationColor = new javax.swing.JComboBox();
        jLabel17 = new javax.swing.JLabel();
        jLabel27 = new javax.swing.JLabel();
        jTextMagnification = new javax.swing.JTextField();
        jButtonBreak = new javax.swing.JButton();
        jPanelColor = new javax.swing.JPanel();
        jLabel1 = new javax.swing.JLabel();
        jLabel2 = new javax.swing.JLabel();
        jButtonToInit = new javax.swing.JButton();
        jCheckRandom = new javax.swing.JCheckBox();
        jLabel19 = new javax.swing.JLabel();
        jLabel15 = new javax.swing.JLabel();
        jButtonRemovePreset = new javax.swing.JButton();
        jButtonAddPreset = new javax.swing.JButton();
        jSpinnerColorNumber = new javax.swing.JSpinner();
        jButtonSave = new javax.swing.JButton();
        panelInit = new sos.panel.JColorPanel();
        panelFinal = new sos.panel.JColorPanel();
        canvasFractFinal = new sos.canvas.JObjectCanvas();
        jComboPresets = new cytoq.JComboColorSets();
        jPanelResult = new javax.swing.JPanel();
        jButtonCustomize = new javax.swing.JButton();
        jLabel3 = new javax.swing.JLabel();
        jLabelResult = new javax.swing.JLabel();
        jLabel16 = new javax.swing.JLabel();
        jLabelEquation = new javax.swing.JLabel();
        jLabel23 = new javax.swing.JLabel();
        jLabel24 = new javax.swing.JLabel();
        jLabelUnit = new javax.swing.JLabel();
        jLabelStatus = new javax.swing.JLabel();

        optionDialog.getContentPane().setLayout(new java.awt.GridBagLayout());

        optionDialog.setTitle("Options");
        optionDialog.setModal(true);
        jLabel7.setForeground(new java.awt.Color(0, 0, 255));
        jLabel7.setText("PREPROCESSING");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        optionDialog.getContentPane().add(jLabel7, gridBagConstraints);

        jLabel9.setForeground(new java.awt.Color(0, 0, 255));
        jLabel9.setText("ALGORITHMs");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(2, 0, 0, 0);
        optionDialog.getContentPane().add(jLabel9, gridBagConstraints);

        jLabel10.setForeground(new java.awt.Color(102, 102, 102));
        jLabel10.setText("color space: ");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        optionDialog.getContentPane().add(jLabel10, gridBagConstraints);

        jComboColorSpace.setModel(new javax.swing.DefaultComboBoxModel(new String[] { "RGB", "CIE L*a*b*" }));
        jComboColorSpace.setSelectedIndex(1);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(0, 10, 0, 0);
        optionDialog.getContentPane().add(jComboColorSpace, gridBagConstraints);

        jLabel13.setForeground(new java.awt.Color(102, 102, 102));
        jLabel13.setText("EPS (decimal): ");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 10, 0, 0);
        optionDialog.getContentPane().add(jLabel13, gridBagConstraints);

        jTextEPS.setText("5");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(0, 10, 0, 0);
        optionDialog.getContentPane().add(jTextEPS, gridBagConstraints);

        jLabel18.setForeground(new java.awt.Color(102, 102, 102));
        jLabel18.setText("sort: ");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 8;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        optionDialog.getContentPane().add(jLabel18, gridBagConstraints);

        jComboSort.setModel(new javax.swing.DefaultComboBoxModel(new String[] { "nothing", "luminance", "preset" }));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 8;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(0, 10, 0, 0);
        optionDialog.getContentPane().add(jComboSort, gridBagConstraints);

        jLabel12.setForeground(new java.awt.Color(102, 102, 102));
        jLabel12.setText("max iteration: ");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 6;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        optionDialog.getContentPane().add(jLabel12, gridBagConstraints);

        jTextMaxIteration.setText("500");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 6;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 10, 0, 0);
        optionDialog.getContentPane().add(jTextMaxIteration, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 17;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        optionDialog.getContentPane().add(jPanelOption, gridBagConstraints);

        jLabel4.setForeground(new java.awt.Color(0, 0, 255));
        jLabel4.setText("OPTIONS");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 10;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(5, 0, 0, 0);
        optionDialog.getContentPane().add(jLabel4, gridBagConstraints);

        jLabel11.setForeground(new java.awt.Color(0, 0, 255));
        jLabel11.setText("RESULTS");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 7;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(2, 0, 0, 0);
        optionDialog.getContentPane().add(jLabel11, gridBagConstraints);

        jLabel25.setForeground(new java.awt.Color(102, 102, 102));
        jLabel25.setText("batch iteration:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.insets = new java.awt.Insets(0, 10, 0, 0);
        optionDialog.getContentPane().add(jLabel25, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(0, 10, 0, 0);
        optionDialog.getContentPane().add(jSpinnerBatchIteration, gridBagConstraints);

        presetsDialog.getContentPane().setLayout(new java.awt.GridBagLayout());

        presetsDialog.setModal(true);
        jLabel14.setText("Name: ");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(0, 5, 0, 0);
        presetsDialog.getContentPane().add(jLabel14, gridBagConstraints);

        jTextPresetsName.setPreferredSize(new java.awt.Dimension(150, 22));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        presetsDialog.getContentPane().add(jTextPresetsName, gridBagConstraints);

        jButtonPresetsNameOK.setText("OK");
        jButtonPresetsNameOK.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButtonPresetsNameOKActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.insets = new java.awt.Insets(5, 0, 5, 5);
        presetsDialog.getContentPane().add(jButtonPresetsNameOK, gridBagConstraints);

        jMenu1.setText("File");
        jMenuSaveImage.setText("Save image");
        jMenuSaveImage.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jMenuSaveImageActionPerformed(evt);
            }
        });

        jMenu1.add(jMenuSaveImage);

        jMenuBar1.add(jMenu1);

        getContentPane().setLayout(new java.awt.GridBagLayout());

        setDefaultCloseOperation(javax.swing.WindowConstants.EXIT_ON_CLOSE);
        setTitle("CytoQ");
        setCursor(new java.awt.Cursor(java.awt.Cursor.DEFAULT_CURSOR));
        addComponentListener(new java.awt.event.ComponentAdapter() {
            public void componentResized(java.awt.event.ComponentEvent evt) {
                formComponentResized(evt);
            }
        });
        addWindowListener(new java.awt.event.WindowAdapter() {
            public void windowClosing(java.awt.event.WindowEvent evt) {
                formWindowClosing(evt);
            }
        });

        jPanelMain.setLayout(new java.awt.GridBagLayout());

        jPanelMain.setBorder(new javax.swing.border.TitledBorder("MAIN"));
        jLabel20.setText("ANALYSIS");
        jLabel20.setForeground(new java.awt.Color(0, 0, 255));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        jPanelMain.add(jLabel20, gridBagConstraints);

        jComboAnalysis.setModel(new javax.swing.DefaultComboBoxModel(new String[] { "Single", "Batch" }));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(0, 10, 0, 0);
        jPanelMain.add(jComboAnalysis, gridBagConstraints);

        jLabel21.setText("ANIMATION");
        jLabel21.setForeground(new java.awt.Color(0, 0, 255));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(5, 0, 0, 0);
        jPanelMain.add(jLabel21, gridBagConstraints);

        jCheckSource.setText("SOURCE");
        jCheckSource.setSelected(true);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 9;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 5, 0, 0);
        jPanelMain.add(jCheckSource, gridBagConstraints);

        jCheckResultInit.setText("Result (INITIAL)");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 10;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 5, 0, 0);
        jPanelMain.add(jCheckResultInit, gridBagConstraints);

        jCheckResultFinal.setText("Result (FINAL)");
        jCheckResultFinal.setSelected(true);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 11;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 5, 0, 0);
        jPanelMain.add(jCheckResultFinal, gridBagConstraints);

        jLabel22.setForeground(new java.awt.Color(0, 0, 255));
        jLabel22.setText("COLOR MAPS");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 12;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(5, 0, 0, 0);
        jPanelMain.add(jLabel22, gridBagConstraints);

        jCheckColor.setSelected(true);
        jCheckColor.setText("Distribution");
        jCheckColor.addItemListener(new java.awt.event.ItemListener() {
            public void itemStateChanged(java.awt.event.ItemEvent evt) {
                jCheckColorItemStateChanged(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 13;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 5, 0, 0);
        jPanelMain.add(jCheckColor, gridBagConstraints);

        jCheckSeg.setText("Segmentation");
        jCheckSeg.addItemListener(new java.awt.event.ItemListener() {
            public void itemStateChanged(java.awt.event.ItemEvent evt) {
                jCheckSegItemStateChanged(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 14;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 5, 0, 0);
        jPanelMain.add(jCheckSeg, gridBagConstraints);

        jButtonOption.setText("OPTIONS");
        jButtonOption.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButtonOptionActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 15;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(5, 0, 0, 0);
        jPanelMain.add(jButtonOption, gridBagConstraints);

        jButtonAnalysis.setForeground(new java.awt.Color(255, 0, 0));
        jButtonAnalysis.setText("START");
        jButtonAnalysis.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButtonAnalysisActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 16;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        jPanelMain.add(jButtonAnalysis, gridBagConstraints);

        jLabel5.setText("ALGORITHM");
        jLabel5.setForeground(new java.awt.Color(0, 0, 255));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        jPanelMain.add(jLabel5, gridBagConstraints);

        jComboAlgorithm.addItemListener(new java.awt.event.ItemListener() {
            public void itemStateChanged(java.awt.event.ItemEvent evt) {
                jComboAlgorithmItemStateChanged(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(0, 10, 0, 0);
        jPanelMain.add(jComboAlgorithm, gridBagConstraints);

        jCheckProcess.setSelected(true);
        jCheckProcess.setText("show");
        jCheckProcess.addItemListener(new java.awt.event.ItemListener() {
            public void itemStateChanged(java.awt.event.ItemEvent evt) {
                jCheckProcessItemStateChanged(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(0, 5, 0, 0);
        jPanelMain.add(jCheckProcess, gridBagConstraints);

        jLabel26.setText("SHOW IMAGES");
        jLabel26.setForeground(new java.awt.Color(0, 0, 255));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 8;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(5, 0, 0, 0);
        jPanelMain.add(jLabel26, gridBagConstraints);

        jComboAnimationColor.setModel(new javax.swing.DefaultComboBoxModel(new String[] { "INITIAL", "FINAL" }));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 7;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        jPanelMain.add(jComboAnimationColor, gridBagConstraints);

        jLabel17.setText("magnification: ");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 6;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(0, 10, 0, 0);
        jPanelMain.add(jLabel17, gridBagConstraints);

        jLabel27.setText("colors: ");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 7;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(0, 10, 0, 0);
        jPanelMain.add(jLabel27, gridBagConstraints);

        jTextMagnification.setText("1.0");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 6;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        jPanelMain.add(jTextMagnification, gridBagConstraints);

        jButtonBreak.setForeground(new java.awt.Color(255, 0, 0));
        jButtonBreak.setText("SUSPEND");
        jButtonBreak.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButtonBreakActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 17;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        jPanelMain.add(jButtonBreak, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridheight = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.ipadx = 10;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTH;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        getContentPane().add(jPanelMain, gridBagConstraints);

        jPanelColor.setLayout(new java.awt.GridBagLayout());

        jPanelColor.setBorder(new javax.swing.border.TitledBorder("COLORS"));
        jLabel1.setForeground(new java.awt.Color(0, 0, 255));
        jLabel1.setText("INITIAL");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        jPanelColor.add(jLabel1, gridBagConstraints);

        jLabel2.setForeground(new java.awt.Color(0, 0, 255));
        jLabel2.setText("FINAL");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 6;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(5, 0, 2, 0);
        jPanelColor.add(jLabel2, gridBagConstraints);

        jButtonToInit.setText("COPY TO INITIAL");
        jButtonToInit.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButtonToInitActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 8;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(0, 10, 0, 0);
        jPanelColor.add(jButtonToInit, gridBagConstraints);

        jCheckRandom.setText("random colors");
        jCheckRandom.addItemListener(new java.awt.event.ItemListener() {
            public void itemStateChanged(java.awt.event.ItemEvent evt) {
                jCheckRandomItemStateChanged(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(0, 10, 0, 0);
        jPanelColor.add(jCheckRandom, gridBagConstraints);

        jLabel19.setForeground(new java.awt.Color(0, 0, 255));
        jLabel19.setText("PIXELS");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 9;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        jPanelColor.add(jLabel19, gridBagConstraints);

        jLabel15.setText("color number:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        jPanelColor.add(jLabel15, gridBagConstraints);

        jButtonRemovePreset.setForeground(new java.awt.Color(0, 0, 255));
        jButtonRemovePreset.setText("REMOVE");
        jButtonRemovePreset.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButtonRemovePresetActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.ipadx = 1;
        gridBagConstraints.insets = new java.awt.Insets(0, 10, 0, 0);
        jPanelColor.add(jButtonRemovePreset, gridBagConstraints);

        jButtonAddPreset.setForeground(new java.awt.Color(0, 0, 255));
        jButtonAddPreset.setText("ADD");
        jButtonAddPreset.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButtonAddPresetActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        jPanelColor.add(jButtonAddPreset, gridBagConstraints);

        jSpinnerColorNumber.addChangeListener(new javax.swing.event.ChangeListener() {
            public void stateChanged(javax.swing.event.ChangeEvent evt) {
                jSpinnerColorNumberStateChanged(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        jPanelColor.add(jSpinnerColorNumber, gridBagConstraints);

        jButtonSave.setForeground(new java.awt.Color(0, 0, 255));
        jButtonSave.setText("SAVE");
        jButtonSave.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButtonSaveActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 4;
        jPanelColor.add(jButtonSave, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        jPanelColor.add(panelInit, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 7;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        jPanelColor.add(panelFinal, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 10;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        jPanelColor.add(canvasFractFinal, gridBagConstraints);

        jComboPresets.addItemListener(new java.awt.event.ItemListener() {
            public void itemStateChanged(java.awt.event.ItemEvent evt) {
                jComboPresetsItemStateChanged(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(0, 10, 0, 0);
        jPanelColor.add(jComboPresets, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.ipadx = 10;
        gridBagConstraints.insets = new java.awt.Insets(5, 0, 5, 5);
        getContentPane().add(jPanelColor, gridBagConstraints);

        jPanelResult.setLayout(new java.awt.GridBagLayout());

        jPanelResult.setBorder(new javax.swing.border.TitledBorder("RESULT"));
        jButtonCustomize.setText("CUSTOMIZE");
        jButtonCustomize.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButtonCustomizeActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.gridwidth = 3;
        gridBagConstraints.insets = new java.awt.Insets(5, 0, 0, 0);
        jPanelResult.add(jButtonCustomize, gridBagConstraints);

        jLabel3.setForeground(new java.awt.Color(0, 0, 255));
        jLabel3.setText("EQUATION");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        jPanelResult.add(jLabel3, gridBagConstraints);

        jLabelResult.setHorizontalAlignment(javax.swing.SwingConstants.RIGHT);
        jLabelResult.setText(" ");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        jPanelResult.add(jLabelResult, gridBagConstraints);

        jLabel16.setText("=");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        jPanelResult.add(jLabel16, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        jPanelResult.add(jLabelEquation, gridBagConstraints);

        jLabel23.setForeground(new java.awt.Color(0, 0, 255));
        jLabel23.setText("ANSWER");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        jPanelResult.add(jLabel23, gridBagConstraints);

        jLabel24.setText("=");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        jPanelResult.add(jLabel24, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 3;
        jPanelResult.add(jLabelUnit, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 5, 5);
        getContentPane().add(jPanelResult, gridBagConstraints);

        jLabelStatus.setText(" ");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        getContentPane().add(jLabelStatus, gridBagConstraints);

        pack();
    }
    // </editor-fold>//GEN-END:initComponents
	
	private void jButtonPresetsNameOKActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButtonPresetsNameOKActionPerformed
		presetsDialog.setVisible(false);
	}//GEN-LAST:event_jButtonPresetsNameOKActionPerformed
	
	private void jSpinnerColorNumberStateChanged(javax.swing.event.ChangeEvent evt) {//GEN-FIRST:event_jSpinnerColorNumberStateChanged
		int newNumber = (Integer)jSpinnerColorNumber.getValue();
		int oldNumber = panelInit.getColorNumber();
		int difference = Math.abs(newNumber-oldNumber);
		for (int i=0; i<difference; i++){
			if (oldNumber<newNumber){
				panelInit.addColor();
				panelFinal.addColor();
			}else{
				panelInit.removeColor();
				panelFinal.removeColor();
			}
		}
		pack();
		
		// additional
		colorDialog.setRectsNumber(newNumber);

		canvasFractFinal.clearGraphicObject();
		rectFraction = new SOSRectangle[newNumber];
		for (int i=0; i<rectFraction.length; i++){
			rectFraction[i] = new SOSRectangle();
			canvasFractFinal.addGraphicObject(rectFraction[i]);
		}
	}//GEN-LAST:event_jSpinnerColorNumberStateChanged
	
	private void jButtonCustomizeActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButtonCustomizeActionPerformed
		algeblaDialog.pack();
		algeblaDialog.setVisible(true);
		
		setEquationFromDialog();
	}//GEN-LAST:event_jButtonCustomizeActionPerformed
	
	private void jMenuSaveImageActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jMenuSaveImageActionPerformed
		// ɏR[hǉ:
	}//GEN-LAST:event_jMenuSaveImageActionPerformed
	
	private void jComboAlgorithmItemStateChanged (java.awt.event.ItemEvent evt) {//GEN-FIRST:event_jComboAlgorithmItemStateChanged
		
		PluginClassLoader loader = new PluginClassLoader(pathAlgorithm);
		try{
			String algorithmName = (String)jComboAlgorithm.getSelectedItem();
			clustering = (Clustering)loader.loadClass(algorithmName).newInstance();
		}catch(Exception e){
			e.printStackTrace();
		}
		if (clustering.getOptionPanel()!=null){
			optionDialog.getContentPane().remove(jPanelOption);
			jPanelOption = clustering.getOptionPanel();
			GridBagConstraints constraints = new GridBagConstraints();
			constraints.gridx = 0;
			constraints.gridy = 10;
			constraints.gridwidth = 2;
			optionDialog.getContentPane().add(jPanelOption, constraints);
			optionDialog.pack();
		}
	}//GEN-LAST:event_jComboAlgorithmItemStateChanged
			
	private void formComponentResized(java.awt.event.ComponentEvent evt) {//GEN-FIRST:event_formComponentResized
		colorDialog.setLocation(0,this.getHeight());
		segDialog.setLocation(this.getWidth()*2/3,this.getHeight());
	}//GEN-LAST:event_formComponentResized
	
	private void jCheckSegItemStateChanged(java.awt.event.ItemEvent evt) {//GEN-FIRST:event_jCheckSegItemStateChanged
		segDialog.setVisible(jCheckSeg.isSelected());
	}//GEN-LAST:event_jCheckSegItemStateChanged
	
	private void jCheckColorItemStateChanged(java.awt.event.ItemEvent evt) {//GEN-FIRST:event_jCheckColorItemStateChanged
		if (jCheckColor.isSelected()){
			colorDialog.setVisible(true);
			segDialog.setLocation(colorDialog.getWidth(),segDialog.getY());
		}else{
			colorDialog.setVisible(false);
			segDialog.setLocation(0,segDialog.getY());
		}
	}//GEN-LAST:event_jCheckColorItemStateChanged
	
	private void jCheckRandomItemStateChanged(java.awt.event.ItemEvent evt) {//GEN-FIRST:event_jCheckRandomItemStateChanged
		if (jCheckRandom.isSelected()){
			jComboPresets.setEnabled(true);
		}else{
			jComboPresets.setEnabled(false);
		}
	}//GEN-LAST:event_jCheckRandomItemStateChanged
	
	private void jComboPresetsItemStateChanged(java.awt.event.ItemEvent evt) {//GEN-FIRST:event_jComboPresetsItemStateChanged
		int index = jComboPresets.getSelectedIndex();
		if (index>-1 && jComboPresets.loaded){
			int[][] currentColor = jComboPresets.getRGB(index);
			setNewColorPanel(currentColor.length);
			panelInit.updateColor(currentColor);
		}
	}//GEN-LAST:event_jComboPresetsItemStateChanged
	
	private void jCheckProcessItemStateChanged(java.awt.event.ItemEvent evt) {//GEN-FIRST:event_jCheckProcessItemStateChanged
		boolean enabled = jCheckProcess.isSelected();
		jTextMagnification.setEnabled(enabled);
		jComboAnimationColor.setEnabled(enabled);
	}//GEN-LAST:event_jCheckProcessItemStateChanged
	
	private void jButtonOptionActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButtonOptionActionPerformed
		optionDialog.setVisible(true);
	}//GEN-LAST:event_jButtonOptionActionPerformed
	
	private void jButtonAnalysisActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButtonAnalysisActionPerformed
		Thread threadAnalysis = new Thread(this);
		threadAnalysis.start();
	}//GEN-LAST:event_jButtonAnalysisActionPerformed
	
	private void jButtonToInitActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButtonToInitActionPerformed
		panelInit.updateColor(panelFinal.getColorArray());
	}//GEN-LAST:event_jButtonToInitActionPerformed
	
	private void jButtonAddPresetActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButtonAddPresetActionPerformed
		String name = getPresetsName();
		if (name.length()==0){
			name = "UNTITLED";
		}
		jComboPresets.addItem(name, panelInit.getColorArray());
		jComboPresets.setSelectedIndex(jComboPresets.getItemCount()-1);
	}//GEN-LAST:event_jButtonAddPresetActionPerformed
	
	private void jButtonRemovePresetActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButtonRemovePresetActionPerformed
		int removeIndex = jComboPresets.getSelectedIndex();
		jComboPresets.removeItemAt(removeIndex);
	}//GEN-LAST:event_jButtonRemovePresetActionPerformed
	
	private void jButtonSaveActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButtonSaveActionPerformed
		jComboPresets.setRGB(panelInit.getColorArray());
		jComboPresets.saveItem();
	}//GEN-LAST:event_jButtonSaveActionPerformed
	
	private void formWindowClosing(java.awt.event.WindowEvent evt) {//GEN-FIRST:event_formWindowClosing
		colorDialog.dispose();
		segDialog.dispose();
		optionDialog.dispose();
		if (twResult != null ){
			twResult.dispose();
		}
		jLabelStatus.setText("");
	}//GEN-LAST:event_formWindowClosing
	
	private void jButtonBreakActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButtonBreakActionPerformed
		// TODO add your handling code here:
		synchronized(this){
			threadSuspended = !threadSuspended;
			if (!threadSuspended){
				notify();
				jButtonBreak.setText("SUSPEND");
			}else{
				jButtonBreak.setText("RESUME");
			}
		}
	}//GEN-LAST:event_jButtonBreakActionPerformed
	
	/**
	 * @param args the command line arguments
	 */
	public static void main(String args[]) {
		java.awt.EventQueue.invokeLater(new Runnable() {
			public void run() {
				new CytoQViewer();
			}
		});
	}
	
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private sos.canvas.JObjectCanvas canvasFractFinal;
    private javax.swing.JButton jButtonAddPreset;
    private javax.swing.JButton jButtonAnalysis;
    private javax.swing.JButton jButtonBreak;
    private javax.swing.JButton jButtonCustomize;
    private javax.swing.JButton jButtonOption;
    private javax.swing.JButton jButtonPresetsNameOK;
    private javax.swing.JButton jButtonRemovePreset;
    private javax.swing.JButton jButtonSave;
    private javax.swing.JButton jButtonToInit;
    private javax.swing.JCheckBox jCheckColor;
    private javax.swing.JCheckBox jCheckProcess;
    private javax.swing.JCheckBox jCheckRandom;
    private javax.swing.JCheckBox jCheckResultFinal;
    private javax.swing.JCheckBox jCheckResultInit;
    private javax.swing.JCheckBox jCheckSeg;
    private javax.swing.JCheckBox jCheckSource;
    private javax.swing.JComboBox jComboAlgorithm;
    private javax.swing.JComboBox jComboAnalysis;
    private javax.swing.JComboBox jComboAnimationColor;
    private javax.swing.JComboBox jComboColorSpace;
    private cytoq.JComboColorSets jComboPresets;
    private javax.swing.JComboBox jComboSort;
    private javax.swing.JFrame jFrameTextResult;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JLabel jLabel10;
    private javax.swing.JLabel jLabel11;
    private javax.swing.JLabel jLabel12;
    private javax.swing.JLabel jLabel13;
    private javax.swing.JLabel jLabel14;
    private javax.swing.JLabel jLabel15;
    private javax.swing.JLabel jLabel16;
    private javax.swing.JLabel jLabel17;
    private javax.swing.JLabel jLabel18;
    private javax.swing.JLabel jLabel19;
    private javax.swing.JLabel jLabel2;
    private javax.swing.JLabel jLabel20;
    private javax.swing.JLabel jLabel21;
    private javax.swing.JLabel jLabel22;
    private javax.swing.JLabel jLabel23;
    private javax.swing.JLabel jLabel24;
    private javax.swing.JLabel jLabel25;
    private javax.swing.JLabel jLabel26;
    private javax.swing.JLabel jLabel27;
    private javax.swing.JLabel jLabel3;
    private javax.swing.JLabel jLabel4;
    private javax.swing.JLabel jLabel5;
    private javax.swing.JLabel jLabel7;
    private javax.swing.JLabel jLabel9;
    private javax.swing.JLabel jLabelEquation;
    private javax.swing.JLabel jLabelResult;
    private javax.swing.JLabel jLabelStatus;
    private javax.swing.JLabel jLabelUnit;
    private javax.swing.JMenu jMenu1;
    private javax.swing.JMenuBar jMenuBar1;
    private javax.swing.JMenuItem jMenuSaveImage;
    private javax.swing.JPanel jPanelColor;
    private javax.swing.JPanel jPanelMain;
    private javax.swing.JPanel jPanelOption;
    private javax.swing.JPanel jPanelResult;
    private javax.swing.JSpinner jSpinnerBatchIteration;
    private javax.swing.JSpinner jSpinnerColorNumber;
    private javax.swing.JTextField jTextEPS;
    private javax.swing.JTextField jTextMagnification;
    private javax.swing.JTextField jTextMaxIteration;
    private javax.swing.JTextField jTextPresetsName;
    private javax.swing.JDialog optionDialog;
    private sos.panel.JColorPanel panelFinal;
    private sos.panel.JColorPanel panelInit;
    private javax.swing.JDialog presetsDialog;
    // End of variables declaration//GEN-END:variables
	
}
