/*
 * Conductor.java
 *
 * Created on 2005/01/11, 10:46
 */

package taktstock;

import java.awt.Dimension;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.geom.AffineTransform;
import java.awt.image.BufferedImage;
import java.io.File;
import java.util.ArrayList;
import java.util.Timer;
import java.util.TimerTask;
import sos.awt.ImageTransform;
import sos.io.SOSImageIO;
import sos.io.SOSFileIO;
import taktstock.animation.AnimationMaker;

/**
 *
 * @author pathologist
 */
public class Conductor {
	private TaktstockViewer viewer;
	private AnimationMaker animationMaker;
	
	private TaktstockFile currentTaktstockFile;
	private ImageTransform imageTransform;
	private AffineTransform samplingTransform = new AffineTransform();
	private AffineTransform invSamplingTransform = new AffineTransform();
	
	private Timer timerFine = new Timer();
	private TimerTask taskFine;
	private Rectangle fineBuffer;
	private Point offsetFineImage = new Point();
	private AffineTransform fineTransform = new AffineTransform();
	
	private TimerTask taskAnimation;
	private Timer timerAnimation = new Timer();
	private ArrayList<KeyFrame> keyFrameArray = new ArrayList<KeyFrame>();
	private volatile KeyFrame currentKeyFrame;
	private volatile boolean frameSuspended;
	private volatile long startTime;
	
	/** Creates a new instance of Conductor */
	public Conductor() {
		viewer = new TaktstockViewer(this);
		viewer.setVisible(true);
		imageTransform = viewer.getImageTransform();
		
		animationMaker = new AnimationMaker(this,viewer);
		animationMaker.setSize(viewer.getWidth(),animationMaker.getHeight());
		animationMaker.setLocation(viewer.getX(),viewer.getHeight());
		
		// run the fine image thread
		createFineImageTask();
		timerFine = new Timer();
		timerFine.schedule(taskFine,0,1000);
	}
	
	/**
	 * @param args the command line arguments
	 */
	public static void main(String args[]) {
		java.awt.EventQueue.invokeLater(new Runnable() {
			public void run() {
				new Conductor();
			}
		});
	}
	
	private void createFineImageTask(){
		taskFine = new TimerTask(){
			public void run(){
				if (fineBuffer!=null){
					// time consuming task 1
					final Rectangle source = fineBuffer;
					BufferedImage fineImage = (BufferedImage)SOSImageIO.getPartialImage(currentTaktstockFile.getFile(),source);
					
					// time consuming task 2
					if (source.equals(fineBuffer)){
						viewer.drawFineImageToOffScreen(fineImage, source.width, source.height);
						fineImage = null;
					}
					
					if (source.equals(fineBuffer)){
						fineBuffer = null;
						viewer.updateCanvas();
					}
				}
			}
		};
	}
	
	public void setPlayList(){
		animationMaker.getPlayList(keyFrameArray);
		if (0<keyFrameArray.size()){
			currentKeyFrame = keyFrameArray.get(0);
			frameSuspended = currentKeyFrame.isPauseFrame();
			viewer.setCurrentComment(currentKeyFrame.getComment());
		}
	}
	
	public void playAnimation(int fps){
		setPlayList();
		
		if (0<keyFrameArray.size()){
			viewer.setSpinnerEnabled(false);
			
			if (!currentTaktstockFile.equals(currentKeyFrame.getFile())){
				currentTaktstockFile = currentKeyFrame.getTaktstockFile();
				loadThumbnail();
			}
			if (currentKeyFrame.isPauseFrame()){
				double[] matrix = currentKeyFrame.transform(0L);
				setFrameMatrix(matrix);
				viewer.drawImage();
			}
			
			createAnimationTask();
			startTime = System.currentTimeMillis();
			timerAnimation.scheduleAtFixedRate(taskAnimation,0, 100/fps);
			
			viewer.setAnimationScreen();
		}
	}
	
	private void createAnimationTask(){
		taskAnimation = new TimerTask(){
			public void run(){
				if (!frameSuspended){
					long elapsedTime = System.currentTimeMillis()-startTime;
					processFrame(elapsedTime);
					viewer.drawThumbnail();
				}
			}
		};
	}
	
	public void processFrame(long elapsedTime){
		if (elapsedTime<currentKeyFrame.getEndTime()){
			double[] matrix = currentKeyFrame.transform(elapsedTime);
			setFrameMatrix(matrix);
		}else{
			if (currentKeyFrame.equals(currentKeyFrame.getNextFrame())){ // end of animation
				viewer.setSpinnerEnabled(true);
				taskAnimation.cancel();
				viewer.drawImage();
				viewer.setCurrentComment(null);
			}else{ // setting for new keyFrame
				currentKeyFrame = currentKeyFrame.getNextFrame();
				if (currentKeyFrame.isShowComment()){
					viewer.setCurrentComment(currentKeyFrame.getComment());
				}else{
					viewer.setCurrentComment(null);
				}
				if (currentKeyFrame.isPauseFrame()){
					frameSuspended = true;
					viewer.drawImage();
				}
				if (!currentTaktstockFile.equals(currentKeyFrame.getTaktstockFile())){
					startTime = System.currentTimeMillis();
					currentTaktstockFile = currentKeyFrame.getTaktstockFile();
					loadThumbnail();
				}
			}
		}
	}
	
	public void resumeAnimation(){
		startTime = System.currentTimeMillis();
		if (!frameSuspended && currentKeyFrame!=null){
			startTime -= currentKeyFrame.getEndTime();
		}
		frameSuspended = false;
	}
	
	public void rewindAnimation(){
		frameSuspended = false;
		if (currentKeyFrame!=null){
			currentKeyFrame = currentKeyFrame.getPrevFrame();
			startTime = System.currentTimeMillis()-currentKeyFrame.getStartTime();
		}
	}
	
	public void saveFineImage(File exportFile){
		Rectangle bounds = getFineBuffer();
		BufferedImage fineImage = (BufferedImage)SOSImageIO.getPartialImage(currentTaktstockFile.getFile(),bounds);
		Dimension destSize = imageTransform.getDestSize();
		fineTransform.setToScale(1.0*destSize.width/bounds.width,1.0*destSize.height/bounds.height);
		
		BufferedImage saveImage = new BufferedImage(destSize.width,destSize.height,fineImage.getType());
		Graphics2D g2 = (Graphics2D)saveImage.getGraphics();
		g2.drawImage(fineImage,fineTransform,viewer);
		SOSImageIO.writeImage(exportFile,SOSImageIO.TYPE_JPEG,saveImage);
	}
	
	public void setFrameMatrix(double[] matrix){
		AffineTransform currentAT = imageTransform.getAffineTransform();
		currentAT.setTransform(matrix[0], matrix[1], matrix[2], matrix[3], matrix[4], matrix[5]);
		currentAT.concatenate(invSamplingTransform);
		currentAT.preConcatenate(viewer.getScreenTransform());
		
		imageTransform.calcMagnification();
	}
	
	public void loadThumbnail(){
		if (currentTaktstockFile == null){
			return;
		}
		
		imageTransform.setImage(null);
		File file = currentTaktstockFile.getFile();
		Dimension originalSize = SOSImageIO.getImageSize(file);
		
		Image img = null;
		int samplingRate = currentTaktstockFile.getSamplingRate();
		if (1<samplingRate){
			img = SOSImageIO.getSubsampledImage(file, samplingRate, samplingRate);
		}else{
			img = SOSImageIO.readImage(file);
		}
		imageTransform.setImage(img);
		
		double x = 1.0*img.getWidth(null)/originalSize.width;
		double y = 1.0*img.getHeight(null)/originalSize.height;
		samplingTransform.setToScale(x,y);
		try{
			invSamplingTransform = samplingTransform.createInverse();
		}catch(Exception e){
			e.printStackTrace();
		}
		
		viewer.showSize(originalSize.width, originalSize.height);
	}
	
	public void setFineBuffer(){
		// load the fine image
		if (1.0<imageTransform.getMagnification()){
			fineBuffer = getFineBuffer();
		}else{
			fineBuffer = null;
		}
	}
	
	private Rectangle getFineBuffer(){
		return invSamplingTransform.createTransformedShape(imageTransform.getVisibleBounds()).getBounds();
	}
	
	public double[] getTransformMatrix(){
		AffineTransform transform = new AffineTransform(imageTransform.getAffineTransform());
		transform.concatenate(samplingTransform);
		double[] matrix = new double[6];
		transform.getMatrix(matrix);
		
		return matrix;
	}
	
	public void setCurrentTaktstockFile(TaktstockFile taktstockFile){
		currentTaktstockFile = taktstockFile;
	}
	
	public void cancelAllTimerTask(){
		timerAnimation.cancel();
		timerFine.cancel();
		
		viewer.setCurrentComment(null);
	}
	
	public void showAnimationMaker(){
		
		animationMaker.setVisible(true);
	}
	
	public void saveCurrentImage(){
		if (currentTaktstockFile!=null){
			currentTaktstockFile.save();
		}else{
			System.out.println("Null file!");
		}
	}
	
	public void addAnimationFile(File file, int samplingRate){
		animationMaker.addFile(file,samplingRate);
	}
	
	public void changeSamplingRate(int samplingRate){
		if (currentTaktstockFile==null){
			return;
		}
		currentTaktstockFile.setSamplingRate(samplingRate);
		AffineTransform transform = imageTransform.getAffineTransform();
		transform.concatenate(samplingTransform);
		loadThumbnail();
		transform.concatenate(invSamplingTransform);
		imageTransform.calcMagnification();
		viewer.drawImage();
	}
	
	public int calculateSamplingRate(File file, Dimension size){
		
		Dimension imageSize = SOSImageIO.getImageSize(file);
		int widthRate = (int)Math.ceil(1.0*imageSize.width/size.width);
		int heightRate = (int)Math.ceil(1.0*imageSize.height/size.height);
		
		return Math.max(widthRate,heightRate);
	}
}
