/*
 * [概要]
 * 
 * =====================================================================================
 * [履歴]
 * 2014-05-09 n.koseki 新規作成
 * 
 * =====================================================================================
 */
package org.dyndns.nuda.mapper;

import java.lang.reflect.Proxy;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;

import javax.sql.DataSource;

import org.dyndns.nuda.mapper.annotation.JDBCQuery;
import org.dyndns.nuda.tools.util.StringUtil;

/**
 * SQLインタフェースを生成するためのビルダです
 * 
 * @author nkoseki
 * 
 */
public class SQLInterfaceFactory {
	static {
		//		ClassLoader loader = Thread.currentThread().getContextClassLoader();
		//		MyClassLoader myLoader = new MyClassLoader(loader);
		//		Thread.currentThread().setContextClassLoader(myLoader);
		
	}
	
	private Connection	con						= null;
	
	private boolean		useAutoCommit			= false;
	
	private boolean		useManualTransaction	= false;
	
	private ClassLoader	currentClassLoader		= null;
	
	private SQLInterfaceFactory() {
		// インスタンス化させない
	}
	
	static {
		//		try {
		//			Class.forName("org.dyndns.nuda.mapper.driver.SorMapDriver");
		//		} catch (ClassNotFoundException e) {
		//		}
	}
	
	/**
	 * このクラスのインスタンスを取得します
	 * 
	 * @return SQLInterfaceFactory
	 */
	public static SQLInterfaceFactory newInstance() {
		SQLInterfaceFactory builder = new SQLInterfaceFactory();
		
		return builder;
	}
	
	/**
	 * 引数に指定されたクラスオブジェクトを元にSQLインタフェースを生成して返します
	 * 
	 * @param <T>
	 *            SQLインタフェース型
	 * @param interfaceClass
	 *            SQLインタフェースクラスオブジェクト
	 * @return SQLインタフェースインスタンス
	 */
	public <T> T create(final Class<T> interfaceClass) {
		
		T result = null;
		
		try {
			this.checkParameter(interfaceClass);
		} catch (Exception e) {
			throw new IllegalArgumentException(e.getMessage());
		}
		
		if (this.currentClassLoader == null) {
			this.currentClassLoader = interfaceClass.getClassLoader();
		}
		
		ClassLoader oldLoader = Thread.currentThread().getContextClassLoader();
		
		//Thread.currentThread().setContextClassLoader(this.currentClassLoader);
		
		JDBCXMLInvocationHandler handler;
		
		if (this.con == null) {
			JDBCQuery jdbcQuery = interfaceClass.getAnnotation(JDBCQuery.class);
			String sqlXMLPath = "";
			if (jdbcQuery != null) {
				sqlXMLPath = jdbcQuery.sqlxml();
			} else {
				// クラス名からXMLファイルロケーションを自動生成
				String className =
					interfaceClass.getCanonicalName().replace(".", "/");
				
				className = className + ".xml";
				
				sqlXMLPath = className;
			}
			
			QueryXMLReader reader = new QueryXMLReader();
			reader
				.read(this.currentClassLoader.getResourceAsStream(sqlXMLPath));
			
			ConnectionBean conBean = reader.getConnectionInfo();
			
			if (conBean.url != null && !conBean.url.isEmpty()) {
				try {
					if (conBean.prop.containsKey("driver")) {
						String driverName = conBean.prop.getProperty("driver");
						
						try {
							Class.forName(driverName);
						} catch (Exception e) {
							String message =
								StringUtil.format(
									"{}",
									"Driver Class Foundation Error");
							throw new IllegalArgumentException(message, e);
						}
						
					}
					Connection c =
						DriverManager.getConnection(conBean.url, conBean.prop);
					this.con = c;
				} catch (SQLException e) {
					throw new IllegalArgumentException(e);
					
				}
			} else {
				throw new IllegalArgumentException("JDBCコネクションを指定してください");
			}
		}
		
		try {
			handler =
				new JDBCXMLInvocationHandler(
					this.con,
					interfaceClass,
					this.currentClassLoader,
					this.useAutoCommit,
					this.useManualTransaction);
			
			Object intf =
				Proxy.newProxyInstance(
					Thread.currentThread().getContextClassLoader(),
					new Class<?>[] { interfaceClass },
					handler);
			
			result = interfaceClass.cast(intf);
			
			//Thread.currentThread().setContextClassLoader(oldLoader);
			
			return result;
		} catch (Exception e) {
			e.printStackTrace();
		}
		
		return result;
	}
	
	private <T> void checkParameter(final Class<T> interfaceClass)
			throws Exception {
		if (interfaceClass == null) {
			String message =
				StringUtil.format("param [interfaceClass] is null");
			throw new Exception(message);
		}
		if (!interfaceClass.isInterface()) {
			String message =
				StringUtil.format(
					"param [interfaceClass({})] is not a interface",
					interfaceClass.getCanonicalName());
			throw new Exception(message);
		}
		//		if (!interfaceClass.isAnnotationPresent(JDBCQuery.class)) {
		//			String message =
		//				StringUtil
		//					.format(
		//						"param [interfaceClass({})] is not presented 'org.dyndns.nuda.mapper.JDBCQuery'",
		//						interfaceClass.getCanonicalName());
		//			throw new Exception(message);
		//		}
		
		//		if (this.con == null) {
		//			String message = StringUtil.format("connection is not set");
		//			throw new Exception(message);
		//		}
	}
	
	/**
	 * データベースと接続するためのコネクションオブジェクトをビルダに設定します
	 * 
	 * @param con
	 *            コネクションオブジェクト
	 * @return 自身のインスタンス
	 */
	public SQLInterfaceFactory connection(final Connection con) {
		if (con != null) {
			this.con = con;
		}
		
		return this;
	}
	
	/**
	 * データベースと接続するためのでーたソースオブジェクトをビルダに設定します
	 * 
	 * @param dataSource
	 *            データソース
	 * @return 自身のインスタンス
	 */
	public SQLInterfaceFactory dataSource(final DataSource dataSource) {
		try {
			if (dataSource != null) {
				this.con = dataSource.getConnection();
			}
			
		} catch (SQLException e) {
			IllegalArgumentException e2 = new IllegalArgumentException(e);
			
			throw e2;
		}
		return this;
	}
	
	/**
	 * ビルダに自動コミットモードフラグを設定します
	 * 
	 * @param useAutoCommit
	 *            自動コミットモードをONにする:true / 自動コミットモードをOFFにする:false
	 * @return 自身のインスタンス
	 */
	public SQLInterfaceFactory useAutoCommit(final boolean useAutoCommit) {
		this.useAutoCommit = useAutoCommit;
		return this;
	}
	
	/**
	 * ビルダにユーザベーストランザクション制御フラグを設定します
	 * 
	 * @param useManualTransaction
	 *            ユーザベーストランザクション制御を使用する:true / ユーザベーストランザクション制御を使用しない:false
	 * @return 自身のインスタンス
	 */
	public SQLInterfaceFactory useManualTransaction(
			final boolean useManualTransaction) {
		this.useManualTransaction = useManualTransaction;
		return this;
	}
	
	/**
	 * SQLInterfaceを生成するときに用いるクラスローダをビルダに設定します
	 * 
	 * @param currentClassLoader
	 *            クラスローダ
	 * @return 自身のインスタンス
	 */
	public SQLInterfaceFactory currentClassLoader(
			final ClassLoader currentClassLoader) {
		if (currentClassLoader != null) {
			this.currentClassLoader = currentClassLoader;
		}
		
		return this;
	}
}
